package com.github.natche.gravatarjavaclient.avatar;

import com.github.natche.gravatarjavaclient.exceptions.GravatarJavaClientException;
import com.github.natche.gravatarjavaclient.utils.InputValidator;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;

import javax.imageio.ImageIO;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.util.Arrays;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * A singleton responsible for saving images to the local file system.
 */
enum GravatarAvatarRequestImageSaver {
    /**
     * The singleton instance.
     */
    INSTANCE;

    /**
     * The number of files this saver has saved.
     */
    private final AtomicInteger savedCount = new AtomicInteger(0);

    /**
     * The image formats the current system supports saving to via {@link ImageIO}.
     * See {@link ImageIO#getWriterFormatNames()}.
     */
    private final ImmutableList<String> SUPPORTED_IMAGE_FORMATS = ImmutableList.copyOf(
            Arrays.stream(ImageIO.getWriterFormatNames())
                    .map(String::toLowerCase)
                    .distinct()
                    .toList()
    );

    /**
     * Returns the number of files which have been saved during this runtime.
     *
     * @return the number of files which have been saved during this runtime
     */
    public int getSavedCount() {
        return savedCount.get();
    }

    /**
     * Saves the image to the provided file encoded using the specified format.
     *
     * @param file   The file to save the image to
     * @param format The format in which to save the image (must be a valid format)
     * @throws NullPointerException        if any parameter is null
     * @throws IllegalArgumentException    if the provided file is a directory, if the provided format
     *                                     is not supported by the current file system, or if the file name
     *                                     is invalid
     * @throws GravatarJavaClientException if the file write fails
     */
    public boolean saveTo(BufferedImage image, File file, String format) {
        Preconditions.checkNotNull(image);
        Preconditions.checkNotNull(file);
        Preconditions.checkArgument(!file.isDirectory());
        Preconditions.checkNotNull(format);
        Preconditions.checkArgument(SUPPORTED_IMAGE_FORMATS.contains(format.toLowerCase()));
        Preconditions.checkArgument(InputValidator.isValidFilename(file.getName()));

        try {
            if (!ImageIO.write(image, format.toLowerCase(), file)) throw new IOException("Failed");
            savedCount.incrementAndGet();
            return true;
        } catch (IOException e) {
            throw new GravatarJavaClientException("Failed to write the image to the file system");
        }
    }
}
