// Copyright (C) 2022 Intel Corporation

// SPDX-License-Identifier: Apache-2.0

package me.lemire.integercompression.vector;

import java.util.Arrays;
import jdk.incubator.vector.*;

/**
 * This is a readable but less efficient version of the VectorBitPacker class.
 *
 */
public class VectorBitPackerTerse {
  static final VectorSpecies<Integer> SPECIES_512 = IntVector.SPECIES_512;
  static final VectorSpecies<Integer> SPECIES_256 = IntVector.SPECIES_256;
  static final int VLEN_512 = 16;
  static final int VLEN_256 = 8;
  static final int BLOCK_SIZE = 256;

  private static void fastpackOddBit(final int[] in, int inpos, final int[] out,
                                     int outpos, int b, final int[] ho,
                                     final int[] lc) {
    final int mask = (1 << b) - 1;
    final int N = 31 / b;
    var iV = IntVector.fromArray(SPECIES_256, in, inpos);
    var oV = iV.and(mask);
    int n = 1;
    for (; n <= N; n++) {
      iV = IntVector.fromArray(SPECIES_256, in, inpos + n * VLEN_256);
      oV = iV.and(mask).lanewise(VectorOperators.LSHL, b * n).or(oV);
    }
    oV.intoArray(out, outpos);
    outpos += VLEN_256;

    final int L = b - 1;
    for (int i = 0; i < L; i++) {
      oV = iV.and(mask).lanewise(VectorOperators.LSHR, ho[i]);
      for (int j = 0; j < lc[i]; j++) {
        iV = IntVector.fromArray(SPECIES_256, in, inpos + n * VLEN_256);
        oV = iV.and(mask)
                 .lanewise(VectorOperators.LSHL, b * j + (b - ho[i]))
                 .or(oV);
        n++;
      }
      oV.intoArray(out, outpos);
      outpos += VLEN_256;
    }
  }

  private static void fastpackOddBitNoMask(final int[] in, int inpos,
                                           final int[] out, int outpos, int b,
                                           final int[] ho, final int[] lc) {
    final int N = 31 / b;
    var iV = IntVector.fromArray(SPECIES_256, in, inpos);
    var oV = iV;
    int n = 1;
    for (; n <= N; n++) {
      iV = IntVector.fromArray(SPECIES_256, in, inpos + n * VLEN_256);
      oV = iV.lanewise(VectorOperators.LSHL, b * n).or(oV);
    }
    oV.intoArray(out, outpos);
    outpos += VLEN_256;

    final int L = b - 1;
    for (int i = 0; i < L; i++) {
      oV = iV.lanewise(VectorOperators.LSHR, ho[i]);
      for (int j = 0; j < lc[i]; j++) {
        iV = IntVector.fromArray(SPECIES_256, in, inpos + n * VLEN_256);
        oV = iV.lanewise(VectorOperators.LSHL, b * j + (b - ho[i])).or(oV);
        n++;
      }
      oV.intoArray(out, outpos);
      outpos += VLEN_256;
    }
  }

  private static void fastUnpackOddBit(final int[] in, int inpos,
                                       final int[] out, int outpos, int b,
                                       final int[] lo, int[] masks, int[] lc) {
    final int mask = (1 << b) - 1;
    final int N = 32 / b;
    var iV = IntVector.fromArray(SPECIES_256, in, inpos);
    int n = 0;
    for (; n < N; n++) {
      iV.lanewise(VectorOperators.LSHR, b * n).and(mask).intoArray(out, outpos);
      outpos += VLEN_256;
    }
    var oV = iV.lanewise(VectorOperators.LSHR, b * n).and(mask);

    final int L = b - 1;
    for (int i = 0; i < L; i++) {
      iV = IntVector.fromArray(SPECIES_256, in, inpos + (i + 1) * VLEN_256);
      oV = iV.and(masks[i]).lanewise(VectorOperators.LSHL, b - lo[i]).or(oV);
      oV.intoArray(out, outpos);
      outpos += VLEN_256;
      int j = 0;
      for (; j < lc[i]; j++) {
        iV.lanewise(VectorOperators.LSHR, b * j + lo[i])
            .and(mask)
            .intoArray(out, outpos);
        outpos += VLEN_256;
        n++;
      }
      oV = iV.lanewise(VectorOperators.LSHR, b * j + lo[i]).and(mask);
    }
  }

  private static void fastpackEvenBit(final int[] in, int inpos,
                                      final int[] out, int outpos, int b,
                                      final int[] ho, final int[] lc) {
    final int mask = (1 << b) - 1;
    final int N = 32 % b == 0 ? (32 / b) - 1 : 32 / b;
    var iV = IntVector.fromArray(SPECIES_512, in, inpos);
    var oV = iV.and(mask);
    int n = 1;
    for (; n <= N; n++) {
      iV = IntVector.fromArray(SPECIES_512, in, inpos + n * VLEN_512);
      oV = iV.and(mask).lanewise(VectorOperators.LSHL, b * n).or(oV);
    }
    oV.intoArray(out, outpos);
    outpos += VLEN_512;

    final int L = (b >>> 1) - 1;
    for (int i = 0; i < L; i++) {
      if (ho[i] != b)
        oV = iV.and(mask).lanewise(VectorOperators.LSHR, ho[i]);
      else
        oV = oV.zero(SPECIES_512);
      for (int j = 0; j < lc[i]; j++) {
        iV = IntVector.fromArray(SPECIES_512, in, inpos + n * VLEN_512);
        oV = iV.and(mask)
                 .lanewise(VectorOperators.LSHL, b * j + (b - ho[i]))
                 .or(oV);
        n++;
      }
      oV.intoArray(out, outpos);
      outpos += VLEN_512;
    }
  }

  private static void fastpackEvenBitNoMask(final int[] in, int inpos,
                                            final int[] out, int outpos, int b,
                                            final int[] ho, final int[] lc) {
    final int N = 32 % b == 0 ? (32 / b) - 1 : 32 / b;
    var iV = IntVector.fromArray(SPECIES_512, in, inpos);
    var oV = iV;
    int n = 1;
    for (; n <= N; n++) {
      iV = IntVector.fromArray(SPECIES_512, in, inpos + n * VLEN_512);
      oV = iV.lanewise(VectorOperators.LSHL, b * n).or(oV);
    }
    oV.intoArray(out, outpos);
    outpos += VLEN_512;

    final int L = (b >>> 1) - 1;
    for (int i = 0; i < L; i++) {
      if (ho[i] != b)
        oV = iV.lanewise(VectorOperators.LSHR, ho[i]);
      else
        oV = oV.zero(SPECIES_512);
      for (int j = 0; j < lc[i]; j++) {
        iV = IntVector.fromArray(SPECIES_512, in, inpos + n * VLEN_512);
        oV = iV.lanewise(VectorOperators.LSHL, b * j + (b - ho[i])).or(oV);
        n++;
      }
      oV.intoArray(out, outpos);
      outpos += VLEN_512;
    }
  }

  private static void fastUnpackEventBit(final int[] in, int inpos,
                                         final int[] out, int outpos, int b,
                                         final int[] lo, int[] masks,
                                         int[] lc) {
    final int mask = (1 << b) - 1;
    final int N = 32 / b;
    var iV = IntVector.fromArray(SPECIES_512, in, inpos);
    int n = 0;
    for (; n < N; n++) {
      iV.lanewise(VectorOperators.LSHR, b * n).and(mask).intoArray(out, outpos);
      outpos += VLEN_512;
    }
    var oV = iV.lanewise(VectorOperators.LSHR, b * n).and(mask);
    if ((b & (b - 1)) == 0)
      oV = oV.zero(SPECIES_512);

    final int L = (b >>> 1) - 1;
    for (int i = 0; i < L; i++) {
      iV = IntVector.fromArray(SPECIES_512, in, inpos + (i + 1) * VLEN_512);
      oV = iV.and(masks[i]).lanewise(VectorOperators.LSHL, b - lo[i]).or(oV);
      oV.intoArray(out, outpos);
      outpos += VLEN_512;
      int j = 0;
      for (; j < lc[i]; j++) {
        iV.lanewise(VectorOperators.LSHR, b * j + lo[i])
            .and(mask)
            .intoArray(out, outpos);
        outpos += VLEN_512;
        n++;
      }
      if ((32 - lo[i]) % b != 0)
        oV = iV.lanewise(VectorOperators.LSHR, b * j + lo[i]).and(mask);
      else
        oV = oV.zero(SPECIES_512);
    }
  }

  public static int slowpack(final int[] in, int inpos, int inlen,
                             final int[] out, int outpos, int b) {
    if (inlen == 0)
      return outpos;
    if (b == 32) {
      System.arraycopy(in, inpos, out, outpos, inlen);
      return outpos + inlen;
    }
    int mask = (1 << b) - 1;
    int c = 0;
    int l = 0;
    int r = 0;
    int val = 0;
    for (int i = 0; i < inlen; i++) {
      val = in[inpos + i] & mask;
      out[outpos] |= val << (c + r);
      c += b;
      l = (32 - r) % b;
      if (c + r >= 32) {
        if (i < inlen - 1 || l != 0)
          outpos++;
        r = l == 0 ? 0 : b - l;
        if (l != 0)
          out[outpos] = val >> (b - r);
        c = 0;
      }
    }
    return outpos;
  }

  public static int slowunpack(final int[] in, int inpos, final int[] out,
                               int outpos, int outlen, int b) {
    if (outlen == 0) {
      return inpos;
    }
    if (b == 32) {
      System.arraycopy(in, inpos, out, outpos, outlen);
      return inpos + outlen;
    }
    int mask = (1 << b) - 1;
    int limit = outpos + outlen;
    int r = 0;
    int val = 0;
    int i = 0;
    for (; outpos < limit; i++) {
      if (r > 0)
        out[outpos++] =
            (val >>> (32 - (b - r))) | ((in[inpos + i] << (b - r)) & mask);
      val = in[inpos + i];
      int j = 0;
      int l = 32 - r;
      int ll = l % b == 0 ? l : l - b;
      while (j < ll && outpos < limit) {
        out[outpos++] = (val >> (j + r)) & mask;
        j += b;
      }
      r = l % b == 0 ? 0 : b - (l % b);
    }
    return inpos + i;
  }

  public static int numCompressedInts(int n, int b) {
    int width = b % 2 == 0 ? VLEN_512 : VLEN_256;
    if (n <= width)
      return n;
    int intsPerVec = (32 / b) * width;
    int q = (n + intsPerVec - 1) / intsPerVec;
    return q * width;
  }

  public static void fastpack(final int[] in, int inpos, final int[] out,
                              int outpos, int b) {
    switch (b) {
    case 0:
      break;
    case 1:
      fastpackOddBit(in, inpos, out, outpos, 1, new int[] {}, new int[] {});
      break;
    case 2:
      fastpackEvenBit(in, inpos, out, outpos, 2, new int[] {}, new int[] {});
      break;
    case 3:
      fastpackOddBit(in, inpos, out, outpos, 3, new int[] {0x2, 0x1},
                     new int[] {0xb, 0xa});
      break;
    case 4:
      fastpackEvenBit(in, inpos, out, outpos, 4, new int[] {0x4},
                      new int[] {0x8});
      break;
    case 5:
      fastpackOddBit(in, inpos, out, outpos, 5, new int[] {0x2, 0x4, 0x1, 0x3},
                     new int[] {0x6, 0x7, 0x6, 0x6});
      break;
    case 6:
      fastpackEvenBit(in, inpos, out, outpos, 6, new int[] {0x2, 0x4},
                      new int[] {0x5, 0x5});
      break;
    case 7:
      fastpackOddBit(in, inpos, out, outpos, 7,
                     new int[] {0x4, 0x1, 0x5, 0x2, 0x6, 0x3},
                     new int[] {0x5, 0x4, 0x5, 0x4, 0x5, 0x4});
      break;
    case 8:
      fastpackEvenBit(in, inpos, out, outpos, 8, new int[] {0x8, 0x8, 0x8},
                      new int[] {0x4, 0x4, 0x4});
      break;
    case 9:
      fastpackOddBit(in, inpos, out, outpos, 9,
                     new int[] {0x5, 0x1, 0x6, 0x2, 0x7, 0x3, 0x8, 0x4},
                     new int[] {0x4, 0x3, 0x4, 0x3, 0x4, 0x3, 0x4, 0x3});
      break;
    case 10:
      fastpackEvenBit(in, inpos, out, outpos, 10,
                      new int[] {0x2, 0x4, 0x6, 0x8},
                      new int[] {0x3, 0x3, 0x3, 0x3});
      break;
    case 11:
      fastpackOddBit(
          in, inpos, out, outpos, 11,
          new int[] {0xa, 0x9, 0x8, 0x7, 0x6, 0x5, 0x4, 0x3, 0x2, 0x1},
          new int[] {0x3, 0x3, 0x3, 0x3, 0x3, 0x3, 0x3, 0x3, 0x3, 0x2});
      break;
    case 12:
      fastpackEvenBit(in, inpos, out, outpos, 12,
                      new int[] {0x8, 0x4, 0xc, 0x8, 0x4},
                      new int[] {0x3, 0x2, 0x3, 0x3, 0x2});
      break;
    case 13:
      fastpackOddBit(in, inpos, out, outpos, 13,
                     new int[] {0x6, 0xc, 0x5, 0xb, 0x4, 0xa, 0x3, 0x9, 0x2,
                                0x8, 0x1, 0x7},
                     new int[] {0x2, 0x3, 0x2, 0x3, 0x2, 0x3, 0x2, 0x3, 0x2,
                                0x3, 0x2, 0x2});
      break;
    case 14:
      fastpackEvenBit(in, inpos, out, outpos, 14,
                      new int[] {0x4, 0x8, 0xc, 0x2, 0x6, 0xa},
                      new int[] {0x2, 0x2, 0x3, 0x2, 0x2, 0x2});
      break;
    case 15:
      fastpackOddBit(in, inpos, out, outpos, 15,
                     new int[] {0x2, 0x4, 0x6, 0x8, 0xa, 0xc, 0xe, 0x1, 0x3,
                                0x5, 0x7, 0x9, 0xb, 0xd},
                     new int[] {0x2, 0x2, 0x2, 0x2, 0x2, 0x2, 0x3, 0x2, 0x2,
                                0x2, 0x2, 0x2, 0x2, 0x2});
      break;
    case 16:
      fastpackEvenBit(in, inpos, out, outpos, 16,
                      new int[] {0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10},
                      new int[] {0x2, 0x2, 0x2, 0x2, 0x2, 0x2, 0x2});
      break;
    case 17:
      fastpackOddBit(in, inpos, out, outpos, 17,
                     new int[] {0xf, 0xd, 0xb, 0x9, 0x7, 0x5, 0x3, 0x1, 0x10,
                                0xe, 0xc, 0xa, 0x8, 0x6, 0x4, 0x2},
                     new int[] {0x2, 0x2, 0x2, 0x2, 0x2, 0x2, 0x2, 0x1, 0x2,
                                0x2, 0x2, 0x2, 0x2, 0x2, 0x2, 0x1});
      break;
    case 18:
      fastpackEvenBit(in, inpos, out, outpos, 18,
                      new int[] {0xe, 0xa, 0x6, 0x2, 0x10, 0xc, 0x8, 0x4},
                      new int[] {0x2, 0x2, 0x2, 0x1, 0x2, 0x2, 0x2, 0x1});
      break;
    case 19:
      fastpackOddBit(in, inpos, out, outpos, 19,
                     new int[] {0xd, 0x7, 0x1, 0xe, 0x8, 0x2, 0xf, 0x9, 0x3,
                                0x10, 0xa, 0x4, 0x11, 0xb, 0x5, 0x12, 0xc, 0x6},
                     new int[] {0x2, 0x2, 0x1, 0x2, 0x2, 0x1, 0x2, 0x2, 0x1,
                                0x2, 0x2, 0x1, 0x2, 0x2, 0x1, 0x2, 0x2, 0x1});
      break;
    case 20:
      fastpackEvenBit(
          in, inpos, out, outpos, 20,
          new int[] {0xc, 0x4, 0x10, 0x8, 0x14, 0xc, 0x4, 0x10, 0x8},
          new int[] {0x2, 0x1, 0x2, 0x1, 0x2, 0x2, 0x1, 0x2, 0x1});
      break;
    case 21:
      fastpackOddBit(
          in, inpos, out, outpos, 21,
          new int[] {0xb,  0x1, 0xc,  0x2, 0xd,  0x3, 0xe,  0x4, 0xf,  0x5,
                     0x10, 0x6, 0x11, 0x7, 0x12, 0x8, 0x13, 0x9, 0x14, 0xa},
          new int[] {0x2, 0x1, 0x2, 0x1, 0x2, 0x1, 0x2, 0x1, 0x2, 0x1,
                     0x2, 0x1, 0x2, 0x1, 0x2, 0x1, 0x2, 0x1, 0x2, 0x1});
      break;
    case 22:
      fastpackEvenBit(
          in, inpos, out, outpos, 22,
          new int[] {0xa, 0x14, 0x8, 0x12, 0x6, 0x10, 0x4, 0xe, 0x2, 0xc},
          new int[] {0x1, 0x2, 0x1, 0x2, 0x1, 0x2, 0x1, 0x2, 0x1, 0x1});
      break;
    case 23:
      fastpackOddBit(in, inpos, out, outpos, 23,
                     new int[] {0x9, 0x12, 0x4, 0xd,  0x16, 0x8, 0x11, 0x3,
                                0xc, 0x15, 0x7, 0x10, 0x2,  0xb, 0x14, 0x6,
                                0xf, 0x1,  0xa, 0x13, 0x5,  0xe},
                     new int[] {0x1, 0x2, 0x1, 0x1, 0x2, 0x1, 0x2, 0x1,
                                0x1, 0x2, 0x1, 0x2, 0x1, 0x1, 0x2, 0x1,
                                0x2, 0x1, 0x1, 0x2, 0x1, 0x1});
      break;
    case 24:
      fastpackEvenBit(
          in, inpos, out, outpos, 24,
          new int[] {0x8, 0x10, 0x18, 0x8, 0x10, 0x18, 0x8, 0x10, 0x18, 0x8,
                     0x10},
          new int[] {0x1, 0x1, 0x2, 0x1, 0x1, 0x2, 0x1, 0x1, 0x2, 0x1, 0x1});
      break;
    case 25:
      fastpackOddBit(in, inpos, out, outpos, 25,
                     new int[] {0x7,  0xe,  0x15, 0x3, 0xa,  0x11, 0x18, 0x6,
                                0xd,  0x14, 0x2,  0x9, 0x10, 0x17, 0x5,  0xc,
                                0x13, 0x1,  0x8,  0xf, 0x16, 0x4,  0xb,  0x12},
                     new int[] {0x1, 0x1, 0x2, 0x1, 0x1, 0x1, 0x2, 0x1,
                                0x1, 0x2, 0x1, 0x1, 0x1, 0x2, 0x1, 0x1,
                                0x2, 0x1, 0x1, 0x1, 0x2, 0x1, 0x1, 0x1});
      break;
    case 26:
      fastpackEvenBit(in, inpos, out, outpos, 26,
                      new int[] {0x6, 0xc, 0x12, 0x18, 0x4, 0xa, 0x10, 0x16,
                                 0x2, 0x8, 0xe, 0x14},
                      new int[] {0x1, 0x1, 0x1, 0x2, 0x1, 0x1, 0x1, 0x2, 0x1,
                                 0x1, 0x1, 0x1});
      break;
    case 27:
      fastpackOddBit(in, inpos, out, outpos, 27,
                     new int[] {0x5,  0xa,  0xf,  0x14, 0x19, 0x3,  0x8,
                                0xd,  0x12, 0x17, 0x1,  0x6,  0xb,  0x10,
                                0x15, 0x1a, 0x4,  0x9,  0xe,  0x13, 0x18,
                                0x2,  0x7,  0xc,  0x11, 0x16},
                     new int[] {0x1, 0x1, 0x1, 0x1, 0x2, 0x1, 0x1, 0x1, 0x1,
                                0x2, 0x1, 0x1, 0x1, 0x1, 0x1, 0x2, 0x1, 0x1,
                                0x1, 0x1, 0x2, 0x1, 0x1, 0x1, 0x1, 0x1});
      break;
    case 28:
      fastpackEvenBit(in, inpos, out, outpos, 28,
                      new int[] {0x4, 0x8, 0xc, 0x10, 0x14, 0x18, 0x1c, 0x4,
                                 0x8, 0xc, 0x10, 0x14, 0x18},
                      new int[] {0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x2, 0x1, 0x1,
                                 0x1, 0x1, 0x1, 0x1});
      break;
    case 29:
      fastpackOddBit(
          in, inpos, out, outpos, 29,
          new int[] {0x3, 0x6, 0x9, 0xc, 0xf,  0x12, 0x15, 0x18, 0x1b, 0x1,
                     0x4, 0x7, 0xa, 0xd, 0x10, 0x13, 0x16, 0x19, 0x1c, 0x2,
                     0x5, 0x8, 0xb, 0xe, 0x11, 0x14, 0x17, 0x1a},
          new int[] {0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x2, 0x1,
                     0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x2, 0x1,
                     0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1});
      break;
    case 30:
      fastpackEvenBit(in, inpos, out, outpos, 30,
                      new int[] {0x2, 0x4, 0x6, 0x8, 0xa, 0xc, 0xe, 0x10, 0x12,
                                 0x14, 0x16, 0x18, 0x1a, 0x1c},
                      new int[] {0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1,
                                 0x1, 0x1, 0x1, 0x1, 0x1});
      break;
    case 31:
      fastpackOddBit(in, inpos, out, outpos, 31,
                     new int[] {0x1,  0x2,  0x3,  0x4,  0x5,  0x6,  0x7,  0x8,
                                0x9,  0xa,  0xb,  0xc,  0xd,  0xe,  0xf,  0x10,
                                0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18,
                                0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e},
                     new int[] {0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1,
                                0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1,
                                0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1,
                                0x1, 0x1, 0x1, 0x1, 0x1, 0x1});
      break;
    case 32:
      System.arraycopy(in, inpos, out, outpos, 256);
      break;
    }
  }

  public static void fastpackNoMask(final int[] in, int inpos, final int[] out,
                                    int outpos, int b) {
    switch (b) {
    case 0:
      break;
    case 1:
      fastpackOddBitNoMask(in, inpos, out, outpos, 1, new int[] {},
                           new int[] {});
      break;
    case 2:
      fastpackEvenBitNoMask(in, inpos, out, outpos, 2, new int[] {},
                            new int[] {});
      break;
    case 3:
      fastpackOddBitNoMask(in, inpos, out, outpos, 3, new int[] {0x2, 0x1},
                           new int[] {0xb, 0xa});
      break;
    case 4:
      fastpackEvenBitNoMask(in, inpos, out, outpos, 4, new int[] {0x4},
                            new int[] {0x8});
      break;
    case 5:
      fastpackOddBitNoMask(in, inpos, out, outpos, 5,
                           new int[] {0x2, 0x4, 0x1, 0x3},
                           new int[] {0x6, 0x7, 0x6, 0x6});
      break;
    case 6:
      fastpackEvenBitNoMask(in, inpos, out, outpos, 6, new int[] {0x2, 0x4},
                            new int[] {0x5, 0x5});
      break;
    case 7:
      fastpackOddBitNoMask(in, inpos, out, outpos, 7,
                           new int[] {0x4, 0x1, 0x5, 0x2, 0x6, 0x3},
                           new int[] {0x5, 0x4, 0x5, 0x4, 0x5, 0x4});
      break;
    case 8:
      fastpackEvenBitNoMask(in, inpos, out, outpos, 8,
                            new int[] {0x8, 0x8, 0x8},
                            new int[] {0x4, 0x4, 0x4});
      break;
    case 9:
      fastpackOddBitNoMask(in, inpos, out, outpos, 9,
                           new int[] {0x5, 0x1, 0x6, 0x2, 0x7, 0x3, 0x8, 0x4},
                           new int[] {0x4, 0x3, 0x4, 0x3, 0x4, 0x3, 0x4, 0x3});
      break;
    case 10:
      fastpackEvenBitNoMask(in, inpos, out, outpos, 10,
                            new int[] {0x2, 0x4, 0x6, 0x8},
                            new int[] {0x3, 0x3, 0x3, 0x3});
      break;
    case 11:
      fastpackOddBitNoMask(
          in, inpos, out, outpos, 11,
          new int[] {0xa, 0x9, 0x8, 0x7, 0x6, 0x5, 0x4, 0x3, 0x2, 0x1},
          new int[] {0x3, 0x3, 0x3, 0x3, 0x3, 0x3, 0x3, 0x3, 0x3, 0x2});
      break;
    case 12:
      fastpackEvenBitNoMask(in, inpos, out, outpos, 12,
                            new int[] {0x8, 0x4, 0xc, 0x8, 0x4},
                            new int[] {0x3, 0x2, 0x3, 0x3, 0x2});
      break;
    case 13:
      fastpackOddBitNoMask(in, inpos, out, outpos, 13,
                           new int[] {0x6, 0xc, 0x5, 0xb, 0x4, 0xa, 0x3, 0x9,
                                      0x2, 0x8, 0x1, 0x7},
                           new int[] {0x2, 0x3, 0x2, 0x3, 0x2, 0x3, 0x2, 0x3,
                                      0x2, 0x3, 0x2, 0x2});
      break;
    case 14:
      fastpackEvenBitNoMask(in, inpos, out, outpos, 14,
                            new int[] {0x4, 0x8, 0xc, 0x2, 0x6, 0xa},
                            new int[] {0x2, 0x2, 0x3, 0x2, 0x2, 0x2});
      break;
    case 15:
      fastpackOddBitNoMask(in, inpos, out, outpos, 15,
                           new int[] {0x2, 0x4, 0x6, 0x8, 0xa, 0xc, 0xe, 0x1,
                                      0x3, 0x5, 0x7, 0x9, 0xb, 0xd},
                           new int[] {0x2, 0x2, 0x2, 0x2, 0x2, 0x2, 0x3, 0x2,
                                      0x2, 0x2, 0x2, 0x2, 0x2, 0x2});
      break;
    case 16:
      fastpackEvenBitNoMask(
          in, inpos, out, outpos, 16,
          new int[] {0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10},
          new int[] {0x2, 0x2, 0x2, 0x2, 0x2, 0x2, 0x2});
      break;
    case 17:
      fastpackOddBitNoMask(in, inpos, out, outpos, 17,
                           new int[] {0xf, 0xd, 0xb, 0x9, 0x7, 0x5, 0x3, 0x1,
                                      0x10, 0xe, 0xc, 0xa, 0x8, 0x6, 0x4, 0x2},
                           new int[] {0x2, 0x2, 0x2, 0x2, 0x2, 0x2, 0x2, 0x1,
                                      0x2, 0x2, 0x2, 0x2, 0x2, 0x2, 0x2, 0x1});
      break;
    case 18:
      fastpackEvenBitNoMask(in, inpos, out, outpos, 18,
                            new int[] {0xe, 0xa, 0x6, 0x2, 0x10, 0xc, 0x8, 0x4},
                            new int[] {0x2, 0x2, 0x2, 0x1, 0x2, 0x2, 0x2, 0x1});
      break;
    case 19:
      fastpackOddBitNoMask(
          in, inpos, out, outpos, 19,
          new int[] {0xd, 0x7, 0x1, 0xe, 0x8, 0x2, 0xf, 0x9, 0x3, 0x10, 0xa,
                     0x4, 0x11, 0xb, 0x5, 0x12, 0xc, 0x6},
          new int[] {0x2, 0x2, 0x1, 0x2, 0x2, 0x1, 0x2, 0x2, 0x1, 0x2, 0x2, 0x1,
                     0x2, 0x2, 0x1, 0x2, 0x2, 0x1});
      break;
    case 20:
      fastpackEvenBitNoMask(
          in, inpos, out, outpos, 20,
          new int[] {0xc, 0x4, 0x10, 0x8, 0x14, 0xc, 0x4, 0x10, 0x8},
          new int[] {0x2, 0x1, 0x2, 0x1, 0x2, 0x2, 0x1, 0x2, 0x1});
      break;
    case 21:
      fastpackOddBitNoMask(
          in, inpos, out, outpos, 21,
          new int[] {0xb,  0x1, 0xc,  0x2, 0xd,  0x3, 0xe,  0x4, 0xf,  0x5,
                     0x10, 0x6, 0x11, 0x7, 0x12, 0x8, 0x13, 0x9, 0x14, 0xa},
          new int[] {0x2, 0x1, 0x2, 0x1, 0x2, 0x1, 0x2, 0x1, 0x2, 0x1,
                     0x2, 0x1, 0x2, 0x1, 0x2, 0x1, 0x2, 0x1, 0x2, 0x1});
      break;
    case 22:
      fastpackEvenBitNoMask(
          in, inpos, out, outpos, 22,
          new int[] {0xa, 0x14, 0x8, 0x12, 0x6, 0x10, 0x4, 0xe, 0x2, 0xc},
          new int[] {0x1, 0x2, 0x1, 0x2, 0x1, 0x2, 0x1, 0x2, 0x1, 0x1});
      break;
    case 23:
      fastpackOddBitNoMask(
          in, inpos, out, outpos, 23,
          new int[] {0x9, 0x12, 0x4, 0xd,  0x16, 0x8, 0x11, 0x3,
                     0xc, 0x15, 0x7, 0x10, 0x2,  0xb, 0x14, 0x6,
                     0xf, 0x1,  0xa, 0x13, 0x5,  0xe},
          new int[] {0x1, 0x2, 0x1, 0x1, 0x2, 0x1, 0x2, 0x1, 0x1, 0x2, 0x1,
                     0x2, 0x1, 0x1, 0x2, 0x1, 0x2, 0x1, 0x1, 0x2, 0x1, 0x1});
      break;
    case 24:
      fastpackEvenBitNoMask(
          in, inpos, out, outpos, 24,
          new int[] {0x8, 0x10, 0x18, 0x8, 0x10, 0x18, 0x8, 0x10, 0x18, 0x8,
                     0x10},
          new int[] {0x1, 0x1, 0x2, 0x1, 0x1, 0x2, 0x1, 0x1, 0x2, 0x1, 0x1});
      break;
    case 25:
      fastpackOddBitNoMask(in, inpos, out, outpos, 25,
                           new int[] {0x7,  0xe,  0x15, 0x3,  0xa,  0x11,
                                      0x18, 0x6,  0xd,  0x14, 0x2,  0x9,
                                      0x10, 0x17, 0x5,  0xc,  0x13, 0x1,
                                      0x8,  0xf,  0x16, 0x4,  0xb,  0x12},
                           new int[] {0x1, 0x1, 0x2, 0x1, 0x1, 0x1, 0x2, 0x1,
                                      0x1, 0x2, 0x1, 0x1, 0x1, 0x2, 0x1, 0x1,
                                      0x2, 0x1, 0x1, 0x1, 0x2, 0x1, 0x1, 0x1});
      break;
    case 26:
      fastpackEvenBitNoMask(in, inpos, out, outpos, 26,
                            new int[] {0x6, 0xc, 0x12, 0x18, 0x4, 0xa, 0x10,
                                       0x16, 0x2, 0x8, 0xe, 0x14},
                            new int[] {0x1, 0x1, 0x1, 0x2, 0x1, 0x1, 0x1, 0x2,
                                       0x1, 0x1, 0x1, 0x1});
      break;
    case 27:
      fastpackOddBitNoMask(
          in, inpos, out, outpos, 27,
          new int[] {0x5,  0xa,  0xf,  0x14, 0x19, 0x3,  0x8,  0xd, 0x12,
                     0x17, 0x1,  0x6,  0xb,  0x10, 0x15, 0x1a, 0x4, 0x9,
                     0xe,  0x13, 0x18, 0x2,  0x7,  0xc,  0x11, 0x16},
          new int[] {0x1, 0x1, 0x1, 0x1, 0x2, 0x1, 0x1, 0x1, 0x1,
                     0x2, 0x1, 0x1, 0x1, 0x1, 0x1, 0x2, 0x1, 0x1,
                     0x1, 0x1, 0x2, 0x1, 0x1, 0x1, 0x1, 0x1});
      break;
    case 28:
      fastpackEvenBitNoMask(in, inpos, out, outpos, 28,
                            new int[] {0x4, 0x8, 0xc, 0x10, 0x14, 0x18, 0x1c,
                                       0x4, 0x8, 0xc, 0x10, 0x14, 0x18},
                            new int[] {0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x2, 0x1,
                                       0x1, 0x1, 0x1, 0x1, 0x1});
      break;
    case 29:
      fastpackOddBitNoMask(
          in, inpos, out, outpos, 29,
          new int[] {0x3, 0x6, 0x9, 0xc, 0xf,  0x12, 0x15, 0x18, 0x1b, 0x1,
                     0x4, 0x7, 0xa, 0xd, 0x10, 0x13, 0x16, 0x19, 0x1c, 0x2,
                     0x5, 0x8, 0xb, 0xe, 0x11, 0x14, 0x17, 0x1a},
          new int[] {0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x2, 0x1,
                     0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x2, 0x1,
                     0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1});
      break;
    case 30:
      fastpackEvenBitNoMask(in, inpos, out, outpos, 30,
                            new int[] {0x2, 0x4, 0x6, 0x8, 0xa, 0xc, 0xe, 0x10,
                                       0x12, 0x14, 0x16, 0x18, 0x1a, 0x1c},
                            new int[] {0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1,
                                       0x1, 0x1, 0x1, 0x1, 0x1, 0x1});
      break;
    case 31:
      fastpackOddBitNoMask(
          in, inpos, out, outpos, 31,
          new int[] {0x1,  0x2,  0x3,  0x4,  0x5,  0x6,  0x7,  0x8,
                     0x9,  0xa,  0xb,  0xc,  0xd,  0xe,  0xf,  0x10,
                     0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18,
                     0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e},
          new int[] {0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1,
                     0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1,
                     0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1});
      break;
    case 32:
      System.arraycopy(in, inpos, out, outpos, 256);
      break;
    }
  }

  public static void fastunpack(final int[] in, int inpos, final int[] out,
                                int outpos, int b) {
    switch (b) {
    case 0:
      Arrays.fill(out, outpos, outpos + 256, 0);
      break;
    case 1:
      fastUnpackOddBit(in, inpos, out, outpos, 1, new int[] {}, new int[] {},
                       new int[] {});
      break;
    case 2:
      fastUnpackEventBit(in, inpos, out, outpos, 2, new int[] {}, new int[] {},
                         new int[] {});
      break;
    case 3:
      fastUnpackOddBit(in, inpos, out, outpos, 3, new int[] {0x1, 0x2},
                       new int[] {0x1, 0x3}, new int[] {0xa, 0xa});
      break;
    case 4:
      fastUnpackEventBit(in, inpos, out, outpos, 4, new int[] {0x4},
                         new int[] {0xf}, new int[] {0x7});
      break;
    case 5:
      fastUnpackOddBit(
          in, inpos, out, outpos, 5, new int[] {0x3, 0x1, 0x4, 0x2},
          new int[] {0x7, 0x1, 0xf, 0x3}, new int[] {0x5, 0x6, 0x5, 0x6});
      break;
    case 6:
      fastUnpackEventBit(in, inpos, out, outpos, 6, new int[] {0x4, 0x2},
                         new int[] {0xf, 0x3}, new int[] {0x4, 0x5});
      break;
    case 7:
      fastUnpackOddBit(in, inpos, out, outpos, 7,
                       new int[] {0x3, 0x6, 0x2, 0x5, 0x1, 0x4},
                       new int[] {0x7, 0x3f, 0x3, 0x1f, 0x1, 0xf},
                       new int[] {0x4, 0x3, 0x4, 0x3, 0x4, 0x4});
      break;
    case 8:
      fastUnpackEventBit(in, inpos, out, outpos, 8, new int[] {0x8, 0x8, 0x8},
                         new int[] {0xff, 0xff, 0xff},
                         new int[] {0x3, 0x3, 0x3});
      break;
    case 9:
      fastUnpackOddBit(in, inpos, out, outpos, 9,
                       new int[] {0x4, 0x8, 0x3, 0x7, 0x2, 0x6, 0x1, 0x5},
                       new int[] {0xf, 0xff, 0x7, 0x7f, 0x3, 0x3f, 0x1, 0x1f},
                       new int[] {0x3, 0x2, 0x3, 0x2, 0x3, 0x2, 0x3, 0x3});
      break;
    case 10:
      fastUnpackEventBit(
          in, inpos, out, outpos, 10, new int[] {0x8, 0x6, 0x4, 0x2},
          new int[] {0xff, 0x3f, 0xf, 0x3}, new int[] {0x2, 0x2, 0x2, 0x3});
      break;
    case 11:
      fastUnpackOddBit(
          in, inpos, out, outpos, 11,
          new int[] {0x1, 0x2, 0x3, 0x4, 0x5, 0x6, 0x7, 0x8, 0x9, 0xa},
          new int[] {0x1, 0x3, 0x7, 0xf, 0x1f, 0x3f, 0x7f, 0xff, 0x1ff, 0x3ff},
          new int[] {0x2, 0x2, 0x2, 0x2, 0x2, 0x2, 0x2, 0x2, 0x2, 0x2});
      break;
    case 12:
      fastUnpackEventBit(in, inpos, out, outpos, 12,
                         new int[] {0x4, 0x8, 0xc, 0x4, 0x8},
                         new int[] {0xf, 0xff, 0xfff, 0xf, 0xff},
                         new int[] {0x2, 0x2, 0x1, 0x2, 0x2});
      break;
    case 13:
      fastUnpackOddBit(in, inpos, out, outpos, 13,
                       new int[] {0x7, 0x1, 0x8, 0x2, 0x9, 0x3, 0xa, 0x4, 0xb,
                                  0x5, 0xc, 0x6},
                       new int[] {0x7f, 0x1, 0xff, 0x3, 0x1ff, 0x7, 0x3ff, 0xf,
                                  0x7ff, 0x1f, 0xfff, 0x3f},
                       new int[] {0x1, 0x2, 0x1, 0x2, 0x1, 0x2, 0x1, 0x2, 0x1,
                                  0x2, 0x1, 0x2});
      break;
    case 14:
      fastUnpackEventBit(in, inpos, out, outpos, 14,
                         new int[] {0xa, 0x6, 0x2, 0xc, 0x8, 0x4},
                         new int[] {0x3ff, 0x3f, 0x3, 0xfff, 0xff, 0xf},
                         new int[] {0x1, 0x1, 0x2, 0x1, 0x1, 0x2});
      break;
    case 15:
      fastUnpackOddBit(in, inpos, out, outpos, 15,
                       new int[] {0xd, 0xb, 0x9, 0x7, 0x5, 0x3, 0x1, 0xe, 0xc,
                                  0xa, 0x8, 0x6, 0x4, 0x2},
                       new int[] {0x1fff, 0x7ff, 0x1ff, 0x7f, 0x1f, 0x7, 0x1,
                                  0x3fff, 0xfff, 0x3ff, 0xff, 0x3f, 0xf, 0x3},
                       new int[] {0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x2, 0x1, 0x1,
                                  0x1, 0x1, 0x1, 0x1, 0x2});
      break;
    case 16:
      fastUnpackEventBit(
          in, inpos, out, outpos, 16,
          new int[] {0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10},
          new int[] {0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff},
          new int[] {0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1});
      break;
    case 17:
      fastUnpackOddBit(in, inpos, out, outpos, 17,
                       new int[] {0x2, 0x4, 0x6, 0x8, 0xa, 0xc, 0xe, 0x10, 0x1,
                                  0x3, 0x5, 0x7, 0x9, 0xb, 0xd, 0xf},
                       new int[] {0x3, 0xf, 0x3f, 0xff, 0x3ff, 0xfff, 0x3fff,
                                  0xffff, 0x1, 0x7, 0x1f, 0x7f, 0x1ff, 0x7ff,
                                  0x1fff, 0x7fff},
                       new int[] {0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x0, 0x1,
                                  0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1});
      break;
    case 18:
      fastUnpackEventBit(
          in, inpos, out, outpos, 18,
          new int[] {0x4, 0x8, 0xc, 0x10, 0x2, 0x6, 0xa, 0xe},
          new int[] {0xf, 0xff, 0xfff, 0xffff, 0x3, 0x3f, 0x3ff, 0x3fff},
          new int[] {0x1, 0x1, 0x1, 0x0, 0x1, 0x1, 0x1, 0x1});
      break;
    case 19:
      fastUnpackOddBit(in, inpos, out, outpos, 19,
                       new int[] {0x6, 0xc, 0x12, 0x5, 0xb, 0x11, 0x4, 0xa,
                                  0x10, 0x3, 0x9, 0xf, 0x2, 0x8, 0xe, 0x1, 0x7,
                                  0xd},
                       new int[] {0x3f, 0xfff, 0x3ffff, 0x1f, 0x7ff, 0x1ffff,
                                  0xf, 0x3ff, 0xffff, 0x7, 0x1ff, 0x7fff, 0x3,
                                  0xff, 0x3fff, 0x1, 0x7f, 0x1fff},
                       new int[] {0x1, 0x1, 0x0, 0x1, 0x1, 0x0, 0x1, 0x1, 0x0,
                                  0x1, 0x1, 0x0, 0x1, 0x1, 0x0, 0x1, 0x1, 0x1});
      break;
    case 20:
      fastUnpackEventBit(
          in, inpos, out, outpos, 20,
          new int[] {0x8, 0x10, 0x4, 0xc, 0x14, 0x8, 0x10, 0x4, 0xc},
          new int[] {0xff, 0xffff, 0xf, 0xfff, 0xfffff, 0xff, 0xffff, 0xf,
                     0xfff},
          new int[] {0x1, 0x0, 0x1, 0x1, 0x0, 0x1, 0x0, 0x1, 0x1});
      break;
    case 21:
      fastUnpackOddBit(
          in, inpos, out, outpos, 21,
          new int[] {0xa, 0x14, 0x9, 0x13, 0x8, 0x12, 0x7, 0x11, 0x6, 0x10,
                     0x5, 0xf,  0x4, 0xe,  0x3, 0xd,  0x2, 0xc,  0x1, 0xb},
          new int[] {0x3ff,   0xfffff, 0x1ff,  0x7ffff, 0xff,   0x3ffff, 0x7f,
                     0x1ffff, 0x3f,    0xffff, 0x1f,    0x7fff, 0xf,     0x3fff,
                     0x7,     0x1fff,  0x3,    0xfff,   0x1,    0x7ff},
          new int[] {0x1, 0x0, 0x1, 0x0, 0x1, 0x0, 0x1, 0x0, 0x1, 0x0,
                     0x1, 0x0, 0x1, 0x0, 0x1, 0x0, 0x1, 0x0, 0x1, 0x1});
      break;
    case 22:
      fastUnpackEventBit(
          in, inpos, out, outpos, 22,
          new int[] {0xc, 0x2, 0xe, 0x4, 0x10, 0x6, 0x12, 0x8, 0x14, 0xa},
          new int[] {0xfff, 0x3, 0x3fff, 0xf, 0xffff, 0x3f, 0x3ffff, 0xff,
                     0xfffff, 0x3ff},
          new int[] {0x0, 0x1, 0x0, 0x1, 0x0, 0x1, 0x0, 0x1, 0x0, 0x1});
      break;
    case 23:
      fastUnpackOddBit(
          in, inpos, out, outpos, 23,
          new int[] {0xe, 0x5,  0x13, 0xa, 0x1,  0xf, 0x6, 0x14,
                     0xb, 0x2,  0x10, 0x7, 0x15, 0xc, 0x3, 0x11,
                     0x8, 0x16, 0xd,  0x4, 0x12, 0x9},
          new int[] {0x3fff,   0x1f,    0x7ffff, 0x3ff,   0x1,    0x7fff,
                     0x3f,     0xfffff, 0x7ff,   0x3,     0xffff, 0x7f,
                     0x1fffff, 0xfff,   0x7,     0x1ffff, 0xff,   0x3fffff,
                     0x1fff,   0xf,     0x3ffff, 0x1ff},
          new int[] {0x0, 0x1, 0x0, 0x0, 0x1, 0x0, 0x1, 0x0, 0x0, 0x1, 0x0,
                     0x1, 0x0, 0x0, 0x1, 0x0, 0x1, 0x0, 0x0, 0x1, 0x0, 0x1});
      break;
    case 24:
      fastUnpackEventBit(
          in, inpos, out, outpos, 24,
          new int[] {0x10, 0x8, 0x18, 0x10, 0x8, 0x18, 0x10, 0x8, 0x18, 0x10,
                     0x8},
          new int[] {0xffff, 0xff, 0xffffff, 0xffff, 0xff, 0xffffff, 0xffff,
                     0xff, 0xffffff, 0xffff, 0xff},
          new int[] {0x0, 0x1, 0x0, 0x0, 0x1, 0x0, 0x0, 0x1, 0x0, 0x0, 0x1});
      break;
    case 25:
      fastUnpackOddBit(
          in, inpos, out, outpos, 25,
          new int[] {0x12, 0xb,  0x4,  0x16, 0xf, 0x8,  0x1,  0x13,
                     0xc,  0x5,  0x17, 0x10, 0x9, 0x2,  0x14, 0xd,
                     0x6,  0x18, 0x11, 0xa,  0x3, 0x15, 0xe,  0x7},
          new int[] {0x3ffff, 0x7ff,   0xf,     0x3fffff, 0x7fff,   0xff,
                     0x1,     0x7ffff, 0xfff,   0x1f,     0x7fffff, 0xffff,
                     0x1ff,   0x3,     0xfffff, 0x1fff,   0x3f,     0xffffff,
                     0x1ffff, 0x3ff,   0x7,     0x1fffff, 0x3fff,   0x7f},
          new int[] {0x0, 0x0, 0x1, 0x0, 0x0, 0x0, 0x1, 0x0,
                     0x0, 0x1, 0x0, 0x0, 0x0, 0x1, 0x0, 0x0,
                     0x1, 0x0, 0x0, 0x0, 0x1, 0x0, 0x0, 0x1});
      break;
    case 26:
      fastUnpackEventBit(in, inpos, out, outpos, 26,
                         new int[] {0x14, 0xe, 0x8, 0x2, 0x16, 0x10, 0xa, 0x4,
                                    0x18, 0x12, 0xc, 0x6},
                         new int[] {0xfffff, 0x3fff, 0xff, 0x3, 0x3fffff,
                                    0xffff, 0x3ff, 0xf, 0xffffff, 0x3ffff,
                                    0xfff, 0x3f},
                         new int[] {0x0, 0x0, 0x0, 0x1, 0x0, 0x0, 0x0, 0x1, 0x0,
                                    0x0, 0x0, 0x1});
      break;
    case 27:
      fastUnpackOddBit(
          in, inpos, out, outpos, 27,
          new int[] {0x16, 0x11, 0xc,  0x7,  0x2,  0x18, 0x13, 0xe,  0x9,
                     0x4,  0x1a, 0x15, 0x10, 0xb,  0x6,  0x1,  0x17, 0x12,
                     0xd,  0x8,  0x3,  0x19, 0x14, 0xf,  0xa,  0x5},
          new int[] {0x3fffff, 0x1ffff, 0xfff, 0x7f,      0x3,       0xffffff,
                     0x7ffff,  0x3fff,  0x1ff, 0xf,       0x3ffffff, 0x1fffff,
                     0xffff,   0x7ff,   0x3f,  0x1,       0x7fffff,  0x3ffff,
                     0x1fff,   0xff,    0x7,   0x1ffffff, 0xfffff,   0x7fff,
                     0x3ff,    0x1f},
          new int[] {0x0, 0x0, 0x0, 0x0, 0x1, 0x0, 0x0, 0x0, 0x0,
                     0x1, 0x0, 0x0, 0x0, 0x0, 0x0, 0x1, 0x0, 0x0,
                     0x0, 0x0, 0x1, 0x0, 0x0, 0x0, 0x0, 0x1});
      break;
    case 28:
      fastUnpackEventBit(in, inpos, out, outpos, 28,
                         new int[] {0x18, 0x14, 0x10, 0xc, 0x8, 0x4, 0x1c, 0x18,
                                    0x14, 0x10, 0xc, 0x8, 0x4},
                         new int[] {0xffffff, 0xfffff, 0xffff, 0xfff, 0xff, 0xf,
                                    0xfffffff, 0xffffff, 0xfffff, 0xffff, 0xfff,
                                    0xff, 0xf},
                         new int[] {0x0, 0x0, 0x0, 0x0, 0x0, 0x1, 0x0, 0x0, 0x0,
                                    0x0, 0x0, 0x0, 0x1});
      break;
    case 29:
      fastUnpackOddBit(
          in, inpos, out, outpos, 29,
          new int[] {0x1a, 0x17, 0x14, 0x11, 0xe, 0xb, 0x8, 0x5, 0x2, 0x1c,
                     0x19, 0x16, 0x13, 0x10, 0xd, 0xa, 0x7, 0x4, 0x1, 0x1b,
                     0x18, 0x15, 0x12, 0xf,  0xc, 0x9, 0x6, 0x3},
          new int[] {0x3ffffff, 0x7fffff, 0xfffff, 0x1ffff, 0x3fff,
                     0x7ff,     0xff,     0x1f,    0x3,     0xfffffff,
                     0x1ffffff, 0x3fffff, 0x7ffff, 0xffff,  0x1fff,
                     0x3ff,     0x7f,     0xf,     0x1,     0x7ffffff,
                     0xffffff,  0x1fffff, 0x3ffff, 0x7fff,  0xfff,
                     0x1ff,     0x3f,     0x7},
          new int[] {0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x1, 0x0,
                     0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x1, 0x0,
                     0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x1});
      break;
    case 30:
      fastUnpackEventBit(in, inpos, out, outpos, 30,
                         new int[] {0x1c, 0x1a, 0x18, 0x16, 0x14, 0x12, 0x10,
                                    0xe, 0xc, 0xa, 0x8, 0x6, 0x4, 0x2},
                         new int[] {0xfffffff, 0x3ffffff, 0xffffff, 0x3fffff,
                                    0xfffff, 0x3ffff, 0xffff, 0x3fff, 0xfff,
                                    0x3ff, 0xff, 0x3f, 0xf, 0x3},
                         new int[] {0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
                                    0x0, 0x0, 0x0, 0x0, 0x1});
      break;
    case 31:
      fastUnpackOddBit(
          in, inpos, out, outpos, 31,
          new int[] {0x1e, 0x1d, 0x1c, 0x1b, 0x1a, 0x19, 0x18, 0x17, 0x16, 0x15,
                     0x14, 0x13, 0x12, 0x11, 0x10, 0xf,  0xe,  0xd,  0xc,  0xb,
                     0xa,  0x9,  0x8,  0x7,  0x6,  0x5,  0x4,  0x3,  0x2,  0x1},
          new int[] {0x3fffffff, 0x1fffffff, 0xfffffff, 0x7ffffff, 0x3ffffff,
                     0x1ffffff,  0xffffff,   0x7fffff,  0x3fffff,  0x1fffff,
                     0xfffff,    0x7ffff,    0x3ffff,   0x1ffff,   0xffff,
                     0x7fff,     0x3fff,     0x1fff,    0xfff,     0x7ff,
                     0x3ff,      0x1ff,      0xff,      0x7f,      0x3f,
                     0x1f,       0xf,        0x7,       0x3,       0x1},
          new int[] {0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
                     0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
                     0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x1});
      break;

    case 32:
      System.arraycopy(in, inpos, out, outpos, 256);
      break;
    }
  }
}
