/*
 * Decompiled with CFR 0.152.
 */
package org.evomaster.client.java.sql;

import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.TreeMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import org.evomaster.client.java.controller.api.dto.database.schema.ColumnDto;
import org.evomaster.client.java.controller.api.dto.database.schema.CompositeTypeColumnDto;
import org.evomaster.client.java.controller.api.dto.database.schema.CompositeTypeDto;
import org.evomaster.client.java.controller.api.dto.database.schema.DatabaseType;
import org.evomaster.client.java.controller.api.dto.database.schema.DbInfoDto;
import org.evomaster.client.java.controller.api.dto.database.schema.EnumeratedTypeDto;
import org.evomaster.client.java.controller.api.dto.database.schema.ForeignKeyDto;
import org.evomaster.client.java.controller.api.dto.database.schema.TableCheckExpressionDto;
import org.evomaster.client.java.controller.api.dto.database.schema.TableDto;
import org.evomaster.client.java.sql.SchemasToSkip;
import org.evomaster.client.java.sql.internal.constraint.DbTableCheckExpression;
import org.evomaster.client.java.sql.internal.constraint.DbTableConstraint;
import org.evomaster.client.java.sql.internal.constraint.DbTableUniqueConstraint;
import org.evomaster.client.java.sql.internal.constraint.TableConstraintExtractor;
import org.evomaster.client.java.sql.internal.constraint.TableConstraintExtractorFactory;
import org.evomaster.client.java.utils.SimpleLogger;

public class DbInfoExtractor {
    public static final String GEOMETRY = "GEOMETRY";
    private static final String TEXT_DATA_TYPE = "text";
    private static final String BIT_DATA_TYPE = "bit";
    private static final String VAR_BIT_DATA_TYPE = "varbit";
    private static final String CHAR_DATA_TYPE = "char";
    private static final String VAR_CHAR_DATA_TYPE = "varchar";
    private static final String BLANK_PADDED_CHAR_DATA_TYPE = "bpchar";
    private static final String NUMERIC_DATA_TYPE = "numeric";

    public static boolean validate(DbInfoDto schema) throws IllegalArgumentException {
        Objects.requireNonNull(schema);
        for (TableDto table : schema.tables) {
            for (ColumnDto column : table.columns) {
                DbInfoExtractor.checkForeignKeyToAutoIncrementPresent(schema, table, column);
                DbInfoExtractor.checkForeignKeyToAutoIncrementMissing(schema, table, column);
                DbInfoExtractor.checkEnumeratedTypeIsDefined(schema, table, column);
            }
        }
        return true;
    }

    private static void checkEnumeratedTypeIsDefined(DbInfoDto schema, TableDto table, ColumnDto column) {
        if (column.isEnumeratedType && schema.enumeraredTypes.stream().noneMatch(k -> k.name.equals(column.type))) {
            throw new IllegalArgumentException("Missing enumerated type declaration for type " + column.type + " in column " + column.name + " of table " + table.name);
        }
    }

    private static void checkForeignKeyToAutoIncrementMissing(DbInfoDto schema, TableDto table, ColumnDto column) {
        if (column.foreignKeyToAutoIncrement) {
            return;
        }
        Optional<ForeignKeyDto> fk = table.foreignKeys.stream().filter(it -> it.sourceColumns.contains(column.name)).findFirst();
        if (!fk.isPresent()) {
            return;
        }
        Optional<TableDto> targetTable = schema.tables.stream().filter(t -> t.name.equals(((ForeignKeyDto)fk.get()).targetTable)).findFirst();
        if (!targetTable.isPresent()) {
            throw new IllegalArgumentException("Foreign key in table " + table.name + " pointing to non-existent table " + fk.get().targetTable);
        }
        List pks = targetTable.get().columns.stream().filter(c -> c.primaryKey).collect(Collectors.toList());
        if (pks.isEmpty()) {
            throw new IllegalArgumentException("No PK in table " + targetTable.get().name + " that has FKs pointing to it");
        }
        for (ColumnDto pk : pks) {
            if (!pk.autoIncrement && !pk.foreignKeyToAutoIncrement) continue;
            throw new IllegalArgumentException("Column " + pk.name + " in table " + pk.table + " is auto-increment, although FK pointing to it does not mark it as autoincrement in " + column.name + " in " + table.name);
        }
    }

    private static void checkForeignKeyToAutoIncrementPresent(DbInfoDto schema, TableDto table, ColumnDto column) {
        if (!column.foreignKeyToAutoIncrement) {
            return;
        }
        Optional<ForeignKeyDto> fk = table.foreignKeys.stream().filter(it -> it.sourceColumns.contains(column.name)).findFirst();
        if (!fk.isPresent()) {
            throw new IllegalArgumentException("No foreign key constraint for marked column " + column.name + " in table " + table.name);
        }
        Optional<TableDto> targetTable = schema.tables.stream().filter(t -> t.name.equals(((ForeignKeyDto)fk.get()).targetTable)).findFirst();
        if (!targetTable.isPresent()) {
            throw new IllegalArgumentException("Foreign key in table " + table.name + " pointing to non-existent table " + fk.get().targetTable);
        }
        List pks = targetTable.get().columns.stream().filter(c -> c.primaryKey).collect(Collectors.toList());
        if (pks.size() != 1) {
            throw new IllegalArgumentException("There must be only 1 PK in table " + targetTable.get().name + " pointed by the FK-to-autoincrement " + column.name + " in " + table.name + ". However, there were: " + pks.size());
        }
        ColumnDto pk = (ColumnDto)pks.get(0);
        if (!pk.autoIncrement && !pk.foreignKeyToAutoIncrement) {
            throw new IllegalArgumentException("Column " + pk.name + " in table " + pk.table + " is not auto-increment, although FK pointing to it does mark itas autoincrement in " + column.name + " in " + table.name);
        }
    }

    public static DbInfoDto extract(Connection connection) throws Exception {
        Objects.requireNonNull(connection);
        DbInfoDto dbInfoDto = new DbInfoDto();
        DatabaseMetaData md = connection.getMetaData();
        String protocol = md.getURL();
        DatabaseType dt = DatabaseType.OTHER;
        if (protocol.contains(":h2")) {
            dt = DatabaseType.H2;
        } else if (protocol.contains(":derby")) {
            dt = DatabaseType.DERBY;
        } else if (protocol.contains(":postgresql")) {
            dt = DatabaseType.POSTGRES;
        } else if (protocol.contains(":mysql")) {
            dt = DatabaseType.MYSQL;
        }
        dbInfoDto.databaseType = dt;
        dbInfoDto.name = connection.getCatalog();
        if (dt.equals((Object)DatabaseType.POSTGRES)) {
            Map<String, Set<String>> enumLabels = DbInfoExtractor.getPostgresEnumTypes(connection);
            DbInfoExtractor.addPostgresEnumTypesToSchema(dbInfoDto, enumLabels);
            dbInfoDto.compositeTypes = DbInfoExtractor.getPostgresCompositeTypes(connection);
        }
        ResultSet tables = dt.equals((Object)DatabaseType.MYSQL) ? md.getTables(null, null, null, new String[]{"TABLE"}) : md.getTables(dbInfoDto.name, null, null, new String[]{"TABLE"});
        HashSet<String> tableNames = new HashSet<String>();
        if (!tables.next()) {
            tables.close();
            dbInfoDto.name = dbInfoDto.name.toLowerCase();
            tables = md.getTables(dbInfoDto.name, null, null, new String[]{"TABLE"});
            if (tables.next()) {
                do {
                    DbInfoExtractor.handleTableEntry(connection, dbInfoDto, md, tables, tableNames);
                } while (tables.next());
            }
        } else {
            do {
                DbInfoExtractor.handleTableEntry(connection, dbInfoDto, md, tables, tableNames);
            } while (tables.next());
        }
        tables.close();
        DbInfoExtractor.addForeignKeyToAutoIncrement(dbInfoDto);
        DbInfoExtractor.addConstraints(connection, dt, dbInfoDto);
        if (dt.equals((Object)DatabaseType.POSTGRES)) {
            List<ColumnAttributes> columnAttributes = DbInfoExtractor.getPostgresColumnAttributes(connection);
            DbInfoExtractor.addColumnAttributes(dbInfoDto, columnAttributes);
        } else if (dt.equals((Object)DatabaseType.H2)) {
            List<DbTableConstraint> h2EnumConstraints = DbInfoExtractor.getH2EnumTypes(dbInfoDto.name, md);
            DbInfoExtractor.addConstraints(dbInfoDto, h2EnumConstraints);
        }
        assert (DbInfoExtractor.validate(dbInfoDto));
        return dbInfoDto;
    }

    private static void addColumnAttributes(DbInfoDto schemaDto, List<ColumnAttributes> listOfColumnAttributes) {
        for (ColumnAttributes columnAttributes : listOfColumnAttributes) {
            String tableName = columnAttributes.tableName;
            String columnName = columnAttributes.columnName;
            ColumnDto columnDto = DbInfoExtractor.getColumnDto(schemaDto, tableName, columnName);
            columnDto.numberOfDimensions = columnAttributes.numberOfDimensions;
        }
    }

    private static ColumnDto getColumnDto(DbInfoDto schemaDto, String tableName, String columnName) {
        TableDto tableDto = schemaDto.tables.stream().filter(t -> t.name.equals(tableName.toLowerCase())).findFirst().orElse(null);
        return tableDto.columns.stream().filter(c -> c.name.equals(columnName.toLowerCase())).findFirst().orElse(null);
    }

    private static String getSchemaName(Connection connection, DatabaseType dt) throws SQLException {
        String schemaName;
        if (dt.equals((Object)DatabaseType.MYSQL)) {
            String getSchemaQuery = "SELECT DATABASE() AS schema_name";
            try (Statement getSchemaStmt = connection.createStatement();){
                ResultSet getSchemaResultSet = getSchemaStmt.executeQuery(getSchemaQuery);
                if (getSchemaResultSet.next()) {
                    schemaName = getSchemaResultSet.getString("schema_name");
                }
                if (connection.getSchema() == null) {
                    schemaName = connection.getCatalog();
                }
                schemaName = connection.getSchema();
            }
        } else {
            try {
                schemaName = connection.getSchema();
            }
            catch (AbstractMethodError | Exception e) {
                schemaName = "public";
            }
            schemaName = schemaName.toUpperCase();
        }
        return schemaName;
    }

    private static List<ColumnAttributes> getPostgresColumnAttributes(Connection connection) throws SQLException {
        String query = "SELECT pg_namespace.nspname as TABLE_NAMESPACE, pg_class.relname as TABLE_NAME, pg_attribute.attname as COLUMN_NAME, pg_attribute.attndims as NUMBER_OF_DIMENSIONS \nFROM pg_attribute \nINNER JOIN pg_class ON pg_class.oid = pg_attribute.attrelid INNER JOIN pg_namespace ON pg_namespace.oid = pg_class.relnamespace WHERE pg_namespace.nspname != 'pg_catalog' ";
        LinkedList<ColumnAttributes> listOfColumnAttributes = new LinkedList<ColumnAttributes>();
        try (Statement stmt = connection.createStatement();){
            ResultSet columnAttributesResultSet = stmt.executeQuery(query);
            while (columnAttributesResultSet.next()) {
                String tableNamesapce = columnAttributesResultSet.getString("TABLE_NAMESPACE");
                String tableName = columnAttributesResultSet.getString("TABLE_NAME");
                String columnName = columnAttributesResultSet.getString("COLUMN_NAME");
                int numberOfDimensions = columnAttributesResultSet.getInt("NUMBER_OF_DIMENSIONS");
                if (numberOfDimensions == 0) continue;
                ColumnAttributes columnAttributes = new ColumnAttributes();
                columnAttributes.tableName = tableName;
                columnAttributes.columnName = columnName;
                columnAttributes.numberOfDimensions = numberOfDimensions;
                listOfColumnAttributes.add(columnAttributes);
            }
        }
        return listOfColumnAttributes;
    }

    private static List<String> getAllCompositeTypeNames(Connection connection) throws SQLException {
        String listAllCompositeTypesQuery = "SELECT      n.nspname as schema, t.typname as typename \nFROM        pg_type t \nLEFT JOIN   pg_catalog.pg_namespace n ON n.oid = t.typnamespace \nWHERE       (t.typrelid = 0 OR (SELECT c.relkind = 'c' FROM pg_catalog.pg_class c WHERE c.oid = t.typrelid)) \nAND     NOT EXISTS(SELECT 1 FROM pg_catalog.pg_type el WHERE el.oid = t.typelem AND el.typarray = t.oid)\nAND     n.nspname NOT IN ('pg_catalog', 'information_schema')AND     t.typtype ='c';";
        ArrayList<String> compositeTypeNames = new ArrayList<String>();
        try (Statement listAllCompositeTypesStmt = connection.createStatement();){
            ResultSet listAllCompositeTypesResultSet = listAllCompositeTypesStmt.executeQuery(listAllCompositeTypesQuery);
            while (listAllCompositeTypesResultSet.next()) {
                compositeTypeNames.add(listAllCompositeTypesResultSet.getString("typename"));
            }
        }
        return compositeTypeNames;
    }

    private static List<CompositeTypeDto> getPostgresCompositeTypes(Connection connection) throws SQLException {
        ArrayList<CompositeTypeDto> compositeTypeDtos = new ArrayList<CompositeTypeDto>();
        List<String> compositeTypeNames = DbInfoExtractor.getAllCompositeTypeNames(connection);
        for (String compositeTypeName : compositeTypeNames) {
            List<CompositeTypeColumnDto> columnDtos = DbInfoExtractor.getAllCompositeTypeColumns(connection, compositeTypeName, compositeTypeNames);
            CompositeTypeDto compositeTypeDto = new CompositeTypeDto();
            compositeTypeDto.name = compositeTypeName;
            compositeTypeDto.columns = columnDtos;
            compositeTypeDtos.add(compositeTypeDto);
        }
        return compositeTypeDtos;
    }

    private static List<CompositeTypeColumnDto> getAllCompositeTypeColumns(Connection connection, String compositeTypeName, List<String> allCompositeTypeNames) throws SQLException {
        String listAttributesQuery = String.format("SELECT pg_attribute.attname AS attname, pg_attribute.attlen  as attlen, pg_type.typname AS typename  FROM pg_attribute  JOIN pg_type ON pg_attribute.atttypid=pg_type.oid  WHERE pg_attribute.attrelid =\n  (SELECT typrelid FROM pg_type WHERE typname = '%s')  ORDER BY pg_attribute.attnum ", compositeTypeName);
        ArrayList<CompositeTypeColumnDto> columnDtos = new ArrayList<CompositeTypeColumnDto>();
        try (Statement listAttributesStmt = connection.createStatement();){
            ResultSet listAttributesResultSet = listAttributesStmt.executeQuery(listAttributesQuery);
            while (listAttributesResultSet.next()) {
                CompositeTypeColumnDto columnDto = new CompositeTypeColumnDto();
                columnDto.name = listAttributesResultSet.getString("attname");
                columnDto.type = listAttributesResultSet.getString("typename");
                int attlen = listAttributesResultSet.getInt("attlen");
                columnDto.nullable = true;
                columnDto.columnTypeIsComposite = allCompositeTypeNames.stream().anyMatch(t -> t.equalsIgnoreCase(columnDto.type));
                if (columnDto.columnTypeIsComposite) {
                    columnDto.size = 0;
                } else {
                    switch (columnDto.type) {
                        case "text": {
                            columnDto.size = Integer.MAX_VALUE;
                            break;
                        }
                        case "numeric": 
                        case "char": 
                        case "varchar": 
                        case "bit": 
                        case "varbit": 
                        case "bpchar": {
                            throw new UnsupportedOperationException("cannot get variable length size of type (varchar, char, varbit, bit, numeric) currently not supported for postgres composite types: " + columnDto.name + " with type " + columnDto.type);
                        }
                        default: {
                            columnDto.size = attlen;
                        }
                    }
                }
                columnDtos.add(columnDto);
            }
        }
        return columnDtos;
    }

    private static List<DbTableConstraint> getH2EnumTypes(String catalogName, DatabaseMetaData md) throws SQLException {
        LinkedList<DbTableConstraint> enumTypesConstraints = new LinkedList<DbTableConstraint>();
        ResultSet tables = md.getTables(catalogName, null, null, new String[]{"TABLE"});
        while (tables.next()) {
            String tableName = tables.getString("TABLE_NAME");
            ResultSet columns = md.getColumns(catalogName, null, tableName, null);
            while (columns.next()) {
                String columnName = columns.getString("COLUMN_NAME");
                String typeName = columns.getString("TYPE_NAME");
                if (!typeName.startsWith("ENUM")) continue;
                String sqlExpression = String.format("(\"%s\" IN %s)", columnName, typeName.substring("ENUM".length()));
                DbTableCheckExpression constraint = new DbTableCheckExpression(tableName, sqlExpression);
                enumTypesConstraints.add(constraint);
            }
        }
        return enumTypesConstraints;
    }

    private static Map<String, Set<String>> getPostgresEnumTypes(Connection connection) throws SQLException {
        String query = "SELECT t.typname, e.enumlabel\nFROM pg_type AS t\n   JOIN pg_enum AS e ON t.oid = e.enumtypid\nORDER BY e.enumsortorder;";
        LinkedHashMap<String, Set<String>> enumLabels = new LinkedHashMap<String, Set<String>>();
        try (Statement stmt = connection.createStatement();){
            ResultSet enumTypeValues = stmt.executeQuery(query);
            while (enumTypeValues.next()) {
                String typeName = enumTypeValues.getString("typname");
                String enumLabel = enumTypeValues.getString("enumlabel");
                if (!enumLabels.containsKey(typeName)) {
                    enumLabels.put(typeName, new HashSet());
                }
                ((Set)enumLabels.get(typeName)).add(enumLabel);
            }
        }
        return enumLabels;
    }

    private static void addPostgresEnumTypesToSchema(DbInfoDto schemaDto, Map<String, Set<String>> enumLabels) {
        enumLabels.forEach((k, v) -> {
            EnumeratedTypeDto enumeratedTypeDto = new EnumeratedTypeDto();
            enumeratedTypeDto.name = k;
            enumeratedTypeDto.values = new ArrayList(v);
            schemaDto.enumeraredTypes.add(enumeratedTypeDto);
        });
    }

    public static void addUniqueConstraintToColumn(TableDto tableDto, String columnName) {
        ColumnDto columnDto = tableDto.columns.stream().filter(c -> c.name.equals(columnName)).findAny().orElse(null);
        if (columnDto == null) {
            throw new IllegalArgumentException("Missing column DTO for column:" + tableDto.name + "." + columnName);
        }
        columnDto.unique = true;
    }

    private static void addConstraints(Connection connection, DatabaseType dt, DbInfoDto schemaDto) throws SQLException {
        TableConstraintExtractor constraintExtractor = TableConstraintExtractorFactory.buildConstraintExtractor(dt);
        if (constraintExtractor != null) {
            List<DbTableConstraint> dbTableConstraints = constraintExtractor.extract(connection, schemaDto);
            DbInfoExtractor.addConstraints(schemaDto, dbTableConstraints);
        } else {
            SimpleLogger.uniqueWarn((String)("WARNING: EvoMaster cannot extract constraints from database " + dt));
        }
    }

    private static void addConstraints(DbInfoDto schemaDto, List<DbTableConstraint> constraintList) {
        for (DbTableConstraint constraint : constraintList) {
            String tableName = constraint.getTableName();
            TableDto tableDto = schemaDto.tables.stream().filter(t -> t.name.equalsIgnoreCase(tableName)).findFirst().orElse(null);
            if (tableDto == null) {
                throw new NullPointerException("TableDto for table " + tableName + " was not found in the schemaDto");
            }
            if (constraint instanceof DbTableCheckExpression) {
                TableCheckExpressionDto constraintDto = new TableCheckExpressionDto();
                DbTableCheckExpression tableCheckExpression = (DbTableCheckExpression)constraint;
                constraintDto.sqlCheckExpression = tableCheckExpression.getSqlCheckExpression();
                tableDto.tableCheckExpressions.add(constraintDto);
                continue;
            }
            if (constraint instanceof DbTableUniqueConstraint) {
                DbTableUniqueConstraint tableUniqueConstraint = (DbTableUniqueConstraint)constraint;
                for (String columnName : tableUniqueConstraint.getUniqueColumnNames()) {
                    DbInfoExtractor.addUniqueConstraintToColumn(tableDto, columnName);
                }
                continue;
            }
            throw new RuntimeException("Unknown constraint type " + constraint.getClass().getName());
        }
    }

    private static String getId(TableDto dto) {
        if (dto.schema == null) {
            return dto.name;
        }
        return dto.schema + "." + dto.name;
    }

    private static void handleTableEntry(Connection connection, DbInfoDto schemaDto, DatabaseMetaData md, ResultSet tables, Set<String> tableIds) throws SQLException {
        List<String> toSkip;
        String tableCatalog = tables.getString("TABLE_CAT");
        String tableSchema = tables.getString("TABLE_SCHEM");
        DatabaseType type = schemaDto.databaseType;
        if (tableSchema == null) {
            if (type.equals((Object)DatabaseType.MYSQL)) {
                tableSchema = tableCatalog;
            } else if (type.equals((Object)DatabaseType.POSTGRES) || type.equals((Object)DatabaseType.H2)) {
                tableSchema = "public";
            }
        }
        if ((toSkip = SchemasToSkip.get(type)) != null && toSkip.contains(tableSchema)) {
            return;
        }
        TableDto tableDto = new TableDto();
        schemaDto.tables.add(tableDto);
        tableDto.name = tables.getString("TABLE_NAME");
        tableDto.schema = tableSchema;
        tableDto.catalog = tableCatalog;
        if (tableIds.contains(DbInfoExtractor.getId(tableDto))) {
            throw new IllegalArgumentException("Cannot handle repeated table " + DbInfoExtractor.getId(tableDto) + " in database");
        }
        tableIds.add(DbInfoExtractor.getId(tableDto));
        HashSet<String> pks = new HashSet<String>();
        TreeMap<Integer, String> primaryKeySequence = new TreeMap<Integer, String>();
        ResultSet rsPK = md.getPrimaryKeys(tableDto.catalog, tableDto.schema, tableDto.name);
        while (rsPK.next()) {
            String pkColumnName = rsPK.getString("COLUMN_NAME");
            short positionInPrimaryKey = rsPK.getShort("KEY_SEQ");
            pks.add(pkColumnName);
            int pkIndex = positionInPrimaryKey - 1;
            primaryKeySequence.put(pkIndex, pkColumnName);
        }
        rsPK.close();
        tableDto.primaryKeySequence.addAll(primaryKeySequence.values());
        ResultSet columns = md.getColumns(tableDto.catalog, tableDto.schema, tableDto.name, null);
        HashSet<String> columnNames = new HashSet<String>();
        while (columns.next()) {
            ColumnDto columnDto = new ColumnDto();
            tableDto.columns.add(columnDto);
            columnDto.table = tableDto.name;
            columnDto.name = columns.getString("COLUMN_NAME");
            if (columnNames.contains(columnDto.name)) {
                throw new IllegalArgumentException("Cannot handle repeated column " + columnDto.name + " in table " + tableDto.name);
            }
            columnNames.add(columnDto.name);
            String typeAsString = columns.getString("TYPE_NAME");
            columnDto.size = columns.getInt("COLUMN_SIZE");
            switch (schemaDto.databaseType) {
                case MYSQL: {
                    DbInfoExtractor.extractMySQLColumn(schemaDto, tableDto, columnDto, typeAsString, columns, connection);
                    break;
                }
                case POSTGRES: {
                    DbInfoExtractor.extractPostgresColumn(schemaDto, columnDto, typeAsString, columns);
                    break;
                }
                case H2: {
                    DbInfoExtractor.extractH2Column(columnDto, typeAsString, columns);
                    break;
                }
                default: {
                    columnDto.nullable = columns.getBoolean("IS_NULLABLE");
                    columnDto.autoIncrement = columns.getBoolean("IS_AUTOINCREMENT");
                    columnDto.type = typeAsString;
                }
            }
            columnDto.primaryKey = pks.contains(columnDto.name);
        }
        columns.close();
        ResultSet fks = md.getImportedKeys(tableDto.catalog, tableDto.schema, tableDto.name);
        while (fks.next()) {
            ForeignKeyDto fkDto = new ForeignKeyDto();
            fkDto.sourceColumns.add(fks.getString("FKCOLUMN_NAME"));
            fkDto.targetTable = fks.getString("PKTABLE_NAME");
            tableDto.foreignKeys.add(fkDto);
        }
        fks.close();
    }

    private static void extractH2Column(ColumnDto columnDto, String typeAsString, ResultSet columns) throws SQLException {
        columnDto.nullable = columns.getBoolean("IS_NULLABLE");
        columnDto.autoIncrement = columns.getBoolean("IS_AUTOINCREMENT");
        if (typeAsString.startsWith("ENUM")) {
            columnDto.type = "VARCHAR";
        } else if (typeAsString.contains("ARRAY")) {
            columnDto.type = DbInfoExtractor.getH2ArrayBaseType(typeAsString);
            columnDto.numberOfDimensions = DbInfoExtractor.getH2ArrayNumberOfDimensions(typeAsString);
        } else {
            columnDto.type = typeAsString;
        }
    }

    private static void extractPostgresColumn(DbInfoDto schemaDto, ColumnDto columnDto, String typeAsString, ResultSet columns) throws SQLException {
        columnDto.nullable = columns.getBoolean("IS_NULLABLE");
        columnDto.autoIncrement = columns.getBoolean("IS_AUTOINCREMENT");
        columnDto.type = typeAsString;
        columnDto.isEnumeratedType = schemaDto.enumeraredTypes.stream().anyMatch(k -> k.name.equals(typeAsString));
        columnDto.isCompositeType = schemaDto.compositeTypes.stream().anyMatch(k -> k.name.equals(typeAsString));
    }

    private static void extractMySQLColumn(DbInfoDto schemaDto, TableDto tableDto, ColumnDto columnDto, String typeAsStringValue, ResultSet columns, Connection connection) throws SQLException {
        int decimalDigitsValue = columns.getInt("DECIMAL_DIGITS");
        int nullableValue = columns.getInt("NULLABLE");
        String isAutoIncrementValue = columns.getString("IS_AUTOINCREMENT");
        String[] attrs = typeAsStringValue.split(" ");
        if (attrs.length == 0) {
            throw new IllegalStateException("missing type info of the column");
        }
        if (attrs[0].equalsIgnoreCase(GEOMETRY)) {
            String sqlQuery = String.format("SELECT DATA_TYPE, table_schema from INFORMATION_SCHEMA.COLUMNS where\n table_schema = '%s' and table_name = '%s' and column_name= '%s' ", tableDto.schema, tableDto.name, columnDto.name);
            try (Statement statement = connection.createStatement();){
                ResultSet rs = statement.executeQuery(sqlQuery);
                if (rs.next()) {
                    String dataType = rs.getString("DATA_TYPE");
                    columnDto.type = dataType.toUpperCase();
                }
                columnDto.type = GEOMETRY;
            }
        } else {
            columnDto.type = attrs[0];
        }
        columnDto.isUnsigned = attrs.length > 1 && IntStream.range(1, attrs.length).anyMatch(i -> attrs[i].equalsIgnoreCase("UNSIGNED"));
        columnDto.nullable = nullableValue == 1;
        columnDto.autoIncrement = isAutoIncrementValue.equalsIgnoreCase("yes");
        if (columnDto.type.equals("DECIMAL")) {
            columnDto.scale = decimalDigitsValue;
            if (columnDto.scale < 0) {
                columnDto.scale = 0;
            }
        }
    }

    private static int getH2ArrayNumberOfDimensions(String typeAsString) {
        if (!typeAsString.contains("ARRAY")) {
            throw new IllegalArgumentException("Cannot get number of dimensions of non-array type " + typeAsString);
        }
        Pattern arrayOnlyPattern = Pattern.compile("ARRAY");
        Matcher arrayOnlyMatcher = arrayOnlyPattern.matcher(typeAsString);
        int numberOfDimensions = 0;
        while (arrayOnlyMatcher.find()) {
            ++numberOfDimensions;
        }
        return numberOfDimensions;
    }

    private static String getH2ArrayBaseType(String typeAsString) {
        if (!typeAsString.contains("ARRAY")) {
            throw new IllegalArgumentException("Cannot get base type from non-array type " + typeAsString);
        }
        Pattern pattern = Pattern.compile("\\s*ARRAY\\s*\\[\\s*\\d+\\s*\\]");
        Matcher matcher = pattern.matcher(typeAsString);
        if (matcher.find()) {
            throw new IllegalArgumentException("Cannot handle array type with maximum length " + typeAsString);
        }
        String baseType = typeAsString.replaceAll("ARRAY", "").trim();
        return baseType.trim();
    }

    private static void addForeignKeyToAutoIncrement(DbInfoDto schema) {
        for (TableDto tableDto : schema.tables) {
            for (ColumnDto columnDto : tableDto.columns) {
                if (!DbInfoExtractor.isFKToAutoIncrementColumn(schema, tableDto, columnDto.name)) continue;
                columnDto.foreignKeyToAutoIncrement = true;
            }
        }
    }

    private static TableDto getTable(DbInfoDto schema, String tableName) {
        return schema.tables.stream().filter(t -> t.name.equalsIgnoreCase(tableName)).findFirst().orElse(null);
    }

    private static ColumnDto getColumn(TableDto table, String columnName) {
        return table.columns.stream().filter(c -> c.name.equalsIgnoreCase(columnName)).findFirst().orElse(null);
    }

    private static boolean isFKToAutoIncrementColumn(DbInfoDto schema, TableDto tableDto, String columnName) {
        Objects.requireNonNull(schema);
        Objects.requireNonNull(tableDto);
        Objects.requireNonNull(columnName);
        if (tableDto.foreignKeys.stream().noneMatch(fk -> fk.sourceColumns.stream().anyMatch(s -> s.equalsIgnoreCase(columnName)))) {
            return false;
        }
        ColumnDto columnDto = DbInfoExtractor.getColumn(tableDto, columnName);
        if (columnDto.autoIncrement) {
            return false;
        }
        for (ForeignKeyDto fk2 : tableDto.foreignKeys) {
            if (!fk2.sourceColumns.stream().anyMatch(s -> s.equalsIgnoreCase(columnName))) continue;
            int positionInFKSequence = fk2.sourceColumns.indexOf(columnName);
            TableDto targetTableDto = DbInfoExtractor.getTable(schema, fk2.targetTable);
            String targetColumnName = (String)targetTableDto.primaryKeySequence.get(positionInFKSequence);
            ColumnDto targetColumnDto = DbInfoExtractor.getColumn(targetTableDto, targetColumnName);
            if (!targetColumnDto.autoIncrement && !DbInfoExtractor.isFKToAutoIncrementColumn(schema, targetTableDto, targetColumnName)) continue;
            return true;
        }
        return false;
    }

    private static class ColumnAttributes {
        public String tableName;
        public String columnName;
        public int numberOfDimensions;

        private ColumnAttributes() {
        }
    }
}

