/*
 * Decompiled with CFR 0.152.
 */
package org.evomaster.client.java.sql;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import org.evomaster.client.java.controller.api.dto.database.schema.DatabaseType;
import org.evomaster.client.java.sql.DbUnsupportedException;
import org.evomaster.client.java.sql.h2.H2VersionUtils;
import org.evomaster.client.java.utils.SimpleLogger;

public class DbCleaner {
    public static void clearTables(Connection connection, List<String> fullyQualifyingTableNames, DatabaseType type) {
        String defaultSchema = DbCleaner.getDefaultSchema(type);
        HashMap<String, Set> schemaToNames = new HashMap<String, Set>();
        for (String string : fullyQualifyingTableNames) {
            if (!string.contains(".")) {
                Set x = schemaToNames.computeIfAbsent(defaultSchema, it -> new HashSet());
                x.add(string);
                continue;
            }
            String[] tokens = string.split("\\.");
            String name = tokens[tokens.length - 1];
            String schema = tokens[tokens.length - 2];
            Set x = schemaToNames.computeIfAbsent(schema, it -> new HashSet());
            x.add(name);
        }
        for (Map.Entry entry : schemaToNames.entrySet()) {
            ArrayList<String> names = new ArrayList<String>((Collection)entry.getValue());
            DbCleaner.clearDatabase(connection, (String)entry.getKey(), null, names, type, true);
        }
    }

    public static void clearDatabase_Postgres(Connection connection, String schemaName, List<String> tableToSkip, List<String> tableToClean) {
        DbCleaner.clearDatabase(DbCleaner.getDefaultRetries(DatabaseType.POSTGRES), connection, DbCleaner.getSchemaName(schemaName, DatabaseType.POSTGRES), tableToSkip, tableToClean, DatabaseType.POSTGRES, false, true, false);
    }

    public static void clearDatabase_H2(Connection connection, List<String> tableToSkip, List<String> tableToClean) {
        String h2Version;
        String schemaName = DbCleaner.getDefaultSchema(DatabaseType.H2);
        try {
            h2Version = H2VersionUtils.getH2Version(connection);
        }
        catch (SQLException e) {
            throw new RuntimeException("Unexpected SQLException while fetching H2 version", e);
        }
        boolean restartIdentityWhenTruncating = H2VersionUtils.isVersionGreaterOrEqual(h2Version, "2.0.0");
        DbCleaner.clearDatabase(DbCleaner.getDefaultRetries(DatabaseType.H2), connection, schemaName, tableToSkip, tableToClean, DatabaseType.H2, false, true, restartIdentityWhenTruncating);
    }

    public static void clearDatabase_MySQL(Connection connection, String schemaName, List<String> tablesToSkip, List<String> tableToClean) {
        DbCleaner.clearDatabase(connection, DbCleaner.getSchemaName(schemaName, DatabaseType.MYSQL), tablesToSkip, tableToClean, DatabaseType.MYSQL, true);
    }

    public static void dropDatabaseTables(Connection connection, String schemaName, List<String> tablesToSkip, DatabaseType type) {
        if (type != DatabaseType.MYSQL && type != DatabaseType.MARIADB) {
            throw new IllegalArgumentException("Dropping tables are not supported by " + type);
        }
        DbCleaner.clearDatabase(DbCleaner.getDefaultRetries(type), connection, DbCleaner.getSchemaName(schemaName, type), tablesToSkip, null, type, true, true, false);
    }

    public static void clearDatabase(Connection connection, String schemaName, List<String> tableToSkip, List<String> tableToClean, DatabaseType type, boolean doResetSequence) {
        boolean restartIdentityWhenTruncating;
        if (doResetSequence && type.equals((Object)DatabaseType.H2)) {
            try {
                String h2Version = H2VersionUtils.getH2Version(connection);
                restartIdentityWhenTruncating = H2VersionUtils.isVersionGreaterOrEqual(h2Version, "2.0.0");
            }
            catch (SQLException ex) {
                throw new RuntimeException("Unexpected SQL exception while getting H2 version", ex);
            }
        } else {
            restartIdentityWhenTruncating = false;
        }
        DbCleaner.clearDatabase(DbCleaner.getDefaultRetries(type), connection, DbCleaner.getSchemaName(schemaName, type), tableToSkip, tableToClean, type, false, doResetSequence, restartIdentityWhenTruncating);
    }

    private static String getSchemaName(String schemaName, DatabaseType type) {
        if (schemaName != null) {
            return schemaName;
        }
        return DbCleaner.getDefaultSchema(type);
    }

    private static void clearDatabase(int retries, Connection connection, String schemaName, List<String> tableToSkip, List<String> tableToClean, DatabaseType type, boolean doDropTable, boolean doResetSequence, boolean restartIdentityWhenTruncating) {
        try {
            Statement statement = connection.createStatement();
            DbCleaner.disableReferentialIntegrity(statement, type);
            List<String> cleanedTable = DbCleaner.cleanDataInTables(tableToSkip, tableToClean, statement, type, schemaName, DbCleaner.isSingleCleanCommand(type), doDropTable, restartIdentityWhenTruncating);
            if (doResetSequence) {
                List<String> sequenceToClean = null;
                if (type == DatabaseType.MYSQL || type == DatabaseType.MARIADB) {
                    sequenceToClean = cleanedTable;
                }
                DbCleaner.resetSequences(statement, type, schemaName, sequenceToClean);
            }
            DbCleaner.enableReferentialIntegrity(statement, type);
            statement.close();
        }
        catch (Exception e) {
            String msg = e.getMessage();
            if (msg != null && msg.toLowerCase().contains("timeout")) {
                if (retries > 0) {
                    SimpleLogger.warn((String)"Timeout issue with cleaning DB. Trying again.");
                    try {
                        Thread.sleep(2000L);
                    }
                    catch (InterruptedException interruptedException) {
                        // empty catch block
                    }
                    DbCleaner.clearDatabase(--retries, connection, schemaName, tableToSkip, tableToClean, type, doDropTable, doResetSequence, restartIdentityWhenTruncating);
                } else {
                    SimpleLogger.error((String)"Giving up cleaning the DB. There are still timeouts.");
                }
            }
            throw new RuntimeException(e);
        }
    }

    private static List<String> cleanDataInTables(List<String> tableToSkip, List<String> tableToClean, Statement statement, DatabaseType type, String schema, boolean singleCommand, boolean doDropTable, boolean restartIdentityWhenTruncating) throws SQLException {
        if (tableToSkip != null && !tableToSkip.isEmpty() && tableToClean != null && !tableToClean.isEmpty()) {
            throw new IllegalArgumentException("tableToSkip and tableToClean cannot be configured at the same time.");
        }
        HashSet<String> tables = new HashSet<String>();
        ResultSet rs = statement.executeQuery(DbCleaner.getAllTableCommand(type, schema));
        while (rs.next()) {
            tables.add(rs.getString(1));
        }
        rs.close();
        if (tables.isEmpty()) {
            throw new IllegalStateException("Could not find any table");
        }
        boolean toskip = tableToSkip != null;
        List<String> tableToHandle = tableToClean != null ? tableToClean : tableToSkip;
        if (tableToHandle != null) {
            tableToHandle = tableToHandle.stream().map(t -> {
                String[] tokens = t.split("\\.");
                return tokens[tokens.length - 1];
            }).collect(Collectors.toList());
        }
        if (tableToHandle != null) {
            for (String skip : tableToHandle) {
                if (!tables.stream().noneMatch(t -> t.equalsIgnoreCase(skip))) continue;
                String msg = "Asked to skip/clean table '" + skip + "', but it does not exist.";
                msg = msg + " Existing tables in schema '" + schema + "': [" + String.join((CharSequence)", ", tables.stream().sorted().collect(Collectors.toList())) + "]";
                throw new IllegalStateException(msg);
            }
        }
        HashSet<String> tablesHaveIdentifies = new HashSet<String>();
        if (type == DatabaseType.MS_SQL_SERVER) {
            ResultSet rst = statement.executeQuery(DbCleaner.getAllTableHasIdentify(type, schema));
            while (rst.next()) {
                tablesHaveIdentifies.add(rst.getString(1));
            }
            rst.close();
        }
        List<String> tth = tableToHandle;
        List<String> tablesToClear = tables.stream().filter(n -> tth == null || toskip && (tth.isEmpty() || tth.stream().noneMatch(skip -> skip.equalsIgnoreCase((String)n))) || !toskip && tth.stream().anyMatch(clean -> clean.equalsIgnoreCase((String)n))).collect(Collectors.toList());
        if (singleCommand) {
            String ts = tablesToClear.stream().sorted().collect(Collectors.joining(","));
            if (type != DatabaseType.POSTGRES) {
                throw new IllegalArgumentException("do not support for cleaning all data by one single command for " + type);
            }
            if (doDropTable) {
                DbCleaner.dropTableIfExists(statement, ts);
            } else {
                DbCleaner.truncateTable(statement, ts, restartIdentityWhenTruncating);
            }
        } else {
            for (String t2 : tablesToClear) {
                if (doDropTable) {
                    DbCleaner.dropTableIfExists(statement, t2);
                    continue;
                }
                if (type == DatabaseType.MS_SQL_SERVER) {
                    DbCleaner.deleteTables(statement, t2, schema, tablesHaveIdentifies);
                    continue;
                }
                DbCleaner.truncateTable(statement, t2, restartIdentityWhenTruncating);
            }
        }
        return tablesToClear;
    }

    private static void dropTableIfExists(Statement statement, String table) throws SQLException {
        statement.executeUpdate("DROP TABLE IF EXISTS " + table);
    }

    private static void deleteTables(Statement statement, String table, String schema, Set<String> tableHasIdentify) throws SQLException {
        String tableWithSchema = table;
        if (!schema.isEmpty() && !schema.equals(DbCleaner.getDefaultSchema(DatabaseType.MS_SQL_SERVER))) {
            tableWithSchema = schema + "." + schema;
        }
        statement.executeUpdate("DELETE FROM " + tableWithSchema);
        if (tableHasIdentify.contains(table)) {
            statement.executeUpdate("DBCC CHECKIDENT ('" + tableWithSchema + "', RESEED, 0)");
        }
    }

    private static void truncateTable(Statement statement, String table, boolean restartIdentityWhenTruncating) throws SQLException {
        if (restartIdentityWhenTruncating) {
            statement.executeUpdate("TRUNCATE TABLE " + table + " RESTART IDENTITY");
        } else {
            statement.executeUpdate("TRUNCATE TABLE " + table);
        }
    }

    private static void resetSequences(Statement s, DatabaseType type, String schemaName, List<String> sequenceToClean) throws SQLException {
        HashSet<String> sequences = new HashSet<String>();
        ResultSet rs = s.executeQuery(DbCleaner.getAllSequenceCommand(type, schemaName));
        while (rs.next()) {
            sequences.add(rs.getString(1));
        }
        rs.close();
        for (String seq : sequences) {
            if (sequenceToClean != null && !sequenceToClean.isEmpty() && !sequenceToClean.stream().anyMatch(x -> x.equalsIgnoreCase(seq))) continue;
            s.executeUpdate(DbCleaner.resetSequenceCommand(seq, type));
        }
    }

    private static void disableReferentialIntegrity(Statement s, DatabaseType type) throws SQLException {
        switch (type) {
            case POSTGRES: {
                break;
            }
            case MS_SQL_SERVER: {
                s.execute("EXEC sp_MSForEachTable \"ALTER TABLE ? NOCHECK CONSTRAINT all\"");
                break;
            }
            case H2: {
                s.execute("SET REFERENTIAL_INTEGRITY FALSE");
                break;
            }
            case MARIADB: 
            case MYSQL: {
                s.execute("SET @@foreign_key_checks = 0;");
                break;
            }
            case OTHER: {
                throw new DbUnsupportedException(type);
            }
        }
    }

    private static void enableReferentialIntegrity(Statement s, DatabaseType type) throws SQLException {
        switch (type) {
            case POSTGRES: {
                break;
            }
            case MS_SQL_SERVER: {
                s.execute("exec sp_MSForEachTable \"ALTER TABLE ? WITH CHECK CHECK CONSTRAINT all\"");
                break;
            }
            case H2: {
                s.execute("SET REFERENTIAL_INTEGRITY TRUE");
                break;
            }
            case MARIADB: 
            case MYSQL: {
                s.execute("SET @@foreign_key_checks = 1;");
                break;
            }
            case OTHER: {
                throw new DbUnsupportedException(type);
            }
        }
    }

    private static int getDefaultRetries(DatabaseType type) {
        switch (type) {
            case POSTGRES: 
            case MS_SQL_SERVER: {
                return 0;
            }
            case H2: 
            case MARIADB: 
            case MYSQL: {
                return 3;
            }
        }
        throw new DbUnsupportedException(type);
    }

    private static String getDefaultSchema(DatabaseType type) {
        switch (type) {
            case H2: {
                return "PUBLIC";
            }
            case MS_SQL_SERVER: {
                return "dbo";
            }
            case MARIADB: 
            case MYSQL: {
                return null;
            }
            case POSTGRES: {
                return "public";
            }
        }
        throw new DbUnsupportedException(type);
    }

    private static boolean isSingleCleanCommand(DatabaseType type) {
        return type == DatabaseType.POSTGRES;
    }

    private static String getAllTableHasIdentify(DatabaseType type, String schema) {
        if (type != DatabaseType.MS_SQL_SERVER) {
            throw new IllegalArgumentException("getAllTableHasIdentify only supports for MS_SQL_SERVER, not for " + type);
        }
        return DbCleaner.getAllTableCommand(type, schema) + " AND OBJECTPROPERTY(OBJECT_ID(TABLE_NAME), 'TableHasIdentity') = 1";
    }

    private static String getAllTableCommand(DatabaseType type, String schema) {
        String command = "SELECT TABLE_NAME FROM INFORMATION_SCHEMA.TABLES where (TABLE_TYPE='TABLE' OR TABLE_TYPE='BASE TABLE')";
        switch (type) {
            case POSTGRES: 
            case MS_SQL_SERVER: 
            case H2: 
            case MARIADB: 
            case MYSQL: {
                if (schema == null || schema.isEmpty()) {
                    return command;
                }
                return command + " AND TABLE_SCHEMA='" + schema + "'";
            }
        }
        throw new DbUnsupportedException(type);
    }

    private static String getAllSequenceCommand(DatabaseType type, String schemaName) {
        String command = "SELECT SEQUENCE_NAME FROM INFORMATION_SCHEMA.SEQUENCES";
        switch (type) {
            case MARIADB: 
            case MYSQL: {
                return DbCleaner.getAllTableCommand(type, schemaName);
            }
            case POSTGRES: 
            case MS_SQL_SERVER: 
            case H2: {
                if (schemaName.isEmpty()) {
                    return command;
                }
                return command + " WHERE SEQUENCE_SCHEMA='" + schemaName + "'";
            }
        }
        throw new DbUnsupportedException(type);
    }

    private static String resetSequenceCommand(String sequence, DatabaseType type) {
        switch (type) {
            case MARIADB: 
            case MYSQL: {
                return "ALTER TABLE " + sequence + " AUTO_INCREMENT=1;";
            }
            case POSTGRES: 
            case MS_SQL_SERVER: 
            case H2: {
                return "ALTER SEQUENCE " + sequence + " RESTART WITH 1";
            }
        }
        throw new DbUnsupportedException(type);
    }
}

