/**
 * Licensed to jclouds, Inc. (jclouds) under one or more
 * contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  jclouds licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.jclouds.cloudstack.features;

import java.util.Set;

import javax.ws.rs.Consumes;
import javax.ws.rs.GET;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.MediaType;

import org.jclouds.cloudstack.domain.DiskOffering;
import org.jclouds.cloudstack.domain.NetworkOffering;
import org.jclouds.cloudstack.domain.ServiceOffering;
import org.jclouds.cloudstack.filters.QuerySigner;
import org.jclouds.cloudstack.options.ListDiskOfferingsOptions;
import org.jclouds.cloudstack.options.ListNetworkOfferingsOptions;
import org.jclouds.cloudstack.options.ListServiceOfferingsOptions;
import org.jclouds.rest.annotations.ExceptionParser;
import org.jclouds.rest.annotations.OnlyElement;
import org.jclouds.rest.annotations.QueryParams;
import org.jclouds.rest.annotations.RequestFilters;
import org.jclouds.rest.annotations.SelectJson;
import org.jclouds.rest.functions.ReturnEmptySetOnNotFoundOr404;
import org.jclouds.rest.functions.ReturnNullOnNotFoundOr404;

import com.google.common.util.concurrent.ListenableFuture;

/**
 * Provides asynchronous access to cloudstack via their REST API.
 * <p/>
 * 
 * @see OfferingClient
 * @see <a href="http://download.cloud.com/releases/2.2.0/api_2.2.12/TOC_User.html" />
 * @author Adrian Cole
 */
@RequestFilters(QuerySigner.class)
@QueryParams(keys = "response", values = "json")
public interface OfferingAsyncClient {

   /**
    * @see OfferingClient#listServiceOfferings
    */
   @GET
   @QueryParams(keys = "command", values = "listServiceOfferings")
   @SelectJson("serviceoffering")
   @Consumes(MediaType.APPLICATION_JSON)
   @ExceptionParser(ReturnEmptySetOnNotFoundOr404.class)
   ListenableFuture<Set<ServiceOffering>> listServiceOfferings(ListServiceOfferingsOptions... options);

   /**
    * @see OfferingClient#getServiceOffering
    */
   @GET
   @QueryParams(keys = "command", values = "listServiceOfferings")
   @SelectJson("serviceoffering")
   @OnlyElement
   @Consumes(MediaType.APPLICATION_JSON)
   @ExceptionParser(ReturnNullOnNotFoundOr404.class)
   ListenableFuture<ServiceOffering> getServiceOffering(@QueryParam("id") long id);

   /**
    * @see OfferingClient#listDiskOfferings
    */
   @GET
   @QueryParams(keys = "command", values = "listDiskOfferings")
   @SelectJson("diskoffering")
   @Consumes(MediaType.APPLICATION_JSON)
   @ExceptionParser(ReturnEmptySetOnNotFoundOr404.class)
   ListenableFuture<Set<DiskOffering>> listDiskOfferings(ListDiskOfferingsOptions... options);

   /**
    * @see OfferingClient#getDiskOffering
    */
   @GET
   @QueryParams(keys = "command", values = "listDiskOfferings")
   @SelectJson("diskoffering")
   @OnlyElement
   @Consumes(MediaType.APPLICATION_JSON)
   @ExceptionParser(ReturnNullOnNotFoundOr404.class)
   ListenableFuture<DiskOffering> getDiskOffering(@QueryParam("id") long id);

   /**
    * @see NetworkOfferingClient#listNetworkOfferings
    */
   @GET
   @QueryParams(keys = "command", values = "listNetworkOfferings")
   @SelectJson("networkoffering")
   @Consumes(MediaType.APPLICATION_JSON)
   @ExceptionParser(ReturnEmptySetOnNotFoundOr404.class)
   ListenableFuture<Set<NetworkOffering>> listNetworkOfferings(ListNetworkOfferingsOptions... options);

   /**
    * @see NetworkOfferingClient#getNetworkOffering
    */
   @GET
   @QueryParams(keys = "command", values = "listNetworkOfferings")
   @SelectJson("networkoffering")
   @OnlyElement
   @Consumes(MediaType.APPLICATION_JSON)
   @ExceptionParser(ReturnNullOnNotFoundOr404.class)
   ListenableFuture<NetworkOffering> getNetworkOffering(@QueryParam("id") long id);

}
