package org.sqlproc.engine;

import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Properties;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.sqlproc.engine.impl.SqlMappingRule;
import org.sqlproc.engine.impl.SqlMetaStatement;

/**
 * Helper class for the parsing of META SQL statements and Mapping rules located in properties file.
 * 
 * The purpose of this class is to load and analyze (=parse) META SQL statements and mapping rules located in the
 * properties repository. The standard properties repository is the external properties file with the name
 * <code>queries.properties</code>.
 * 
 * <p>
 * All the properties with the key <code>LIST_..._SQL</code> are parsed as META SQL statements using ANTLR based
 * grammar. <br>
 * All the properties with the key <code>LIST_..._FIELDS</code> are parsed as Mapping rules using ANTLR based grammar.<br>
 * A pair of META SQL statement and Mapping rule forms one named SQL Engine instance.<br>
 * All the properties with the key <code>SET_...</code> are taken as optional features used in the process of SQL Engine
 * instance construction.<br>
 * <p>
 * In the process of ANTLR based parsing different kinds of incorrect stuff can cause SqlEngineException to be raised.
 * 
 * <p>
 * To initialize this class, Spring DI configuration can be utilized, like the next one:<br>
 * 
 * <pre>
 * &lt;beans ...&gt;
 *   ...
 *   &lt;bean id="sqlQueries" class="org.springframework.beans.factory.config.PropertiesFactoryBean"&gt;
 *     &lt;property name="location"&gt;
 *       &lt;value>classpath:queries.properties&lt;/value&gt;
 *     &lt;/property&gt;
 *   &lt;/bean&gt;
 *   &lt;bean id="sqlLoader" class="org.sqlproc.engine.SqlEngineLoader"&gt;
 *     &lt;constructor-arg ref="sqlQueries" /&gt;
 *   &lt;/bean&gt;
 * &lt;/beans&gt;
 * </pre>
 * 
 * and use the next code to obtain instance of SQL Engine
 * 
 * <pre>
 * SqlEngine sqlEngine = sqlLoader.getSqlEngine(&quot;ALL&quot;);
 * </pre>
 * <p>
 * Another possibility is to utilize SqlPropertiesLoader.
 * 
 * <pre>
 * SqlPropertiesLoader loader = new SqlPropertiesLoader(&quot;queries.properties&quot;, this.getClass());
 * SqlEngineLoader sqlLoader = new SqlEngineLoader(loader.getProperties());
 * SqlEngine sqlEngine = sqlLoader.getSqlEngine(&quot;ALL&quot;);
 * </pre>
 * 
 * <p>
 * For more info please see the User's tutorial.
 * 
 * @author <a href="mailto:Vladimir.Hudec@gmail.com">Vladimir Hudec</a>
 */
public class SqlEngineLoader {

    /**
     * The internal slf4j logger.
     */
    protected final Logger logger = LoggerFactory.getLogger(getClass());

    private static final String COMMON_PREFIX = "LIST_";
    private static final String SQL_SUFFIX = "_SQL";
    private static final String FIELDS_SUFFIX = "_FIELDS";
    private static final int lCOMMON_PREFIX = COMMON_PREFIX.length();
    private static final int lSQL_SUFFIX = SQL_SUFFIX.length();
    private static final int lFIELDS_SUFFIX = FIELDS_SUFFIX.length();
    private static final String SET_PREFIX = "SET_";
    private static final int lSET_PREFIX = SET_PREFIX.length();
    private static final String FIELDS_REFERENCE = "#";
    private static final int lFIELDS_REFERENCE = FIELDS_REFERENCE.length();

    /**
     * The collection of named SQL Engine (the primary SQL Processor class) instances.
     */
    private Map<String, SqlEngine> engines = new HashMap<String, SqlEngine>();
    /**
     * The collection of named META SQL statements
     */
    private Map<String, String> sqls = new HashMap<String, String>();
    /**
     * The collection of named Mapping rules
     */
    private Map<String, String> fields = new HashMap<String, String>();
    /**
     * The collection of SQL Processor optional features
     */
    private Map<String, Object> features = new HashMap<String, Object>();

    /**
     * Creates a new instance of SqlEngineLoader from the properties repository (which is in fact a collection of META
     * SQL statements, Mapping rules and Optional features. During the instance construction all the statements are
     * parsed and the collection of named SQL Engine instances are established. Later these instances are used for the
     * SQL queries invocation.
     * 
     * @param props
     *            The collection of META SQL statements, Mapping rules and Optional features
     * @throws SqlEngineException
     *             Mainly in the case the provided statements or rules are not compliant with the ANTLR based grammar.
     */
    public SqlEngineLoader(Properties props) throws SqlEngineException {
        this(props, null, (String[]) null);
    }

    /**
     * Creates a new instance of SqlEngineLoader from the properties repository (which is in fact a collection of META
     * SQL statements, Mapping rules and Optional features. During the instance construction all the statements are
     * parsed and the collection of named SQL Engine instances are established. Later these instances are used for the
     * SQL queries invocation. Every instance of SQL Engined is accompanied with SQL Monitor for the runtime statistics
     * gathering. For the creation of these monitors the SQL Monitor Factory can be used.
     * 
     * @param props
     *            The collection of META SQL statements, Mapping rules and Optional features
     * @param monitorFactory
     *            The monitor factory used in the process of SQL Monitor instances creation.
     * @throws SqlEngineException
     *             Mainly in the case the provided statements or rules are not compliant with the ANTLR based grammar.
     */
    public SqlEngineLoader(Properties props, SqlMonitorFactory monitorFactory) throws SqlEngineException {
        this(props, monitorFactory, (String[]) null);
    }

    /**
     * Creates a new instance of SqlEngineLoader from the properties repository (which is in fact a collection of META
     * SQL statements, Mapping rules and Optional features. During the instance construction all the statements are
     * parsed and the collection of named SQL Engine instances are established. Later these instances are used for the
     * SQL queries invocation. Every instance of SQL Engined is accompanied with SQL Monitor for the runtime statistics
     * gathering. For the creation of these monitors the SQL Monitor Factory can be used.
     * 
     * @param props
     *            The collection of META SQL statements, Mapping rules and Optional features
     * @param monitorFactory
     *            The monitor factory used in the process of SQL Monitor instances creation.
     * @param selectQueries
     *            Only statements and rules with the names in this set are picked up from the properties repository.
     * @throws SqlEngineException
     *             Mainly in the case the provided statements or rules are not compliant with the ANTLR based grammar.
     */
    public SqlEngineLoader(Properties props, SqlMonitorFactory monitorFactory, String... selectQueries)
            throws SqlEngineException {
        if (logger.isDebugEnabled()) {
            logger.debug(">> SqlEngineLoader, props=" + props + ", monitorFactory=" + monitorFactory
                    + ", selectQueries=" + selectQueries);
        }

        try {
            Set<String> setSelectQueries = (selectQueries != null) ? new HashSet<String>(Arrays.asList(selectQueries))
                    : null;

            StringBuilder errors = new StringBuilder();

            for (Entry<Object, Object> entry : props.entrySet()) {
                String key = (String) entry.getKey();
                String value = (String) entry.getValue();
                String name = null;

                if (key.startsWith(COMMON_PREFIX) && key.endsWith(SQL_SUFFIX)) {
                    name = key.substring(lCOMMON_PREFIX, key.length() - lSQL_SUFFIX);
                    if (setSelectQueries == null || setSelectQueries.contains(name)) {
                        if (sqls.containsKey(name))
                            errors.append("Dumplikatni nazev SQL: ").append(key).append("\n");
                        else
                            sqls.put(name, value);
                    }
                } else if (key.startsWith(COMMON_PREFIX) && key.endsWith(FIELDS_SUFFIX)) {
                    name = key.substring(lCOMMON_PREFIX, key.length() - lFIELDS_SUFFIX);
                    if (setSelectQueries == null || setSelectQueries.contains(name)) {
                        if (fields.containsKey(name))
                            errors.append("Dumplikatni nazev FIELDS: ").append(key).append("\n");
                        else
                            fields.put(name, value);
                    }
                } else if (key.startsWith(SET_PREFIX)) {
                    name = key.substring(lSET_PREFIX);
                    if ("true".equalsIgnoreCase(value))
                        features.put(name, Boolean.TRUE);
                    else if ("false".equalsIgnoreCase(value))
                        features.put(name, Boolean.FALSE);
                    else
                        features.put(name, value);
                } else {
                    errors.append("Chybny nazev SQL/FIELDS/SET: ").append(key).append("\n");
                    continue;
                }
            }

            for (String name : sqls.keySet()) {
                if (!fields.containsKey(name))
                    errors.append("K nazvu SQL neexistuje FIELDS: ").append(name).append("\n");
            }
            for (String name : fields.keySet()) {
                if (!sqls.containsKey(name))
                    errors.append("K nazvu FIELDS neexistuje SQLS: ").append(name).append("\n");
            }

            if (errors.length() > 0)
                throw new SqlEngineException(errors.toString());

            for (String name : sqls.keySet()) {
                SqlMetaStatement stmt = null;
                try {
                    stmt = SqlMetaStatement.getInstance(sqls.get(name));
                } catch (SqlEngineException see) {
                    errors.append(see.getMessage());
                }
                SqlMappingRule mapping = null;
                try {
                    String sMapping = fields.get(name).trim();
                    if (sMapping.startsWith(FIELDS_REFERENCE)) {
                        String sRealMapping = props.getProperty(sMapping.substring(lFIELDS_REFERENCE).trim());
                        if (sRealMapping == null)
                            errors.append("K referenci neexistuje FIELDS: ").append(name).append("->").append(sMapping)
                                    .append("\n");
                        else
                            mapping = SqlMappingRule.getInstance(sRealMapping);
                    } else
                        mapping = SqlMappingRule.getInstance(sMapping);
                } catch (SqlEngineException see) {
                    errors.append(see.getMessage());
                }
                SqlMonitor monitor = (monitorFactory != null) ? monitorFactory.getSqlMonitor(name, features) : null;
                if (stmt != null && mapping != null)
                    engines.put(name, new SqlEngine(name, stmt, mapping, monitor, features));
            }

            if (errors.length() > 0)
                throw new SqlEngineException(errors.toString());
        } finally {
            if (logger.isDebugEnabled()) {
                logger.debug("<< SqlEngineLoader, engines=" + engines + ", sqls=" + sqls + ", fields=" + fields
                        + ", features=" + features);
            }
        }
    }

    /**
     * Returns the collection of names of all initialized/constructed SQL Engine instances.
     * 
     * @return Collection of all initialized SQL Engine instances' names
     */
    public Collection<String> getNames() {
        return engines.keySet();
    }

    /**
     * Returns the named META SQL statement.
     * 
     * @param name
     *            the name of the required META SQL statement
     * @return the META SQL statement
     */
    public String getMetaSql(String name) {
        return sqls.get(name);
    }

    /**
     * Returns the named Mapping rule.
     * 
     * @param name
     *            the name of the required Mapping rule
     * @return the Mapping rule
     */
    public String getMappingRule(String name) {
        return fields.get(name);
    }

    /**
     * Returns the named SQL Engine instance (the primary SQL Processor class).
     * 
     * @param name
     *            the name of the required SQL Engine instance
     * @return the SQL Engine instance
     */
    public SqlEngine getSqlEngine(String name) {
        return engines.get(name);
    }

    /**
     * Returns the SQL Monitor instance devoted to the named SQL Engine instance.
     * 
     * @param name
     *            the name of the SQL Engine instance
     * @return the SQL Monitor instance
     */
    public SqlMonitor getSqlMonitor(String name) {
        return getSqlEngine(name).getMonitor();
    }
}
