package org.sqlproc.engine.impl;

import java.util.List;
import java.util.Map;

import org.sqlproc.engine.SqlFeature;
import org.sqlproc.engine.SqlOrder;

/**
 * The crate (design pattern) for all input parameters for
 * {@link org.sqlproc.engine.impl.SqlMetaStatement#process(Object, Object, List, Map)}.
 * 
 * Also can work as a context for a dynamic ANSI SQL Query generation, which the main contract is
 * {@link org.sqlproc.engine.impl.SqlMetaElement#process(SqlProcessContext)}.
 * 
 * @author <a href="mailto:Vladimir.Hudec@gmail.com">Vladimir Hudec</a>
 */
class SqlProcessContext implements SqlFeature {

    /**
     * The SQL statement dynamic parameters.
     */
    Object dynamicInputValues;
    /**
     * The SQL statement static parameters.
     */
    Object staticInputValues;
    /**
     * TThe list of ordering directives.
     */
    List<SqlOrder> order;
    /**
     * The configuration of SQL Processor - optional features.
     */
    Map<String, Object> features;

    /**
     * Creates a new instance with empty static input values.
     * 
     * @param dynamicInputValues
     *            dynamic input values
     * @param order
     *            ordering rules
     * @param features
     *            optional features
     */
    SqlProcessContext(Object dynamicInputValues, List<SqlOrder> order, Map<String, Object> features) {
        this(dynamicInputValues, null, order, features);
    }

    /**
     * Creates a new instance.
     * 
     * @param dynamicInputValues
     *            dynamic input values
     * @param staticInputValues
     *            static input values
     * @param order
     *            ordering rules
     * @param features
     *            optional features
     */
    SqlProcessContext(Object dynamicInputValues, Object staticInputValues, List<SqlOrder> order,
            Map<String, Object> features) {
        this.dynamicInputValues = (dynamicInputValues != null) ? dynamicInputValues : new Object();
        this.staticInputValues = staticInputValues;
        this.order = order;
        this.features = features;
    }

    /**
     * Convenient method to obtain String feature based on the name.
     * 
     * @param name
     *            name of the feature
     * @return value of the feature
     */
    public String getFeature(String name) {
        Object o = features.get(name);
        return (o != null && o instanceof String) ? (String) o : null;
    }

    /**
     * Convenient method to obtain boolean feature based on the name.
     * 
     * @param name
     *            name of the feature
     * @return value of the feature
     */
    public boolean isFeature(String name) {
        Object o = features.get(name);
        return (o != null && o instanceof Boolean && ((Boolean) o)) ? true : false;
    }

    /**
     * Convenient method to obtain Integer feature based on the name.
     * 
     * @param name
     *            name of the feature
     * @return value of the feature
     */
    public Integer getFeatureAsInt(String name) {
        Object o = features.get(name);
        if (o == null || !(o instanceof String))
            return null;
        try {
            return Integer.parseInt((String) o);
        } catch (NumberFormatException nfe) {
            return null;
        }
    }

    /**
     * Convenient method to obtain the index of the ordering rule.
     * 
     * @param orderId
     *            order number
     * @return index of the ordering rule
     */
    public int getOrderIndex(int orderId) {
        if (order == null || order.isEmpty())
            return -1;
        for (int i = 0; i < order.size(); i++) {
            SqlOrder sqlOrder = order.get(i);
            if (sqlOrder.getOrderId() == orderId)
                return i;
        }
        return -1;
    }

    /**
     * Convenient method to obtain the ordering rule based on index.
     * 
     * @param orderIndex
     *            index of the ordering rule
     * @return ordering rule
     */
    public SqlOrder.Order getOrder(int orderIndex) {
        if (orderIndex < 0 || orderIndex >= order.size())
            return SqlOrder.Order.NONE;
        return order.get(orderIndex).getOrderDirrection();
    }
}
