package org.sqlproc.engine;

/**
 * Optional features, which can be activated in the process of SQL Processor initialization. The internal class
 * SqlContext implements this interface.
 * 
 * <p>
 * The features can be incorporated into queries.properties file in the form of <code> SET_...</code> properties.
 * 
 * <p>
 * For example the SQL Processor supports the special searching feature based on text fragments. Lets have a table
 * PERSON with two columns - ID and NAME. <br>
 * In queries.properties there's the next definition:
 * 
 * <pre>
 * SET_LIKE_STRING=like
 * SET_WILDCARD_CHARACTER=%
 * SET_SURROUND_QUERY_LIKE=true
 * SET_SURROUND_QUERY_MIN_LEN=2
 * 
 * LIST_LIKE_SQL= \
 *   select p.ID id, p.NAME name \
 *   from PERSON p \
 *   where 1=1 \
 *   {& id=:id} \
 *   {& UPPER(name) like :+name} \
 *   {#1 order by ID} \
 *   {#2 order by NAME}
 * LIST_LIKE_FIELDS=id name
 * </pre>
 * 
 * <p>
 * The special searching capability is activated with the next statement in queries.properties:
 * 
 * <pre>
 * SET_SURROUND_QUERY_LIKE = true
 * </pre>
 * 
 * In this case every query with the <code>like</code> keyword is identified and dynamic input parameter, which belongs
 * to this query condition, is handled in a special way. The value for this parameter is surrounded with wildcard
 * character <code>%</code>. This character is defined in queries.properties with the key
 * <code>SET_WILDCARD_CHARACTER</code>. In the runtime to activate this feature, the parameter value has to have the
 * minimal length = 2. This minimal length is defined in queries.properties with the key
 * <code>SET_SURROUND_QUERY_MIN_LEN</code>.
 * 
 * In the case of SQL Processor initialization
 * 
 * <pre>
 * SqlPropertiesLoader loader = new SqlPropertiesLoader(&quot;queries.properties&quot;, this.getClass());
 * SqlEngineLoader sqlLoader = new SqlEngineLoader(loader.getProperties());
 * SqlEngine sqlEngine = sqlLoader.getSqlEngine(&quot;LIKE&quot;);
 * </pre>
 * 
 * there's created an instance of SqlEngine with the name <code>LIKE</code>.
 * 
 * <p>
 * Next there's an instance person of the class Person with the value <code>an</code> for the attribute name. The
 * invocation
 * 
 * <pre>
 * List&lt;Person&gt; list = sqlEngine.query(session, Person.class, person, SqlOrder.getDescOrder(2));
 * </pre>
 * 
 * produces the next SQL execution
 * 
 * <pre>
 * select p.ID id, p.NAME name from PERSON p where 1=1  AND  UPPER(name) like ?  order by NAME DESC
 * </pre>
 * 
 * In the result list there are all table rows with name, which contains the text fragment <code>an</code>.
 * 
 * <p>
 * For more info please see the User's tutorial.
 * 
 * @author <a href="mailto:Vladimir.Hudec@gmail.com">Vladimir Hudec</a>
 */
public interface SqlFeature {

    /**
     * <code>SET_WILDCARD_CHARACTER</code> is the key for the wildcard character for SQL <code>like</code> command in
     * the properties.
     */
    public static final String WILDCARD_CHARACTER = "WILDCARD_CHARACTER";
    /**
     * This is the default value for the key <code>SET_WILDCARD_CHARACTER</code> .
     */
    public static final String DEFAULT_WILDCARD_CHARACTER = "%";
    /**
     * <code>SET_SURROUND_QUERY_LIKE</code> is the key for the special SQL Processor behavior. In the case the value of
     * this property is <code>true</code>, the SQL Processor puts wildcard character as a prefix and postfix for all
     * string values in the SQL select commands <code>like</code>. These string values should have to have the minimal
     * length greater or equal to <code>SET_SURROUND_QUERY_MIN_LEN</code>.
     */
    public static final String SURROUND_QUERY_LIKE = "SURROUND_QUERY_LIKE";
    /**
     * <code>SET_SURROUND_QUERY_MIN_LEN</code> is the minimal length of the string input values for the SQL
     * <code>like</code> command to switch on the SQL Processor special behavior described above in the runtime.
     */
    public static final String SURROUND_QUERY_MIN_LEN = "SURROUND_QUERY_MIN_LEN";
    /**
     * This is the default value for the key <code>SET_SURROUND_QUERY_MIN_LEN</code>.
     */
    public static final Integer DEFAULT_SURROUND_QUERY_MIN_LEN = 2;
    /**
     * <code>SET_LIKE_STRING</code> is the key for the <code>like</code> sql command. For example for the Informix
     * database it can be <code>matches</code>.
     */
    public static final String LIKE_STRING = "LIKE_STRING";
    /**
     * This is the default value for the key <code>SET_LIKE_STRING</code>.
     */
    public static final String DEFAULT_LIKE_STRING = "like";
    /**
     * <code>SET_METHODS_ENUM_IN</code> lists the methods used in the translation from a Java type to a JDBC datatype
     * for enumerations.
     */
    public static final String METHODS_ENUM_IN = "METHODS_ENUM_IN";
    /**
     * This is the default value for the key <code>SET_METHODS_ENUM_IN</code>.
     */
    public static final String[] DEFAULT_METHODS_ENUM_IN = new String[] { "getCode", "getValue", "getName", "name" };
    /**
     * <code>SET_METHODS_ENUM_OUT</code> lists the methods used in the translation from a JDBC datatype to a Java type
     * for enumerations.
     */
    public static final String METHODS_ENUM_OUT = "METHODS_ENUM_OUT";
    /**
     * This is the default value for the key <code>SET_METHODS_ENUM_OUT</code>.
     */
    public static final String[] DEFAULT_METHODS_ENUM_OUT = new String[] { "fromCode", "fromValue", "valueOf" };
}
