package org.sqlproc.engine.impl.type;

import java.lang.reflect.Method;

import org.hibernate.Hibernate;
import org.hibernate.Query;
import org.hibernate.type.Type;
import org.joda.time.DateTime;
import org.joda.time.LocalDate;
import org.sqlproc.engine.SqlRuntimeException;
import org.sqlproc.engine.impl.BeanUtils;

/**
 * The SQL META Types for DATE.
 * 
 * @author <a href="mailto:Vladimir.Hudec@gmail.com">Vladimir Hudec</a>
 */
public class SqlDateType extends SqlGenericType {

    /**
     * {@inheritDoc}
     */
    @Override
    public Type getHibernateType() {
        return Hibernate.DATE;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Class<?>[] getClassTypes() {
        return new Class[] {};
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String[] getMetaTypes() {
        return new String[] { "DATE" };
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void setResult(Object resultInstance, String attributeName, Object resultValue, boolean ingoreError)
            throws SqlRuntimeException {
        if (logger.isTraceEnabled()) {
            logger.trace(">>> setResult " + getMetaTypes()[0] + ": resultInstance=" + resultInstance
                    + ", attributeName=" + attributeName + ", resultValue=" + resultValue);
        }
        Method m = BeanUtils.getSetter(resultInstance, attributeName, java.util.Date.class);
        if (m != null) {
            BeanUtils.simpleInvokeMethod(m, resultInstance, resultValue);
        } else {
            m = BeanUtils.getSetter(resultInstance, attributeName, LocalDate.class);
            if (m != null) {
                LocalDate dt = new LocalDate(resultValue);
                BeanUtils.simpleInvokeMethod(m, resultInstance, dt);
            } else if (ingoreError) {
                logger.error("Incorrect datetime " + resultValue + " for " + attributeName + " in " + resultInstance);
            } else {
                throw new SqlRuntimeException("Incorrect datetime " + resultValue + " for " + attributeName + " in "
                        + resultInstance);
            }
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void setParameter(Query query, String paramName, Object inputValue, Class<?> inputType, boolean ingoreError)
            throws SqlRuntimeException {
        if (logger.isTraceEnabled()) {
            logger.trace(">>> setParameter " + getMetaTypes()[0] + ": paramName=" + paramName + ", inputValue="
                    + inputValue + ", inputType=" + inputType);
        }
        if (inputValue == null) {
            query.setParameter(paramName, inputValue, Hibernate.DATE);
        } else {
            DateTime dt = null;
            if (inputValue instanceof LocalDate)
                dt = ((LocalDate) inputValue).toDateTimeAtStartOfDay();
            else
                dt = new DateTime(inputValue).withTime(0, 0, 0, 0);
            query.setTimestamp(paramName, dt.toDate());
        }
    }
}
