/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codedeploy.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.runtime.TypeConverter;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about a configuration for automatically rolling back to a previous version of an application revision
 * when a deployment doesn't complete successfully.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AutoRollbackConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<AutoRollbackConfiguration.Builder, AutoRollbackConfiguration> {
    private static final SdkField<Boolean> ENABLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(AutoRollbackConfiguration::enabled)).setter(setter(Builder::enabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("enabled").build()).build();

    private static final SdkField<List<String>> EVENTS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(AutoRollbackConfiguration::eventsAsStrings))
            .setter(setter(Builder::eventsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("events").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ENABLED_FIELD, EVENTS_FIELD));

    private static final long serialVersionUID = 1L;

    private final Boolean enabled;

    private final List<String> events;

    private AutoRollbackConfiguration(BuilderImpl builder) {
        this.enabled = builder.enabled;
        this.events = builder.events;
    }

    /**
     * <p>
     * Indicates whether a defined automatic rollback configuration is currently enabled.
     * </p>
     * 
     * @return Indicates whether a defined automatic rollback configuration is currently enabled.
     */
    public Boolean enabled() {
        return enabled;
    }

    /**
     * <p>
     * The event type or types that trigger a rollback.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The event type or types that trigger a rollback.
     */
    public List<AutoRollbackEvent> events() {
        return TypeConverter.convert(events, AutoRollbackEvent::fromValue);
    }

    /**
     * <p>
     * The event type or types that trigger a rollback.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The event type or types that trigger a rollback.
     */
    public List<String> eventsAsStrings() {
        return events;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(enabled());
        hashCode = 31 * hashCode + Objects.hashCode(eventsAsStrings());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AutoRollbackConfiguration)) {
            return false;
        }
        AutoRollbackConfiguration other = (AutoRollbackConfiguration) obj;
        return Objects.equals(enabled(), other.enabled()) && Objects.equals(eventsAsStrings(), other.eventsAsStrings());
    }

    @Override
    public String toString() {
        return ToString.builder("AutoRollbackConfiguration").add("Enabled", enabled()).add("Events", eventsAsStrings()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "enabled":
            return Optional.ofNullable(clazz.cast(enabled()));
        case "events":
            return Optional.ofNullable(clazz.cast(eventsAsStrings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<AutoRollbackConfiguration, T> g) {
        return obj -> g.apply((AutoRollbackConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AutoRollbackConfiguration> {
        /**
         * <p>
         * Indicates whether a defined automatic rollback configuration is currently enabled.
         * </p>
         * 
         * @param enabled
         *        Indicates whether a defined automatic rollback configuration is currently enabled.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enabled(Boolean enabled);

        /**
         * <p>
         * The event type or types that trigger a rollback.
         * </p>
         * 
         * @param events
         *        The event type or types that trigger a rollback.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventsWithStrings(Collection<String> events);

        /**
         * <p>
         * The event type or types that trigger a rollback.
         * </p>
         * 
         * @param events
         *        The event type or types that trigger a rollback.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventsWithStrings(String... events);

        /**
         * <p>
         * The event type or types that trigger a rollback.
         * </p>
         * 
         * @param events
         *        The event type or types that trigger a rollback.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder events(Collection<AutoRollbackEvent> events);

        /**
         * <p>
         * The event type or types that trigger a rollback.
         * </p>
         * 
         * @param events
         *        The event type or types that trigger a rollback.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder events(AutoRollbackEvent... events);
    }

    static final class BuilderImpl implements Builder {
        private Boolean enabled;

        private List<String> events = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(AutoRollbackConfiguration model) {
            enabled(model.enabled);
            eventsWithStrings(model.events);
        }

        public final Boolean getEnabled() {
            return enabled;
        }

        @Override
        public final Builder enabled(Boolean enabled) {
            this.enabled = enabled;
            return this;
        }

        public final void setEnabled(Boolean enabled) {
            this.enabled = enabled;
        }

        public final Collection<String> getEvents() {
            return events;
        }

        @Override
        public final Builder eventsWithStrings(Collection<String> events) {
            this.events = AutoRollbackEventsListCopier.copy(events);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder eventsWithStrings(String... events) {
            eventsWithStrings(Arrays.asList(events));
            return this;
        }

        @Override
        public final Builder events(Collection<AutoRollbackEvent> events) {
            this.events = AutoRollbackEventsListCopier.copyEnumToString(events);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder events(AutoRollbackEvent... events) {
            events(Arrays.asList(events));
            return this;
        }

        public final void setEvents(Collection<String> events) {
            this.events = AutoRollbackEventsListCopier.copy(events);
        }

        @Override
        public AutoRollbackConfiguration build() {
            return new AutoRollbackConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
