/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codedeploy.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the input of a BatchGetApplicationRevisions operation.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class BatchGetApplicationRevisionsRequest extends CodeDeployRequest implements
        ToCopyableBuilder<BatchGetApplicationRevisionsRequest.Builder, BatchGetApplicationRevisionsRequest> {
    private static final SdkField<String> APPLICATION_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(BatchGetApplicationRevisionsRequest::applicationName)).setter(setter(Builder::applicationName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("applicationName").build()).build();

    private static final SdkField<List<RevisionLocation>> REVISIONS_FIELD = SdkField
            .<List<RevisionLocation>> builder(MarshallingType.LIST)
            .getter(getter(BatchGetApplicationRevisionsRequest::revisions))
            .setter(setter(Builder::revisions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("revisions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<RevisionLocation> builder(MarshallingType.SDK_POJO)
                                            .constructor(RevisionLocation::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(APPLICATION_NAME_FIELD,
            REVISIONS_FIELD));

    private final String applicationName;

    private final List<RevisionLocation> revisions;

    private BatchGetApplicationRevisionsRequest(BuilderImpl builder) {
        super(builder);
        this.applicationName = builder.applicationName;
        this.revisions = builder.revisions;
    }

    /**
     * <p>
     * The name of an AWS CodeDeploy application about which to get revision information.
     * </p>
     * 
     * @return The name of an AWS CodeDeploy application about which to get revision information.
     */
    public String applicationName() {
        return applicationName;
    }

    /**
     * Returns true if the Revisions property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasRevisions() {
        return revisions != null && !(revisions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of <code>RevisionLocation</code> objects that specify information to get about the application
     * revisions, including type and location. The maximum number of <code>RevisionLocation</code> objects you can
     * specify is 25.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasRevisions()} to see if a value was sent in this field.
     * </p>
     * 
     * @return An array of <code>RevisionLocation</code> objects that specify information to get about the application
     *         revisions, including type and location. The maximum number of <code>RevisionLocation</code> objects you
     *         can specify is 25.
     */
    public List<RevisionLocation> revisions() {
        return revisions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(applicationName());
        hashCode = 31 * hashCode + Objects.hashCode(revisions());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof BatchGetApplicationRevisionsRequest)) {
            return false;
        }
        BatchGetApplicationRevisionsRequest other = (BatchGetApplicationRevisionsRequest) obj;
        return Objects.equals(applicationName(), other.applicationName()) && Objects.equals(revisions(), other.revisions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("BatchGetApplicationRevisionsRequest").add("ApplicationName", applicationName())
                .add("Revisions", revisions()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "applicationName":
            return Optional.ofNullable(clazz.cast(applicationName()));
        case "revisions":
            return Optional.ofNullable(clazz.cast(revisions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<BatchGetApplicationRevisionsRequest, T> g) {
        return obj -> g.apply((BatchGetApplicationRevisionsRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends CodeDeployRequest.Builder, SdkPojo,
            CopyableBuilder<Builder, BatchGetApplicationRevisionsRequest> {
        /**
         * <p>
         * The name of an AWS CodeDeploy application about which to get revision information.
         * </p>
         * 
         * @param applicationName
         *        The name of an AWS CodeDeploy application about which to get revision information.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder applicationName(String applicationName);

        /**
         * <p>
         * An array of <code>RevisionLocation</code> objects that specify information to get about the application
         * revisions, including type and location. The maximum number of <code>RevisionLocation</code> objects you can
         * specify is 25.
         * </p>
         * 
         * @param revisions
         *        An array of <code>RevisionLocation</code> objects that specify information to get about the
         *        application revisions, including type and location. The maximum number of
         *        <code>RevisionLocation</code> objects you can specify is 25.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder revisions(Collection<RevisionLocation> revisions);

        /**
         * <p>
         * An array of <code>RevisionLocation</code> objects that specify information to get about the application
         * revisions, including type and location. The maximum number of <code>RevisionLocation</code> objects you can
         * specify is 25.
         * </p>
         * 
         * @param revisions
         *        An array of <code>RevisionLocation</code> objects that specify information to get about the
         *        application revisions, including type and location. The maximum number of
         *        <code>RevisionLocation</code> objects you can specify is 25.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder revisions(RevisionLocation... revisions);

        /**
         * <p>
         * An array of <code>RevisionLocation</code> objects that specify information to get about the application
         * revisions, including type and location. The maximum number of <code>RevisionLocation</code> objects you can
         * specify is 25.
         * </p>
         * This is a convenience that creates an instance of the {@link List<RevisionLocation>.Builder} avoiding the
         * need to create one manually via {@link List<RevisionLocation>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<RevisionLocation>.Builder#build()} is called immediately and
         * its result is passed to {@link #revisions(List<RevisionLocation>)}.
         * 
         * @param revisions
         *        a consumer that will call methods on {@link List<RevisionLocation>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #revisions(List<RevisionLocation>)
         */
        Builder revisions(Consumer<RevisionLocation.Builder>... revisions);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends CodeDeployRequest.BuilderImpl implements Builder {
        private String applicationName;

        private List<RevisionLocation> revisions = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(BatchGetApplicationRevisionsRequest model) {
            super(model);
            applicationName(model.applicationName);
            revisions(model.revisions);
        }

        public final String getApplicationName() {
            return applicationName;
        }

        @Override
        public final Builder applicationName(String applicationName) {
            this.applicationName = applicationName;
            return this;
        }

        public final void setApplicationName(String applicationName) {
            this.applicationName = applicationName;
        }

        public final Collection<RevisionLocation.Builder> getRevisions() {
            return revisions != null ? revisions.stream().map(RevisionLocation::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder revisions(Collection<RevisionLocation> revisions) {
            this.revisions = RevisionLocationListCopier.copy(revisions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder revisions(RevisionLocation... revisions) {
            revisions(Arrays.asList(revisions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder revisions(Consumer<RevisionLocation.Builder>... revisions) {
            revisions(Stream.of(revisions).map(c -> RevisionLocation.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setRevisions(Collection<RevisionLocation.BuilderImpl> revisions) {
            this.revisions = RevisionLocationListCopier.copyFromBuilder(revisions);
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public BatchGetApplicationRevisionsRequest build() {
            return new BatchGetApplicationRevisionsRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
