/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codedeploy.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about the target Lambda function during an Lambda deployment.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class LambdaTarget implements SdkPojo, Serializable, ToCopyableBuilder<LambdaTarget.Builder, LambdaTarget> {
    private static final SdkField<String> DEPLOYMENT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("deploymentId").getter(getter(LambdaTarget::deploymentId)).setter(setter(Builder::deploymentId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("deploymentId").build()).build();

    private static final SdkField<String> TARGET_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("targetId").getter(getter(LambdaTarget::targetId)).setter(setter(Builder::targetId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("targetId").build()).build();

    private static final SdkField<String> TARGET_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("targetArn").getter(getter(LambdaTarget::targetArn)).setter(setter(Builder::targetArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("targetArn").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(LambdaTarget::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<Instant> LAST_UPDATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("lastUpdatedAt").getter(getter(LambdaTarget::lastUpdatedAt)).setter(setter(Builder::lastUpdatedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastUpdatedAt").build()).build();

    private static final SdkField<List<LifecycleEvent>> LIFECYCLE_EVENTS_FIELD = SdkField
            .<List<LifecycleEvent>> builder(MarshallingType.LIST)
            .memberName("lifecycleEvents")
            .getter(getter(LambdaTarget::lifecycleEvents))
            .setter(setter(Builder::lifecycleEvents))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lifecycleEvents").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<LifecycleEvent> builder(MarshallingType.SDK_POJO)
                                            .constructor(LifecycleEvent::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<LambdaFunctionInfo> LAMBDA_FUNCTION_INFO_FIELD = SdkField
            .<LambdaFunctionInfo> builder(MarshallingType.SDK_POJO).memberName("lambdaFunctionInfo")
            .getter(getter(LambdaTarget::lambdaFunctionInfo)).setter(setter(Builder::lambdaFunctionInfo))
            .constructor(LambdaFunctionInfo::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lambdaFunctionInfo").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DEPLOYMENT_ID_FIELD,
            TARGET_ID_FIELD, TARGET_ARN_FIELD, STATUS_FIELD, LAST_UPDATED_AT_FIELD, LIFECYCLE_EVENTS_FIELD,
            LAMBDA_FUNCTION_INFO_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("deploymentId", DEPLOYMENT_ID_FIELD);
                    put("targetId", TARGET_ID_FIELD);
                    put("targetArn", TARGET_ARN_FIELD);
                    put("status", STATUS_FIELD);
                    put("lastUpdatedAt", LAST_UPDATED_AT_FIELD);
                    put("lifecycleEvents", LIFECYCLE_EVENTS_FIELD);
                    put("lambdaFunctionInfo", LAMBDA_FUNCTION_INFO_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final String deploymentId;

    private final String targetId;

    private final String targetArn;

    private final String status;

    private final Instant lastUpdatedAt;

    private final List<LifecycleEvent> lifecycleEvents;

    private final LambdaFunctionInfo lambdaFunctionInfo;

    private LambdaTarget(BuilderImpl builder) {
        this.deploymentId = builder.deploymentId;
        this.targetId = builder.targetId;
        this.targetArn = builder.targetArn;
        this.status = builder.status;
        this.lastUpdatedAt = builder.lastUpdatedAt;
        this.lifecycleEvents = builder.lifecycleEvents;
        this.lambdaFunctionInfo = builder.lambdaFunctionInfo;
    }

    /**
     * <p>
     * The unique ID of a deployment.
     * </p>
     * 
     * @return The unique ID of a deployment.
     */
    public final String deploymentId() {
        return deploymentId;
    }

    /**
     * <p>
     * The unique ID of a deployment target that has a type of <code>lambdaTarget</code>.
     * </p>
     * 
     * @return The unique ID of a deployment target that has a type of <code>lambdaTarget</code>.
     */
    public final String targetId() {
        return targetId;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the target.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the target.
     */
    public final String targetArn() {
        return targetArn;
    }

    /**
     * <p>
     * The status an Lambda deployment's target Lambda function.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link TargetStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status an Lambda deployment's target Lambda function.
     * @see TargetStatus
     */
    public final TargetStatus status() {
        return TargetStatus.fromValue(status);
    }

    /**
     * <p>
     * The status an Lambda deployment's target Lambda function.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link TargetStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status an Lambda deployment's target Lambda function.
     * @see TargetStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The date and time when the target Lambda function was updated by a deployment.
     * </p>
     * 
     * @return The date and time when the target Lambda function was updated by a deployment.
     */
    public final Instant lastUpdatedAt() {
        return lastUpdatedAt;
    }

    /**
     * For responses, this returns true if the service returned a value for the LifecycleEvents property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasLifecycleEvents() {
        return lifecycleEvents != null && !(lifecycleEvents instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The lifecycle events of the deployment to this target Lambda function.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasLifecycleEvents} method.
     * </p>
     * 
     * @return The lifecycle events of the deployment to this target Lambda function.
     */
    public final List<LifecycleEvent> lifecycleEvents() {
        return lifecycleEvents;
    }

    /**
     * <p>
     * A <code>LambdaFunctionInfo</code> object that describes a target Lambda function.
     * </p>
     * 
     * @return A <code>LambdaFunctionInfo</code> object that describes a target Lambda function.
     */
    public final LambdaFunctionInfo lambdaFunctionInfo() {
        return lambdaFunctionInfo;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(deploymentId());
        hashCode = 31 * hashCode + Objects.hashCode(targetId());
        hashCode = 31 * hashCode + Objects.hashCode(targetArn());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdatedAt());
        hashCode = 31 * hashCode + Objects.hashCode(hasLifecycleEvents() ? lifecycleEvents() : null);
        hashCode = 31 * hashCode + Objects.hashCode(lambdaFunctionInfo());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof LambdaTarget)) {
            return false;
        }
        LambdaTarget other = (LambdaTarget) obj;
        return Objects.equals(deploymentId(), other.deploymentId()) && Objects.equals(targetId(), other.targetId())
                && Objects.equals(targetArn(), other.targetArn()) && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(lastUpdatedAt(), other.lastUpdatedAt()) && hasLifecycleEvents() == other.hasLifecycleEvents()
                && Objects.equals(lifecycleEvents(), other.lifecycleEvents())
                && Objects.equals(lambdaFunctionInfo(), other.lambdaFunctionInfo());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("LambdaTarget").add("DeploymentId", deploymentId()).add("TargetId", targetId())
                .add("TargetArn", targetArn()).add("Status", statusAsString()).add("LastUpdatedAt", lastUpdatedAt())
                .add("LifecycleEvents", hasLifecycleEvents() ? lifecycleEvents() : null)
                .add("LambdaFunctionInfo", lambdaFunctionInfo()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "deploymentId":
            return Optional.ofNullable(clazz.cast(deploymentId()));
        case "targetId":
            return Optional.ofNullable(clazz.cast(targetId()));
        case "targetArn":
            return Optional.ofNullable(clazz.cast(targetArn()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "lastUpdatedAt":
            return Optional.ofNullable(clazz.cast(lastUpdatedAt()));
        case "lifecycleEvents":
            return Optional.ofNullable(clazz.cast(lifecycleEvents()));
        case "lambdaFunctionInfo":
            return Optional.ofNullable(clazz.cast(lambdaFunctionInfo()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<LambdaTarget, T> g) {
        return obj -> g.apply((LambdaTarget) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, LambdaTarget> {
        /**
         * <p>
         * The unique ID of a deployment.
         * </p>
         * 
         * @param deploymentId
         *        The unique ID of a deployment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deploymentId(String deploymentId);

        /**
         * <p>
         * The unique ID of a deployment target that has a type of <code>lambdaTarget</code>.
         * </p>
         * 
         * @param targetId
         *        The unique ID of a deployment target that has a type of <code>lambdaTarget</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetId(String targetId);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the target.
         * </p>
         * 
         * @param targetArn
         *        The Amazon Resource Name (ARN) of the target.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetArn(String targetArn);

        /**
         * <p>
         * The status an Lambda deployment's target Lambda function.
         * </p>
         * 
         * @param status
         *        The status an Lambda deployment's target Lambda function.
         * @see TargetStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TargetStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status an Lambda deployment's target Lambda function.
         * </p>
         * 
         * @param status
         *        The status an Lambda deployment's target Lambda function.
         * @see TargetStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TargetStatus
         */
        Builder status(TargetStatus status);

        /**
         * <p>
         * The date and time when the target Lambda function was updated by a deployment.
         * </p>
         * 
         * @param lastUpdatedAt
         *        The date and time when the target Lambda function was updated by a deployment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdatedAt(Instant lastUpdatedAt);

        /**
         * <p>
         * The lifecycle events of the deployment to this target Lambda function.
         * </p>
         * 
         * @param lifecycleEvents
         *        The lifecycle events of the deployment to this target Lambda function.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lifecycleEvents(Collection<LifecycleEvent> lifecycleEvents);

        /**
         * <p>
         * The lifecycle events of the deployment to this target Lambda function.
         * </p>
         * 
         * @param lifecycleEvents
         *        The lifecycle events of the deployment to this target Lambda function.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lifecycleEvents(LifecycleEvent... lifecycleEvents);

        /**
         * <p>
         * The lifecycle events of the deployment to this target Lambda function.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.codedeploy.model.LifecycleEvent.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.codedeploy.model.LifecycleEvent#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.codedeploy.model.LifecycleEvent.Builder#build()} is called immediately
         * and its result is passed to {@link #lifecycleEvents(List<LifecycleEvent>)}.
         * 
         * @param lifecycleEvents
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.codedeploy.model.LifecycleEvent.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #lifecycleEvents(java.util.Collection<LifecycleEvent>)
         */
        Builder lifecycleEvents(Consumer<LifecycleEvent.Builder>... lifecycleEvents);

        /**
         * <p>
         * A <code>LambdaFunctionInfo</code> object that describes a target Lambda function.
         * </p>
         * 
         * @param lambdaFunctionInfo
         *        A <code>LambdaFunctionInfo</code> object that describes a target Lambda function.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lambdaFunctionInfo(LambdaFunctionInfo lambdaFunctionInfo);

        /**
         * <p>
         * A <code>LambdaFunctionInfo</code> object that describes a target Lambda function.
         * </p>
         * This is a convenience method that creates an instance of the {@link LambdaFunctionInfo.Builder} avoiding the
         * need to create one manually via {@link LambdaFunctionInfo#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link LambdaFunctionInfo.Builder#build()} is called immediately and its
         * result is passed to {@link #lambdaFunctionInfo(LambdaFunctionInfo)}.
         * 
         * @param lambdaFunctionInfo
         *        a consumer that will call methods on {@link LambdaFunctionInfo.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #lambdaFunctionInfo(LambdaFunctionInfo)
         */
        default Builder lambdaFunctionInfo(Consumer<LambdaFunctionInfo.Builder> lambdaFunctionInfo) {
            return lambdaFunctionInfo(LambdaFunctionInfo.builder().applyMutation(lambdaFunctionInfo).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String deploymentId;

        private String targetId;

        private String targetArn;

        private String status;

        private Instant lastUpdatedAt;

        private List<LifecycleEvent> lifecycleEvents = DefaultSdkAutoConstructList.getInstance();

        private LambdaFunctionInfo lambdaFunctionInfo;

        private BuilderImpl() {
        }

        private BuilderImpl(LambdaTarget model) {
            deploymentId(model.deploymentId);
            targetId(model.targetId);
            targetArn(model.targetArn);
            status(model.status);
            lastUpdatedAt(model.lastUpdatedAt);
            lifecycleEvents(model.lifecycleEvents);
            lambdaFunctionInfo(model.lambdaFunctionInfo);
        }

        public final String getDeploymentId() {
            return deploymentId;
        }

        public final void setDeploymentId(String deploymentId) {
            this.deploymentId = deploymentId;
        }

        @Override
        public final Builder deploymentId(String deploymentId) {
            this.deploymentId = deploymentId;
            return this;
        }

        public final String getTargetId() {
            return targetId;
        }

        public final void setTargetId(String targetId) {
            this.targetId = targetId;
        }

        @Override
        public final Builder targetId(String targetId) {
            this.targetId = targetId;
            return this;
        }

        public final String getTargetArn() {
            return targetArn;
        }

        public final void setTargetArn(String targetArn) {
            this.targetArn = targetArn;
        }

        @Override
        public final Builder targetArn(String targetArn) {
            this.targetArn = targetArn;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(TargetStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final Instant getLastUpdatedAt() {
            return lastUpdatedAt;
        }

        public final void setLastUpdatedAt(Instant lastUpdatedAt) {
            this.lastUpdatedAt = lastUpdatedAt;
        }

        @Override
        public final Builder lastUpdatedAt(Instant lastUpdatedAt) {
            this.lastUpdatedAt = lastUpdatedAt;
            return this;
        }

        public final List<LifecycleEvent.Builder> getLifecycleEvents() {
            List<LifecycleEvent.Builder> result = LifecycleEventListCopier.copyToBuilder(this.lifecycleEvents);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setLifecycleEvents(Collection<LifecycleEvent.BuilderImpl> lifecycleEvents) {
            this.lifecycleEvents = LifecycleEventListCopier.copyFromBuilder(lifecycleEvents);
        }

        @Override
        public final Builder lifecycleEvents(Collection<LifecycleEvent> lifecycleEvents) {
            this.lifecycleEvents = LifecycleEventListCopier.copy(lifecycleEvents);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder lifecycleEvents(LifecycleEvent... lifecycleEvents) {
            lifecycleEvents(Arrays.asList(lifecycleEvents));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder lifecycleEvents(Consumer<LifecycleEvent.Builder>... lifecycleEvents) {
            lifecycleEvents(Stream.of(lifecycleEvents).map(c -> LifecycleEvent.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final LambdaFunctionInfo.Builder getLambdaFunctionInfo() {
            return lambdaFunctionInfo != null ? lambdaFunctionInfo.toBuilder() : null;
        }

        public final void setLambdaFunctionInfo(LambdaFunctionInfo.BuilderImpl lambdaFunctionInfo) {
            this.lambdaFunctionInfo = lambdaFunctionInfo != null ? lambdaFunctionInfo.build() : null;
        }

        @Override
        public final Builder lambdaFunctionInfo(LambdaFunctionInfo lambdaFunctionInfo) {
            this.lambdaFunctionInfo = lambdaFunctionInfo;
            return this;
        }

        @Override
        public LambdaTarget build() {
            return new LambdaTarget(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
