/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codedeploy.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about an application revision.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class GenericRevisionInfo implements SdkPojo, Serializable,
        ToCopyableBuilder<GenericRevisionInfo.Builder, GenericRevisionInfo> {
    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GenericRevisionInfo::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("description").build()).build();

    private static final SdkField<List<String>> DEPLOYMENT_GROUPS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(GenericRevisionInfo::deploymentGroups))
            .setter(setter(Builder::deploymentGroups))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("deploymentGroups").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Instant> FIRST_USED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(GenericRevisionInfo::firstUsedTime)).setter(setter(Builder::firstUsedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("firstUsedTime").build()).build();

    private static final SdkField<Instant> LAST_USED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(GenericRevisionInfo::lastUsedTime)).setter(setter(Builder::lastUsedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastUsedTime").build()).build();

    private static final SdkField<Instant> REGISTER_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(GenericRevisionInfo::registerTime)).setter(setter(Builder::registerTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("registerTime").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DESCRIPTION_FIELD,
            DEPLOYMENT_GROUPS_FIELD, FIRST_USED_TIME_FIELD, LAST_USED_TIME_FIELD, REGISTER_TIME_FIELD));

    private static final long serialVersionUID = 1L;

    private final String description;

    private final List<String> deploymentGroups;

    private final Instant firstUsedTime;

    private final Instant lastUsedTime;

    private final Instant registerTime;

    private GenericRevisionInfo(BuilderImpl builder) {
        this.description = builder.description;
        this.deploymentGroups = builder.deploymentGroups;
        this.firstUsedTime = builder.firstUsedTime;
        this.lastUsedTime = builder.lastUsedTime;
        this.registerTime = builder.registerTime;
    }

    /**
     * <p>
     * A comment about the revision.
     * </p>
     * 
     * @return A comment about the revision.
     */
    public String description() {
        return description;
    }

    /**
     * <p>
     * The deployment groups for which this is the current target revision.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The deployment groups for which this is the current target revision.
     */
    public List<String> deploymentGroups() {
        return deploymentGroups;
    }

    /**
     * <p>
     * When the revision was first used by AWS CodeDeploy.
     * </p>
     * 
     * @return When the revision was first used by AWS CodeDeploy.
     */
    public Instant firstUsedTime() {
        return firstUsedTime;
    }

    /**
     * <p>
     * When the revision was last used by AWS CodeDeploy.
     * </p>
     * 
     * @return When the revision was last used by AWS CodeDeploy.
     */
    public Instant lastUsedTime() {
        return lastUsedTime;
    }

    /**
     * <p>
     * When the revision was registered with AWS CodeDeploy.
     * </p>
     * 
     * @return When the revision was registered with AWS CodeDeploy.
     */
    public Instant registerTime() {
        return registerTime;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(deploymentGroups());
        hashCode = 31 * hashCode + Objects.hashCode(firstUsedTime());
        hashCode = 31 * hashCode + Objects.hashCode(lastUsedTime());
        hashCode = 31 * hashCode + Objects.hashCode(registerTime());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GenericRevisionInfo)) {
            return false;
        }
        GenericRevisionInfo other = (GenericRevisionInfo) obj;
        return Objects.equals(description(), other.description()) && Objects.equals(deploymentGroups(), other.deploymentGroups())
                && Objects.equals(firstUsedTime(), other.firstUsedTime()) && Objects.equals(lastUsedTime(), other.lastUsedTime())
                && Objects.equals(registerTime(), other.registerTime());
    }

    @Override
    public String toString() {
        return ToString.builder("GenericRevisionInfo").add("Description", description())
                .add("DeploymentGroups", deploymentGroups()).add("FirstUsedTime", firstUsedTime())
                .add("LastUsedTime", lastUsedTime()).add("RegisterTime", registerTime()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "description":
            return Optional.ofNullable(clazz.cast(description()));
        case "deploymentGroups":
            return Optional.ofNullable(clazz.cast(deploymentGroups()));
        case "firstUsedTime":
            return Optional.ofNullable(clazz.cast(firstUsedTime()));
        case "lastUsedTime":
            return Optional.ofNullable(clazz.cast(lastUsedTime()));
        case "registerTime":
            return Optional.ofNullable(clazz.cast(registerTime()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<GenericRevisionInfo, T> g) {
        return obj -> g.apply((GenericRevisionInfo) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, GenericRevisionInfo> {
        /**
         * <p>
         * A comment about the revision.
         * </p>
         * 
         * @param description
         *        A comment about the revision.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The deployment groups for which this is the current target revision.
         * </p>
         * 
         * @param deploymentGroups
         *        The deployment groups for which this is the current target revision.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deploymentGroups(Collection<String> deploymentGroups);

        /**
         * <p>
         * The deployment groups for which this is the current target revision.
         * </p>
         * 
         * @param deploymentGroups
         *        The deployment groups for which this is the current target revision.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deploymentGroups(String... deploymentGroups);

        /**
         * <p>
         * When the revision was first used by AWS CodeDeploy.
         * </p>
         * 
         * @param firstUsedTime
         *        When the revision was first used by AWS CodeDeploy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder firstUsedTime(Instant firstUsedTime);

        /**
         * <p>
         * When the revision was last used by AWS CodeDeploy.
         * </p>
         * 
         * @param lastUsedTime
         *        When the revision was last used by AWS CodeDeploy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUsedTime(Instant lastUsedTime);

        /**
         * <p>
         * When the revision was registered with AWS CodeDeploy.
         * </p>
         * 
         * @param registerTime
         *        When the revision was registered with AWS CodeDeploy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder registerTime(Instant registerTime);
    }

    static final class BuilderImpl implements Builder {
        private String description;

        private List<String> deploymentGroups = DefaultSdkAutoConstructList.getInstance();

        private Instant firstUsedTime;

        private Instant lastUsedTime;

        private Instant registerTime;

        private BuilderImpl() {
        }

        private BuilderImpl(GenericRevisionInfo model) {
            description(model.description);
            deploymentGroups(model.deploymentGroups);
            firstUsedTime(model.firstUsedTime);
            lastUsedTime(model.lastUsedTime);
            registerTime(model.registerTime);
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final Collection<String> getDeploymentGroups() {
            return deploymentGroups;
        }

        @Override
        public final Builder deploymentGroups(Collection<String> deploymentGroups) {
            this.deploymentGroups = DeploymentGroupsListCopier.copy(deploymentGroups);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder deploymentGroups(String... deploymentGroups) {
            deploymentGroups(Arrays.asList(deploymentGroups));
            return this;
        }

        public final void setDeploymentGroups(Collection<String> deploymentGroups) {
            this.deploymentGroups = DeploymentGroupsListCopier.copy(deploymentGroups);
        }

        public final Instant getFirstUsedTime() {
            return firstUsedTime;
        }

        @Override
        public final Builder firstUsedTime(Instant firstUsedTime) {
            this.firstUsedTime = firstUsedTime;
            return this;
        }

        public final void setFirstUsedTime(Instant firstUsedTime) {
            this.firstUsedTime = firstUsedTime;
        }

        public final Instant getLastUsedTime() {
            return lastUsedTime;
        }

        @Override
        public final Builder lastUsedTime(Instant lastUsedTime) {
            this.lastUsedTime = lastUsedTime;
            return this;
        }

        public final void setLastUsedTime(Instant lastUsedTime) {
            this.lastUsedTime = lastUsedTime;
        }

        public final Instant getRegisterTime() {
            return registerTime;
        }

        @Override
        public final Builder registerTime(Instant registerTime) {
            this.registerTime = registerTime;
            return this;
        }

        public final void setRegisterTime(Instant registerTime) {
            this.registerTime = registerTime;
        }

        @Override
        public GenericRevisionInfo build() {
            return new GenericRevisionInfo(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
