/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codedeploy.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about a deployment lifecycle event.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class LifecycleEvent implements SdkPojo, Serializable, ToCopyableBuilder<LifecycleEvent.Builder, LifecycleEvent> {
    private static final SdkField<String> LIFECYCLE_EVENT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(LifecycleEvent::lifecycleEventName)).setter(setter(Builder::lifecycleEventName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lifecycleEventName").build())
            .build();

    private static final SdkField<Diagnostics> DIAGNOSTICS_FIELD = SdkField.<Diagnostics> builder(MarshallingType.SDK_POJO)
            .getter(getter(LifecycleEvent::diagnostics)).setter(setter(Builder::diagnostics)).constructor(Diagnostics::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("diagnostics").build()).build();

    private static final SdkField<Instant> START_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(LifecycleEvent::startTime)).setter(setter(Builder::startTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("startTime").build()).build();

    private static final SdkField<Instant> END_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(LifecycleEvent::endTime)).setter(setter(Builder::endTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("endTime").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(LifecycleEvent::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(LIFECYCLE_EVENT_NAME_FIELD,
            DIAGNOSTICS_FIELD, START_TIME_FIELD, END_TIME_FIELD, STATUS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String lifecycleEventName;

    private final Diagnostics diagnostics;

    private final Instant startTime;

    private final Instant endTime;

    private final String status;

    private LifecycleEvent(BuilderImpl builder) {
        this.lifecycleEventName = builder.lifecycleEventName;
        this.diagnostics = builder.diagnostics;
        this.startTime = builder.startTime;
        this.endTime = builder.endTime;
        this.status = builder.status;
    }

    /**
     * <p>
     * The deployment lifecycle event name, such as ApplicationStop, BeforeInstall, AfterInstall, ApplicationStart, or
     * ValidateService.
     * </p>
     * 
     * @return The deployment lifecycle event name, such as ApplicationStop, BeforeInstall, AfterInstall,
     *         ApplicationStart, or ValidateService.
     */
    public String lifecycleEventName() {
        return lifecycleEventName;
    }

    /**
     * <p>
     * Diagnostic information about the deployment lifecycle event.
     * </p>
     * 
     * @return Diagnostic information about the deployment lifecycle event.
     */
    public Diagnostics diagnostics() {
        return diagnostics;
    }

    /**
     * <p>
     * A timestamp that indicates when the deployment lifecycle event started.
     * </p>
     * 
     * @return A timestamp that indicates when the deployment lifecycle event started.
     */
    public Instant startTime() {
        return startTime;
    }

    /**
     * <p>
     * A timestamp that indicates when the deployment lifecycle event ended.
     * </p>
     * 
     * @return A timestamp that indicates when the deployment lifecycle event ended.
     */
    public Instant endTime() {
        return endTime;
    }

    /**
     * <p>
     * The deployment lifecycle event status:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Pending: The deployment lifecycle event is pending.
     * </p>
     * </li>
     * <li>
     * <p>
     * InProgress: The deployment lifecycle event is in progress.
     * </p>
     * </li>
     * <li>
     * <p>
     * Succeeded: The deployment lifecycle event ran successfully.
     * </p>
     * </li>
     * <li>
     * <p>
     * Failed: The deployment lifecycle event has failed.
     * </p>
     * </li>
     * <li>
     * <p>
     * Skipped: The deployment lifecycle event has been skipped.
     * </p>
     * </li>
     * <li>
     * <p>
     * Unknown: The deployment lifecycle event is unknown.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link LifecycleEventStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The deployment lifecycle event status:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Pending: The deployment lifecycle event is pending.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         InProgress: The deployment lifecycle event is in progress.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Succeeded: The deployment lifecycle event ran successfully.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Failed: The deployment lifecycle event has failed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Skipped: The deployment lifecycle event has been skipped.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Unknown: The deployment lifecycle event is unknown.
     *         </p>
     *         </li>
     * @see LifecycleEventStatus
     */
    public LifecycleEventStatus status() {
        return LifecycleEventStatus.fromValue(status);
    }

    /**
     * <p>
     * The deployment lifecycle event status:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Pending: The deployment lifecycle event is pending.
     * </p>
     * </li>
     * <li>
     * <p>
     * InProgress: The deployment lifecycle event is in progress.
     * </p>
     * </li>
     * <li>
     * <p>
     * Succeeded: The deployment lifecycle event ran successfully.
     * </p>
     * </li>
     * <li>
     * <p>
     * Failed: The deployment lifecycle event has failed.
     * </p>
     * </li>
     * <li>
     * <p>
     * Skipped: The deployment lifecycle event has been skipped.
     * </p>
     * </li>
     * <li>
     * <p>
     * Unknown: The deployment lifecycle event is unknown.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link LifecycleEventStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The deployment lifecycle event status:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Pending: The deployment lifecycle event is pending.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         InProgress: The deployment lifecycle event is in progress.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Succeeded: The deployment lifecycle event ran successfully.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Failed: The deployment lifecycle event has failed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Skipped: The deployment lifecycle event has been skipped.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Unknown: The deployment lifecycle event is unknown.
     *         </p>
     *         </li>
     * @see LifecycleEventStatus
     */
    public String statusAsString() {
        return status;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(lifecycleEventName());
        hashCode = 31 * hashCode + Objects.hashCode(diagnostics());
        hashCode = 31 * hashCode + Objects.hashCode(startTime());
        hashCode = 31 * hashCode + Objects.hashCode(endTime());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof LifecycleEvent)) {
            return false;
        }
        LifecycleEvent other = (LifecycleEvent) obj;
        return Objects.equals(lifecycleEventName(), other.lifecycleEventName())
                && Objects.equals(diagnostics(), other.diagnostics()) && Objects.equals(startTime(), other.startTime())
                && Objects.equals(endTime(), other.endTime()) && Objects.equals(statusAsString(), other.statusAsString());
    }

    @Override
    public String toString() {
        return ToString.builder("LifecycleEvent").add("LifecycleEventName", lifecycleEventName())
                .add("Diagnostics", diagnostics()).add("StartTime", startTime()).add("EndTime", endTime())
                .add("Status", statusAsString()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "lifecycleEventName":
            return Optional.ofNullable(clazz.cast(lifecycleEventName()));
        case "diagnostics":
            return Optional.ofNullable(clazz.cast(diagnostics()));
        case "startTime":
            return Optional.ofNullable(clazz.cast(startTime()));
        case "endTime":
            return Optional.ofNullable(clazz.cast(endTime()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<LifecycleEvent, T> g) {
        return obj -> g.apply((LifecycleEvent) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, LifecycleEvent> {
        /**
         * <p>
         * The deployment lifecycle event name, such as ApplicationStop, BeforeInstall, AfterInstall, ApplicationStart,
         * or ValidateService.
         * </p>
         * 
         * @param lifecycleEventName
         *        The deployment lifecycle event name, such as ApplicationStop, BeforeInstall, AfterInstall,
         *        ApplicationStart, or ValidateService.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lifecycleEventName(String lifecycleEventName);

        /**
         * <p>
         * Diagnostic information about the deployment lifecycle event.
         * </p>
         * 
         * @param diagnostics
         *        Diagnostic information about the deployment lifecycle event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder diagnostics(Diagnostics diagnostics);

        /**
         * <p>
         * Diagnostic information about the deployment lifecycle event.
         * </p>
         * This is a convenience that creates an instance of the {@link Diagnostics.Builder} avoiding the need to create
         * one manually via {@link Diagnostics#builder()}.
         *
         * When the {@link Consumer} completes, {@link Diagnostics.Builder#build()} is called immediately and its result
         * is passed to {@link #diagnostics(Diagnostics)}.
         * 
         * @param diagnostics
         *        a consumer that will call methods on {@link Diagnostics.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #diagnostics(Diagnostics)
         */
        default Builder diagnostics(Consumer<Diagnostics.Builder> diagnostics) {
            return diagnostics(Diagnostics.builder().applyMutation(diagnostics).build());
        }

        /**
         * <p>
         * A timestamp that indicates when the deployment lifecycle event started.
         * </p>
         * 
         * @param startTime
         *        A timestamp that indicates when the deployment lifecycle event started.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTime(Instant startTime);

        /**
         * <p>
         * A timestamp that indicates when the deployment lifecycle event ended.
         * </p>
         * 
         * @param endTime
         *        A timestamp that indicates when the deployment lifecycle event ended.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endTime(Instant endTime);

        /**
         * <p>
         * The deployment lifecycle event status:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Pending: The deployment lifecycle event is pending.
         * </p>
         * </li>
         * <li>
         * <p>
         * InProgress: The deployment lifecycle event is in progress.
         * </p>
         * </li>
         * <li>
         * <p>
         * Succeeded: The deployment lifecycle event ran successfully.
         * </p>
         * </li>
         * <li>
         * <p>
         * Failed: The deployment lifecycle event has failed.
         * </p>
         * </li>
         * <li>
         * <p>
         * Skipped: The deployment lifecycle event has been skipped.
         * </p>
         * </li>
         * <li>
         * <p>
         * Unknown: The deployment lifecycle event is unknown.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The deployment lifecycle event status:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        Pending: The deployment lifecycle event is pending.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        InProgress: The deployment lifecycle event is in progress.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Succeeded: The deployment lifecycle event ran successfully.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Failed: The deployment lifecycle event has failed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Skipped: The deployment lifecycle event has been skipped.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Unknown: The deployment lifecycle event is unknown.
         *        </p>
         *        </li>
         * @see LifecycleEventStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LifecycleEventStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The deployment lifecycle event status:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Pending: The deployment lifecycle event is pending.
         * </p>
         * </li>
         * <li>
         * <p>
         * InProgress: The deployment lifecycle event is in progress.
         * </p>
         * </li>
         * <li>
         * <p>
         * Succeeded: The deployment lifecycle event ran successfully.
         * </p>
         * </li>
         * <li>
         * <p>
         * Failed: The deployment lifecycle event has failed.
         * </p>
         * </li>
         * <li>
         * <p>
         * Skipped: The deployment lifecycle event has been skipped.
         * </p>
         * </li>
         * <li>
         * <p>
         * Unknown: The deployment lifecycle event is unknown.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The deployment lifecycle event status:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        Pending: The deployment lifecycle event is pending.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        InProgress: The deployment lifecycle event is in progress.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Succeeded: The deployment lifecycle event ran successfully.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Failed: The deployment lifecycle event has failed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Skipped: The deployment lifecycle event has been skipped.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Unknown: The deployment lifecycle event is unknown.
         *        </p>
         *        </li>
         * @see LifecycleEventStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LifecycleEventStatus
         */
        Builder status(LifecycleEventStatus status);
    }

    static final class BuilderImpl implements Builder {
        private String lifecycleEventName;

        private Diagnostics diagnostics;

        private Instant startTime;

        private Instant endTime;

        private String status;

        private BuilderImpl() {
        }

        private BuilderImpl(LifecycleEvent model) {
            lifecycleEventName(model.lifecycleEventName);
            diagnostics(model.diagnostics);
            startTime(model.startTime);
            endTime(model.endTime);
            status(model.status);
        }

        public final String getLifecycleEventName() {
            return lifecycleEventName;
        }

        @Override
        public final Builder lifecycleEventName(String lifecycleEventName) {
            this.lifecycleEventName = lifecycleEventName;
            return this;
        }

        public final void setLifecycleEventName(String lifecycleEventName) {
            this.lifecycleEventName = lifecycleEventName;
        }

        public final Diagnostics.Builder getDiagnostics() {
            return diagnostics != null ? diagnostics.toBuilder() : null;
        }

        @Override
        public final Builder diagnostics(Diagnostics diagnostics) {
            this.diagnostics = diagnostics;
            return this;
        }

        public final void setDiagnostics(Diagnostics.BuilderImpl diagnostics) {
            this.diagnostics = diagnostics != null ? diagnostics.build() : null;
        }

        public final Instant getStartTime() {
            return startTime;
        }

        @Override
        public final Builder startTime(Instant startTime) {
            this.startTime = startTime;
            return this;
        }

        public final void setStartTime(Instant startTime) {
            this.startTime = startTime;
        }

        public final Instant getEndTime() {
            return endTime;
        }

        @Override
        public final Builder endTime(Instant endTime) {
            this.endTime = endTime;
            return this;
        }

        public final void setEndTime(Instant endTime) {
            this.endTime = endTime;
        }

        public final String getStatusAsString() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(LifecycleEventStatus status) {
            this.status(status.toString());
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public LifecycleEvent build() {
            return new LifecycleEvent(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
