/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codedeploy.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about the target to be updated by an CloudFormation blue/green deployment. This target type is used for
 * all deployments initiated by a CloudFormation stack update.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CloudFormationTarget implements SdkPojo, Serializable,
        ToCopyableBuilder<CloudFormationTarget.Builder, CloudFormationTarget> {
    private static final SdkField<String> DEPLOYMENT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("deploymentId").getter(getter(CloudFormationTarget::deploymentId)).setter(setter(Builder::deploymentId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("deploymentId").build()).build();

    private static final SdkField<String> TARGET_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("targetId").getter(getter(CloudFormationTarget::targetId)).setter(setter(Builder::targetId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("targetId").build()).build();

    private static final SdkField<Instant> LAST_UPDATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("lastUpdatedAt").getter(getter(CloudFormationTarget::lastUpdatedAt))
            .setter(setter(Builder::lastUpdatedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastUpdatedAt").build()).build();

    private static final SdkField<List<LifecycleEvent>> LIFECYCLE_EVENTS_FIELD = SdkField
            .<List<LifecycleEvent>> builder(MarshallingType.LIST)
            .memberName("lifecycleEvents")
            .getter(getter(CloudFormationTarget::lifecycleEvents))
            .setter(setter(Builder::lifecycleEvents))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lifecycleEvents").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<LifecycleEvent> builder(MarshallingType.SDK_POJO)
                                            .constructor(LifecycleEvent::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(CloudFormationTarget::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<String> RESOURCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("resourceType").getter(getter(CloudFormationTarget::resourceType)).setter(setter(Builder::resourceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourceType").build()).build();

    private static final SdkField<Double> TARGET_VERSION_WEIGHT_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("targetVersionWeight").getter(getter(CloudFormationTarget::targetVersionWeight))
            .setter(setter(Builder::targetVersionWeight))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("targetVersionWeight").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DEPLOYMENT_ID_FIELD,
            TARGET_ID_FIELD, LAST_UPDATED_AT_FIELD, LIFECYCLE_EVENTS_FIELD, STATUS_FIELD, RESOURCE_TYPE_FIELD,
            TARGET_VERSION_WEIGHT_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String deploymentId;

    private final String targetId;

    private final Instant lastUpdatedAt;

    private final List<LifecycleEvent> lifecycleEvents;

    private final String status;

    private final String resourceType;

    private final Double targetVersionWeight;

    private CloudFormationTarget(BuilderImpl builder) {
        this.deploymentId = builder.deploymentId;
        this.targetId = builder.targetId;
        this.lastUpdatedAt = builder.lastUpdatedAt;
        this.lifecycleEvents = builder.lifecycleEvents;
        this.status = builder.status;
        this.resourceType = builder.resourceType;
        this.targetVersionWeight = builder.targetVersionWeight;
    }

    /**
     * <p>
     * The unique ID of an CloudFormation blue/green deployment.
     * </p>
     * 
     * @return The unique ID of an CloudFormation blue/green deployment.
     */
    public final String deploymentId() {
        return deploymentId;
    }

    /**
     * <p>
     * The unique ID of a deployment target that has a type of <code>CloudFormationTarget</code>.
     * </p>
     * 
     * @return The unique ID of a deployment target that has a type of <code>CloudFormationTarget</code>.
     */
    public final String targetId() {
        return targetId;
    }

    /**
     * <p>
     * The date and time when the target application was updated by an CloudFormation blue/green deployment.
     * </p>
     * 
     * @return The date and time when the target application was updated by an CloudFormation blue/green deployment.
     */
    public final Instant lastUpdatedAt() {
        return lastUpdatedAt;
    }

    /**
     * For responses, this returns true if the service returned a value for the LifecycleEvents property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasLifecycleEvents() {
        return lifecycleEvents != null && !(lifecycleEvents instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The lifecycle events of the CloudFormation blue/green deployment to this target application.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasLifecycleEvents} method.
     * </p>
     * 
     * @return The lifecycle events of the CloudFormation blue/green deployment to this target application.
     */
    public final List<LifecycleEvent> lifecycleEvents() {
        return lifecycleEvents;
    }

    /**
     * <p>
     * The status of an CloudFormation blue/green deployment's target application.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link TargetStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of an CloudFormation blue/green deployment's target application.
     * @see TargetStatus
     */
    public final TargetStatus status() {
        return TargetStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of an CloudFormation blue/green deployment's target application.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link TargetStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of an CloudFormation blue/green deployment's target application.
     * @see TargetStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The resource type for the CloudFormation blue/green deployment.
     * </p>
     * 
     * @return The resource type for the CloudFormation blue/green deployment.
     */
    public final String resourceType() {
        return resourceType;
    }

    /**
     * <p>
     * The percentage of production traffic that the target version of an CloudFormation blue/green deployment receives.
     * </p>
     * 
     * @return The percentage of production traffic that the target version of an CloudFormation blue/green deployment
     *         receives.
     */
    public final Double targetVersionWeight() {
        return targetVersionWeight;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(deploymentId());
        hashCode = 31 * hashCode + Objects.hashCode(targetId());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdatedAt());
        hashCode = 31 * hashCode + Objects.hashCode(hasLifecycleEvents() ? lifecycleEvents() : null);
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(resourceType());
        hashCode = 31 * hashCode + Objects.hashCode(targetVersionWeight());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CloudFormationTarget)) {
            return false;
        }
        CloudFormationTarget other = (CloudFormationTarget) obj;
        return Objects.equals(deploymentId(), other.deploymentId()) && Objects.equals(targetId(), other.targetId())
                && Objects.equals(lastUpdatedAt(), other.lastUpdatedAt()) && hasLifecycleEvents() == other.hasLifecycleEvents()
                && Objects.equals(lifecycleEvents(), other.lifecycleEvents())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(resourceType(), other.resourceType())
                && Objects.equals(targetVersionWeight(), other.targetVersionWeight());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CloudFormationTarget").add("DeploymentId", deploymentId()).add("TargetId", targetId())
                .add("LastUpdatedAt", lastUpdatedAt()).add("LifecycleEvents", hasLifecycleEvents() ? lifecycleEvents() : null)
                .add("Status", statusAsString()).add("ResourceType", resourceType())
                .add("TargetVersionWeight", targetVersionWeight()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "deploymentId":
            return Optional.ofNullable(clazz.cast(deploymentId()));
        case "targetId":
            return Optional.ofNullable(clazz.cast(targetId()));
        case "lastUpdatedAt":
            return Optional.ofNullable(clazz.cast(lastUpdatedAt()));
        case "lifecycleEvents":
            return Optional.ofNullable(clazz.cast(lifecycleEvents()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "resourceType":
            return Optional.ofNullable(clazz.cast(resourceType()));
        case "targetVersionWeight":
            return Optional.ofNullable(clazz.cast(targetVersionWeight()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("deploymentId", DEPLOYMENT_ID_FIELD);
        map.put("targetId", TARGET_ID_FIELD);
        map.put("lastUpdatedAt", LAST_UPDATED_AT_FIELD);
        map.put("lifecycleEvents", LIFECYCLE_EVENTS_FIELD);
        map.put("status", STATUS_FIELD);
        map.put("resourceType", RESOURCE_TYPE_FIELD);
        map.put("targetVersionWeight", TARGET_VERSION_WEIGHT_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<CloudFormationTarget, T> g) {
        return obj -> g.apply((CloudFormationTarget) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CloudFormationTarget> {
        /**
         * <p>
         * The unique ID of an CloudFormation blue/green deployment.
         * </p>
         * 
         * @param deploymentId
         *        The unique ID of an CloudFormation blue/green deployment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deploymentId(String deploymentId);

        /**
         * <p>
         * The unique ID of a deployment target that has a type of <code>CloudFormationTarget</code>.
         * </p>
         * 
         * @param targetId
         *        The unique ID of a deployment target that has a type of <code>CloudFormationTarget</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetId(String targetId);

        /**
         * <p>
         * The date and time when the target application was updated by an CloudFormation blue/green deployment.
         * </p>
         * 
         * @param lastUpdatedAt
         *        The date and time when the target application was updated by an CloudFormation blue/green deployment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdatedAt(Instant lastUpdatedAt);

        /**
         * <p>
         * The lifecycle events of the CloudFormation blue/green deployment to this target application.
         * </p>
         * 
         * @param lifecycleEvents
         *        The lifecycle events of the CloudFormation blue/green deployment to this target application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lifecycleEvents(Collection<LifecycleEvent> lifecycleEvents);

        /**
         * <p>
         * The lifecycle events of the CloudFormation blue/green deployment to this target application.
         * </p>
         * 
         * @param lifecycleEvents
         *        The lifecycle events of the CloudFormation blue/green deployment to this target application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lifecycleEvents(LifecycleEvent... lifecycleEvents);

        /**
         * <p>
         * The lifecycle events of the CloudFormation blue/green deployment to this target application.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.codedeploy.model.LifecycleEvent.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.codedeploy.model.LifecycleEvent#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.codedeploy.model.LifecycleEvent.Builder#build()} is called immediately
         * and its result is passed to {@link #lifecycleEvents(List<LifecycleEvent>)}.
         * 
         * @param lifecycleEvents
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.codedeploy.model.LifecycleEvent.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #lifecycleEvents(java.util.Collection<LifecycleEvent>)
         */
        Builder lifecycleEvents(Consumer<LifecycleEvent.Builder>... lifecycleEvents);

        /**
         * <p>
         * The status of an CloudFormation blue/green deployment's target application.
         * </p>
         * 
         * @param status
         *        The status of an CloudFormation blue/green deployment's target application.
         * @see TargetStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TargetStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of an CloudFormation blue/green deployment's target application.
         * </p>
         * 
         * @param status
         *        The status of an CloudFormation blue/green deployment's target application.
         * @see TargetStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TargetStatus
         */
        Builder status(TargetStatus status);

        /**
         * <p>
         * The resource type for the CloudFormation blue/green deployment.
         * </p>
         * 
         * @param resourceType
         *        The resource type for the CloudFormation blue/green deployment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceType(String resourceType);

        /**
         * <p>
         * The percentage of production traffic that the target version of an CloudFormation blue/green deployment
         * receives.
         * </p>
         * 
         * @param targetVersionWeight
         *        The percentage of production traffic that the target version of an CloudFormation blue/green
         *        deployment receives.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetVersionWeight(Double targetVersionWeight);
    }

    static final class BuilderImpl implements Builder {
        private String deploymentId;

        private String targetId;

        private Instant lastUpdatedAt;

        private List<LifecycleEvent> lifecycleEvents = DefaultSdkAutoConstructList.getInstance();

        private String status;

        private String resourceType;

        private Double targetVersionWeight;

        private BuilderImpl() {
        }

        private BuilderImpl(CloudFormationTarget model) {
            deploymentId(model.deploymentId);
            targetId(model.targetId);
            lastUpdatedAt(model.lastUpdatedAt);
            lifecycleEvents(model.lifecycleEvents);
            status(model.status);
            resourceType(model.resourceType);
            targetVersionWeight(model.targetVersionWeight);
        }

        public final String getDeploymentId() {
            return deploymentId;
        }

        public final void setDeploymentId(String deploymentId) {
            this.deploymentId = deploymentId;
        }

        @Override
        public final Builder deploymentId(String deploymentId) {
            this.deploymentId = deploymentId;
            return this;
        }

        public final String getTargetId() {
            return targetId;
        }

        public final void setTargetId(String targetId) {
            this.targetId = targetId;
        }

        @Override
        public final Builder targetId(String targetId) {
            this.targetId = targetId;
            return this;
        }

        public final Instant getLastUpdatedAt() {
            return lastUpdatedAt;
        }

        public final void setLastUpdatedAt(Instant lastUpdatedAt) {
            this.lastUpdatedAt = lastUpdatedAt;
        }

        @Override
        public final Builder lastUpdatedAt(Instant lastUpdatedAt) {
            this.lastUpdatedAt = lastUpdatedAt;
            return this;
        }

        public final List<LifecycleEvent.Builder> getLifecycleEvents() {
            List<LifecycleEvent.Builder> result = LifecycleEventListCopier.copyToBuilder(this.lifecycleEvents);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setLifecycleEvents(Collection<LifecycleEvent.BuilderImpl> lifecycleEvents) {
            this.lifecycleEvents = LifecycleEventListCopier.copyFromBuilder(lifecycleEvents);
        }

        @Override
        public final Builder lifecycleEvents(Collection<LifecycleEvent> lifecycleEvents) {
            this.lifecycleEvents = LifecycleEventListCopier.copy(lifecycleEvents);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder lifecycleEvents(LifecycleEvent... lifecycleEvents) {
            lifecycleEvents(Arrays.asList(lifecycleEvents));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder lifecycleEvents(Consumer<LifecycleEvent.Builder>... lifecycleEvents) {
            lifecycleEvents(Stream.of(lifecycleEvents).map(c -> LifecycleEvent.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(TargetStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final String getResourceType() {
            return resourceType;
        }

        public final void setResourceType(String resourceType) {
            this.resourceType = resourceType;
        }

        @Override
        public final Builder resourceType(String resourceType) {
            this.resourceType = resourceType;
            return this;
        }

        public final Double getTargetVersionWeight() {
            return targetVersionWeight;
        }

        public final void setTargetVersionWeight(Double targetVersionWeight) {
            this.targetVersionWeight = targetVersionWeight;
        }

        @Override
        public final Builder targetVersionWeight(Double targetVersionWeight) {
            this.targetVersionWeight = targetVersionWeight;
            return this;
        }

        @Override
        public CloudFormationTarget build() {
            return new CloudFormationTarget(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
