/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codedeploy.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about whether instances in the original environment are terminated when a blue/green deployment is
 * successful.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class BlueInstanceTerminationOption implements SdkPojo, Serializable,
        ToCopyableBuilder<BlueInstanceTerminationOption.Builder, BlueInstanceTerminationOption> {
    private static final SdkField<String> ACTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(BlueInstanceTerminationOption::actionAsString)).setter(setter(Builder::action))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("action").build()).build();

    private static final SdkField<Integer> TERMINATION_WAIT_TIME_IN_MINUTES_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(BlueInstanceTerminationOption::terminationWaitTimeInMinutes))
            .setter(setter(Builder::terminationWaitTimeInMinutes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("terminationWaitTimeInMinutes")
                    .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ACTION_FIELD,
            TERMINATION_WAIT_TIME_IN_MINUTES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String action;

    private final Integer terminationWaitTimeInMinutes;

    private BlueInstanceTerminationOption(BuilderImpl builder) {
        this.action = builder.action;
        this.terminationWaitTimeInMinutes = builder.terminationWaitTimeInMinutes;
    }

    /**
     * <p>
     * The action to take on instances in the original environment after a successful blue/green deployment.
     * </p>
     * <ul>
     * <li>
     * <p>
     * TERMINATE: Instances are terminated after a specified wait time.
     * </p>
     * </li>
     * <li>
     * <p>
     * KEEP_ALIVE: Instances are left running after they are deregistered from the load balancer and removed from the
     * deployment group.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #action} will
     * return {@link InstanceAction#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #actionAsString}.
     * </p>
     * 
     * @return The action to take on instances in the original environment after a successful blue/green deployment.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         TERMINATE: Instances are terminated after a specified wait time.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         KEEP_ALIVE: Instances are left running after they are deregistered from the load balancer and removed
     *         from the deployment group.
     *         </p>
     *         </li>
     * @see InstanceAction
     */
    public InstanceAction action() {
        return InstanceAction.fromValue(action);
    }

    /**
     * <p>
     * The action to take on instances in the original environment after a successful blue/green deployment.
     * </p>
     * <ul>
     * <li>
     * <p>
     * TERMINATE: Instances are terminated after a specified wait time.
     * </p>
     * </li>
     * <li>
     * <p>
     * KEEP_ALIVE: Instances are left running after they are deregistered from the load balancer and removed from the
     * deployment group.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #action} will
     * return {@link InstanceAction#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #actionAsString}.
     * </p>
     * 
     * @return The action to take on instances in the original environment after a successful blue/green deployment.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         TERMINATE: Instances are terminated after a specified wait time.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         KEEP_ALIVE: Instances are left running after they are deregistered from the load balancer and removed
     *         from the deployment group.
     *         </p>
     *         </li>
     * @see InstanceAction
     */
    public String actionAsString() {
        return action;
    }

    /**
     * <p>
     * The number of minutes to wait after a successful blue/green deployment before terminating instances from the
     * original environment. The maximum setting is 2880 minutes (2 days).
     * </p>
     * 
     * @return The number of minutes to wait after a successful blue/green deployment before terminating instances from
     *         the original environment. The maximum setting is 2880 minutes (2 days).
     */
    public Integer terminationWaitTimeInMinutes() {
        return terminationWaitTimeInMinutes;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(actionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(terminationWaitTimeInMinutes());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof BlueInstanceTerminationOption)) {
            return false;
        }
        BlueInstanceTerminationOption other = (BlueInstanceTerminationOption) obj;
        return Objects.equals(actionAsString(), other.actionAsString())
                && Objects.equals(terminationWaitTimeInMinutes(), other.terminationWaitTimeInMinutes());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("BlueInstanceTerminationOption").add("Action", actionAsString())
                .add("TerminationWaitTimeInMinutes", terminationWaitTimeInMinutes()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "action":
            return Optional.ofNullable(clazz.cast(actionAsString()));
        case "terminationWaitTimeInMinutes":
            return Optional.ofNullable(clazz.cast(terminationWaitTimeInMinutes()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<BlueInstanceTerminationOption, T> g) {
        return obj -> g.apply((BlueInstanceTerminationOption) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, BlueInstanceTerminationOption> {
        /**
         * <p>
         * The action to take on instances in the original environment after a successful blue/green deployment.
         * </p>
         * <ul>
         * <li>
         * <p>
         * TERMINATE: Instances are terminated after a specified wait time.
         * </p>
         * </li>
         * <li>
         * <p>
         * KEEP_ALIVE: Instances are left running after they are deregistered from the load balancer and removed from
         * the deployment group.
         * </p>
         * </li>
         * </ul>
         * 
         * @param action
         *        The action to take on instances in the original environment after a successful blue/green
         *        deployment.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        TERMINATE: Instances are terminated after a specified wait time.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        KEEP_ALIVE: Instances are left running after they are deregistered from the load balancer and removed
         *        from the deployment group.
         *        </p>
         *        </li>
         * @see InstanceAction
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InstanceAction
         */
        Builder action(String action);

        /**
         * <p>
         * The action to take on instances in the original environment after a successful blue/green deployment.
         * </p>
         * <ul>
         * <li>
         * <p>
         * TERMINATE: Instances are terminated after a specified wait time.
         * </p>
         * </li>
         * <li>
         * <p>
         * KEEP_ALIVE: Instances are left running after they are deregistered from the load balancer and removed from
         * the deployment group.
         * </p>
         * </li>
         * </ul>
         * 
         * @param action
         *        The action to take on instances in the original environment after a successful blue/green
         *        deployment.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        TERMINATE: Instances are terminated after a specified wait time.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        KEEP_ALIVE: Instances are left running after they are deregistered from the load balancer and removed
         *        from the deployment group.
         *        </p>
         *        </li>
         * @see InstanceAction
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InstanceAction
         */
        Builder action(InstanceAction action);

        /**
         * <p>
         * The number of minutes to wait after a successful blue/green deployment before terminating instances from the
         * original environment. The maximum setting is 2880 minutes (2 days).
         * </p>
         * 
         * @param terminationWaitTimeInMinutes
         *        The number of minutes to wait after a successful blue/green deployment before terminating instances
         *        from the original environment. The maximum setting is 2880 minutes (2 days).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder terminationWaitTimeInMinutes(Integer terminationWaitTimeInMinutes);
    }

    static final class BuilderImpl implements Builder {
        private String action;

        private Integer terminationWaitTimeInMinutes;

        private BuilderImpl() {
        }

        private BuilderImpl(BlueInstanceTerminationOption model) {
            action(model.action);
            terminationWaitTimeInMinutes(model.terminationWaitTimeInMinutes);
        }

        public final String getActionAsString() {
            return action;
        }

        @Override
        public final Builder action(String action) {
            this.action = action;
            return this;
        }

        @Override
        public final Builder action(InstanceAction action) {
            this.action(action.toString());
            return this;
        }

        public final void setAction(String action) {
            this.action = action;
        }

        public final Integer getTerminationWaitTimeInMinutes() {
            return terminationWaitTimeInMinutes;
        }

        @Override
        public final Builder terminationWaitTimeInMinutes(Integer terminationWaitTimeInMinutes) {
            this.terminationWaitTimeInMinutes = terminationWaitTimeInMinutes;
            return this;
        }

        public final void setTerminationWaitTimeInMinutes(Integer terminationWaitTimeInMinutes) {
            this.terminationWaitTimeInMinutes = terminationWaitTimeInMinutes;
        }

        @Override
        public BlueInstanceTerminationOption build() {
            return new BlueInstanceTerminationOption(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
