/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codedeploy.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A target Amazon EC2 or on-premises instance during a deployment that uses the EC2/On-premises compute platform.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class InstanceTarget implements SdkPojo, Serializable, ToCopyableBuilder<InstanceTarget.Builder, InstanceTarget> {
    private static final SdkField<String> DEPLOYMENT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(InstanceTarget::deploymentId)).setter(setter(Builder::deploymentId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("deploymentId").build()).build();

    private static final SdkField<String> TARGET_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(InstanceTarget::targetId)).setter(setter(Builder::targetId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("targetId").build()).build();

    private static final SdkField<String> TARGET_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(InstanceTarget::targetArn)).setter(setter(Builder::targetArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("targetArn").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(InstanceTarget::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<Instant> LAST_UPDATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(InstanceTarget::lastUpdatedAt)).setter(setter(Builder::lastUpdatedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastUpdatedAt").build()).build();

    private static final SdkField<List<LifecycleEvent>> LIFECYCLE_EVENTS_FIELD = SdkField
            .<List<LifecycleEvent>> builder(MarshallingType.LIST)
            .getter(getter(InstanceTarget::lifecycleEvents))
            .setter(setter(Builder::lifecycleEvents))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lifecycleEvents").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<LifecycleEvent> builder(MarshallingType.SDK_POJO)
                                            .constructor(LifecycleEvent::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> INSTANCE_LABEL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(InstanceTarget::instanceLabelAsString)).setter(setter(Builder::instanceLabel))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("instanceLabel").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections
            .unmodifiableList(Arrays.asList(DEPLOYMENT_ID_FIELD, TARGET_ID_FIELD, TARGET_ARN_FIELD, STATUS_FIELD,
                    LAST_UPDATED_AT_FIELD, LIFECYCLE_EVENTS_FIELD, INSTANCE_LABEL_FIELD));

    private static final long serialVersionUID = 1L;

    private final String deploymentId;

    private final String targetId;

    private final String targetArn;

    private final String status;

    private final Instant lastUpdatedAt;

    private final List<LifecycleEvent> lifecycleEvents;

    private final String instanceLabel;

    private InstanceTarget(BuilderImpl builder) {
        this.deploymentId = builder.deploymentId;
        this.targetId = builder.targetId;
        this.targetArn = builder.targetArn;
        this.status = builder.status;
        this.lastUpdatedAt = builder.lastUpdatedAt;
        this.lifecycleEvents = builder.lifecycleEvents;
        this.instanceLabel = builder.instanceLabel;
    }

    /**
     * <p>
     * The unique ID of a deployment.
     * </p>
     * 
     * @return The unique ID of a deployment.
     */
    public String deploymentId() {
        return deploymentId;
    }

    /**
     * <p>
     * The unique ID of a deployment target that has a type of <code>instanceTarget</code>.
     * </p>
     * 
     * @return The unique ID of a deployment target that has a type of <code>instanceTarget</code>.
     */
    public String targetId() {
        return targetId;
    }

    /**
     * <p>
     * The ARN of the target.
     * </p>
     * 
     * @return The ARN of the target.
     */
    public String targetArn() {
        return targetArn;
    }

    /**
     * <p>
     * The status an EC2/On-premises deployment's target instance.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link TargetStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status an EC2/On-premises deployment's target instance.
     * @see TargetStatus
     */
    public TargetStatus status() {
        return TargetStatus.fromValue(status);
    }

    /**
     * <p>
     * The status an EC2/On-premises deployment's target instance.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link TargetStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status an EC2/On-premises deployment's target instance.
     * @see TargetStatus
     */
    public String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The date and time when the target instance was updated by a deployment.
     * </p>
     * 
     * @return The date and time when the target instance was updated by a deployment.
     */
    public Instant lastUpdatedAt() {
        return lastUpdatedAt;
    }

    /**
     * <p>
     * The lifecycle events of the deployment to this target instance.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The lifecycle events of the deployment to this target instance.
     */
    public List<LifecycleEvent> lifecycleEvents() {
        return lifecycleEvents;
    }

    /**
     * <p>
     * A label that identifies whether the instance is an original target (<code>BLUE</code>) or a replacement target (
     * <code>GREEN</code>).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceLabel}
     * will return {@link TargetLabel#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #instanceLabelAsString}.
     * </p>
     * 
     * @return A label that identifies whether the instance is an original target (<code>BLUE</code>) or a replacement
     *         target (<code>GREEN</code>).
     * @see TargetLabel
     */
    public TargetLabel instanceLabel() {
        return TargetLabel.fromValue(instanceLabel);
    }

    /**
     * <p>
     * A label that identifies whether the instance is an original target (<code>BLUE</code>) or a replacement target (
     * <code>GREEN</code>).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceLabel}
     * will return {@link TargetLabel#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #instanceLabelAsString}.
     * </p>
     * 
     * @return A label that identifies whether the instance is an original target (<code>BLUE</code>) or a replacement
     *         target (<code>GREEN</code>).
     * @see TargetLabel
     */
    public String instanceLabelAsString() {
        return instanceLabel;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(deploymentId());
        hashCode = 31 * hashCode + Objects.hashCode(targetId());
        hashCode = 31 * hashCode + Objects.hashCode(targetArn());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdatedAt());
        hashCode = 31 * hashCode + Objects.hashCode(lifecycleEvents());
        hashCode = 31 * hashCode + Objects.hashCode(instanceLabelAsString());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof InstanceTarget)) {
            return false;
        }
        InstanceTarget other = (InstanceTarget) obj;
        return Objects.equals(deploymentId(), other.deploymentId()) && Objects.equals(targetId(), other.targetId())
                && Objects.equals(targetArn(), other.targetArn()) && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(lastUpdatedAt(), other.lastUpdatedAt())
                && Objects.equals(lifecycleEvents(), other.lifecycleEvents())
                && Objects.equals(instanceLabelAsString(), other.instanceLabelAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("InstanceTarget").add("DeploymentId", deploymentId()).add("TargetId", targetId())
                .add("TargetArn", targetArn()).add("Status", statusAsString()).add("LastUpdatedAt", lastUpdatedAt())
                .add("LifecycleEvents", lifecycleEvents()).add("InstanceLabel", instanceLabelAsString()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "deploymentId":
            return Optional.ofNullable(clazz.cast(deploymentId()));
        case "targetId":
            return Optional.ofNullable(clazz.cast(targetId()));
        case "targetArn":
            return Optional.ofNullable(clazz.cast(targetArn()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "lastUpdatedAt":
            return Optional.ofNullable(clazz.cast(lastUpdatedAt()));
        case "lifecycleEvents":
            return Optional.ofNullable(clazz.cast(lifecycleEvents()));
        case "instanceLabel":
            return Optional.ofNullable(clazz.cast(instanceLabelAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<InstanceTarget, T> g) {
        return obj -> g.apply((InstanceTarget) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, InstanceTarget> {
        /**
         * <p>
         * The unique ID of a deployment.
         * </p>
         * 
         * @param deploymentId
         *        The unique ID of a deployment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deploymentId(String deploymentId);

        /**
         * <p>
         * The unique ID of a deployment target that has a type of <code>instanceTarget</code>.
         * </p>
         * 
         * @param targetId
         *        The unique ID of a deployment target that has a type of <code>instanceTarget</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetId(String targetId);

        /**
         * <p>
         * The ARN of the target.
         * </p>
         * 
         * @param targetArn
         *        The ARN of the target.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetArn(String targetArn);

        /**
         * <p>
         * The status an EC2/On-premises deployment's target instance.
         * </p>
         * 
         * @param status
         *        The status an EC2/On-premises deployment's target instance.
         * @see TargetStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TargetStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status an EC2/On-premises deployment's target instance.
         * </p>
         * 
         * @param status
         *        The status an EC2/On-premises deployment's target instance.
         * @see TargetStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TargetStatus
         */
        Builder status(TargetStatus status);

        /**
         * <p>
         * The date and time when the target instance was updated by a deployment.
         * </p>
         * 
         * @param lastUpdatedAt
         *        The date and time when the target instance was updated by a deployment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdatedAt(Instant lastUpdatedAt);

        /**
         * <p>
         * The lifecycle events of the deployment to this target instance.
         * </p>
         * 
         * @param lifecycleEvents
         *        The lifecycle events of the deployment to this target instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lifecycleEvents(Collection<LifecycleEvent> lifecycleEvents);

        /**
         * <p>
         * The lifecycle events of the deployment to this target instance.
         * </p>
         * 
         * @param lifecycleEvents
         *        The lifecycle events of the deployment to this target instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lifecycleEvents(LifecycleEvent... lifecycleEvents);

        /**
         * <p>
         * The lifecycle events of the deployment to this target instance.
         * </p>
         * This is a convenience that creates an instance of the {@link List<LifecycleEvent>.Builder} avoiding the need
         * to create one manually via {@link List<LifecycleEvent>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<LifecycleEvent>.Builder#build()} is called immediately and
         * its result is passed to {@link #lifecycleEvents(List<LifecycleEvent>)}.
         * 
         * @param lifecycleEvents
         *        a consumer that will call methods on {@link List<LifecycleEvent>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #lifecycleEvents(List<LifecycleEvent>)
         */
        Builder lifecycleEvents(Consumer<LifecycleEvent.Builder>... lifecycleEvents);

        /**
         * <p>
         * A label that identifies whether the instance is an original target (<code>BLUE</code>) or a replacement
         * target (<code>GREEN</code>).
         * </p>
         * 
         * @param instanceLabel
         *        A label that identifies whether the instance is an original target (<code>BLUE</code>) or a
         *        replacement target (<code>GREEN</code>).
         * @see TargetLabel
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TargetLabel
         */
        Builder instanceLabel(String instanceLabel);

        /**
         * <p>
         * A label that identifies whether the instance is an original target (<code>BLUE</code>) or a replacement
         * target (<code>GREEN</code>).
         * </p>
         * 
         * @param instanceLabel
         *        A label that identifies whether the instance is an original target (<code>BLUE</code>) or a
         *        replacement target (<code>GREEN</code>).
         * @see TargetLabel
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TargetLabel
         */
        Builder instanceLabel(TargetLabel instanceLabel);
    }

    static final class BuilderImpl implements Builder {
        private String deploymentId;

        private String targetId;

        private String targetArn;

        private String status;

        private Instant lastUpdatedAt;

        private List<LifecycleEvent> lifecycleEvents = DefaultSdkAutoConstructList.getInstance();

        private String instanceLabel;

        private BuilderImpl() {
        }

        private BuilderImpl(InstanceTarget model) {
            deploymentId(model.deploymentId);
            targetId(model.targetId);
            targetArn(model.targetArn);
            status(model.status);
            lastUpdatedAt(model.lastUpdatedAt);
            lifecycleEvents(model.lifecycleEvents);
            instanceLabel(model.instanceLabel);
        }

        public final String getDeploymentId() {
            return deploymentId;
        }

        @Override
        public final Builder deploymentId(String deploymentId) {
            this.deploymentId = deploymentId;
            return this;
        }

        public final void setDeploymentId(String deploymentId) {
            this.deploymentId = deploymentId;
        }

        public final String getTargetId() {
            return targetId;
        }

        @Override
        public final Builder targetId(String targetId) {
            this.targetId = targetId;
            return this;
        }

        public final void setTargetId(String targetId) {
            this.targetId = targetId;
        }

        public final String getTargetArn() {
            return targetArn;
        }

        @Override
        public final Builder targetArn(String targetArn) {
            this.targetArn = targetArn;
            return this;
        }

        public final void setTargetArn(String targetArn) {
            this.targetArn = targetArn;
        }

        public final String getStatusAsString() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(TargetStatus status) {
            this.status(status.toString());
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final Instant getLastUpdatedAt() {
            return lastUpdatedAt;
        }

        @Override
        public final Builder lastUpdatedAt(Instant lastUpdatedAt) {
            this.lastUpdatedAt = lastUpdatedAt;
            return this;
        }

        public final void setLastUpdatedAt(Instant lastUpdatedAt) {
            this.lastUpdatedAt = lastUpdatedAt;
        }

        public final Collection<LifecycleEvent.Builder> getLifecycleEvents() {
            return lifecycleEvents != null ? lifecycleEvents.stream().map(LifecycleEvent::toBuilder).collect(Collectors.toList())
                    : null;
        }

        @Override
        public final Builder lifecycleEvents(Collection<LifecycleEvent> lifecycleEvents) {
            this.lifecycleEvents = LifecycleEventListCopier.copy(lifecycleEvents);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder lifecycleEvents(LifecycleEvent... lifecycleEvents) {
            lifecycleEvents(Arrays.asList(lifecycleEvents));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder lifecycleEvents(Consumer<LifecycleEvent.Builder>... lifecycleEvents) {
            lifecycleEvents(Stream.of(lifecycleEvents).map(c -> LifecycleEvent.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setLifecycleEvents(Collection<LifecycleEvent.BuilderImpl> lifecycleEvents) {
            this.lifecycleEvents = LifecycleEventListCopier.copyFromBuilder(lifecycleEvents);
        }

        public final String getInstanceLabelAsString() {
            return instanceLabel;
        }

        @Override
        public final Builder instanceLabel(String instanceLabel) {
            this.instanceLabel = instanceLabel;
            return this;
        }

        @Override
        public final Builder instanceLabel(TargetLabel instanceLabel) {
            this.instanceLabel(instanceLabel.toString());
            return this;
        }

        public final void setInstanceLabel(String instanceLabel) {
            this.instanceLabel = instanceLabel;
        }

        @Override
        public InstanceTarget build() {
            return new InstanceTarget(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
