/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.elastictranscoder.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Parameters required for transcoding audio.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AudioParameters implements SdkPojo, Serializable, ToCopyableBuilder<AudioParameters.Builder, AudioParameters> {
    private static final SdkField<String> CODEC_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(AudioParameters::codec)).setter(setter(Builder::codec))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Codec").build()).build();

    private static final SdkField<String> SAMPLE_RATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(AudioParameters::sampleRate)).setter(setter(Builder::sampleRate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SampleRate").build()).build();

    private static final SdkField<String> BIT_RATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(AudioParameters::bitRate)).setter(setter(Builder::bitRate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BitRate").build()).build();

    private static final SdkField<String> CHANNELS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(AudioParameters::channels)).setter(setter(Builder::channels))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Channels").build()).build();

    private static final SdkField<String> AUDIO_PACKING_MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(AudioParameters::audioPackingMode)).setter(setter(Builder::audioPackingMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AudioPackingMode").build()).build();

    private static final SdkField<AudioCodecOptions> CODEC_OPTIONS_FIELD = SdkField
            .<AudioCodecOptions> builder(MarshallingType.SDK_POJO).getter(getter(AudioParameters::codecOptions))
            .setter(setter(Builder::codecOptions)).constructor(AudioCodecOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CodecOptions").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CODEC_FIELD,
            SAMPLE_RATE_FIELD, BIT_RATE_FIELD, CHANNELS_FIELD, AUDIO_PACKING_MODE_FIELD, CODEC_OPTIONS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String codec;

    private final String sampleRate;

    private final String bitRate;

    private final String channels;

    private final String audioPackingMode;

    private final AudioCodecOptions codecOptions;

    private AudioParameters(BuilderImpl builder) {
        this.codec = builder.codec;
        this.sampleRate = builder.sampleRate;
        this.bitRate = builder.bitRate;
        this.channels = builder.channels;
        this.audioPackingMode = builder.audioPackingMode;
        this.codecOptions = builder.codecOptions;
    }

    /**
     * <p>
     * The audio codec for the output file. Valid values include <code>aac</code>, <code>flac</code>, <code>mp2</code>,
     * <code>mp3</code>, <code>pcm</code>, and <code>vorbis</code>.
     * </p>
     * 
     * @return The audio codec for the output file. Valid values include <code>aac</code>, <code>flac</code>,
     *         <code>mp2</code>, <code>mp3</code>, <code>pcm</code>, and <code>vorbis</code>.
     */
    public String codec() {
        return codec;
    }

    /**
     * <p>
     * The sample rate of the audio stream in the output file, in Hertz. Valid values include:
     * </p>
     * <p>
     * <code>auto</code>, <code>22050</code>, <code>32000</code>, <code>44100</code>, <code>48000</code>,
     * <code>96000</code>
     * </p>
     * <p>
     * If you specify <code>auto</code>, Elastic Transcoder automatically detects the sample rate.
     * </p>
     * 
     * @return The sample rate of the audio stream in the output file, in Hertz. Valid values include:</p>
     *         <p>
     *         <code>auto</code>, <code>22050</code>, <code>32000</code>, <code>44100</code>, <code>48000</code>,
     *         <code>96000</code>
     *         </p>
     *         <p>
     *         If you specify <code>auto</code>, Elastic Transcoder automatically detects the sample rate.
     */
    public String sampleRate() {
        return sampleRate;
    }

    /**
     * <p>
     * The bit rate of the audio stream in the output file, in kilobits/second. Enter an integer between 64 and 320,
     * inclusive.
     * </p>
     * 
     * @return The bit rate of the audio stream in the output file, in kilobits/second. Enter an integer between 64 and
     *         320, inclusive.
     */
    public String bitRate() {
        return bitRate;
    }

    /**
     * <p>
     * The number of audio channels in the output file. The following values are valid:
     * </p>
     * <p>
     * <code>auto</code>, <code>0</code>, <code>1</code>, <code>2</code>
     * </p>
     * <p>
     * One channel carries the information played by a single speaker. For example, a stereo track with two channels
     * sends one channel to the left speaker, and the other channel to the right speaker. The output channels are
     * organized into tracks. If you want Elastic Transcoder to automatically detect the number of audio channels in the
     * input file and use that value for the output file, select <code>auto</code>.
     * </p>
     * <p>
     * The output of a specific channel value and inputs are as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>auto</code> <b> channel specified, with any input:</b> Pass through up to eight input channels.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>0</code> <b> channels specified, with any input:</b> Audio omitted from the output.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>1</code> <b> channel specified, with at least one input channel:</b> Mono sound.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>2</code> <b> channels specified, with any input:</b> Two identical mono channels or stereo. For more
     * information about tracks, see <code>Audio:AudioPackingMode.</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information about how Elastic Transcoder organizes channels and tracks, see
     * <code>Audio:AudioPackingMode</code>.
     * </p>
     * 
     * @return The number of audio channels in the output file. The following values are valid:</p>
     *         <p>
     *         <code>auto</code>, <code>0</code>, <code>1</code>, <code>2</code>
     *         </p>
     *         <p>
     *         One channel carries the information played by a single speaker. For example, a stereo track with two
     *         channels sends one channel to the left speaker, and the other channel to the right speaker. The output
     *         channels are organized into tracks. If you want Elastic Transcoder to automatically detect the number of
     *         audio channels in the input file and use that value for the output file, select <code>auto</code>.
     *         </p>
     *         <p>
     *         The output of a specific channel value and inputs are as follows:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>auto</code> <b> channel specified, with any input:</b> Pass through up to eight input channels.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>0</code> <b> channels specified, with any input:</b> Audio omitted from the output.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>1</code> <b> channel specified, with at least one input channel:</b> Mono sound.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>2</code> <b> channels specified, with any input:</b> Two identical mono channels or stereo. For
     *         more information about tracks, see <code>Audio:AudioPackingMode.</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For more information about how Elastic Transcoder organizes channels and tracks, see
     *         <code>Audio:AudioPackingMode</code>.
     */
    public String channels() {
        return channels;
    }

    /**
     * <p>
     * The method of organizing audio channels and tracks. Use <code>Audio:Channels</code> to specify the number of
     * channels in your output, and <code>Audio:AudioPackingMode</code> to specify the number of tracks and their
     * relation to the channels. If you do not specify an <code>Audio:AudioPackingMode</code>, Elastic Transcoder uses
     * <code>SingleTrack</code>.
     * </p>
     * <p>
     * The following values are valid:
     * </p>
     * <p>
     * <code>SingleTrack</code>, <code>OneChannelPerTrack</code>, and <code>OneChannelPerTrackWithMosTo8Tracks</code>
     * </p>
     * <p>
     * When you specify <code>SingleTrack</code>, Elastic Transcoder creates a single track for your output. The track
     * can have up to eight channels. Use <code>SingleTrack</code> for all non-<code>mxf</code> containers.
     * </p>
     * <p>
     * The outputs of <code>SingleTrack</code> for a specific channel value and inputs are as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>0</code> <b> channels with any input:</b> Audio omitted from the output
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>1, 2, or auto </code> <b>channels with no audio input:</b> Audio omitted from the output
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>1 </code> <b>channel with any input with audio:</b> One track with one channel, downmixed if necessary
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>2 </code> <b>channels with one track with one channel:</b> One track with two identical channels
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>2 or auto </code> <b>channels with two tracks with one channel each:</b> One track with two channels
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>2 or auto </code> <b>channels with one track with two channels:</b> One track with two channels
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>2 </code> <b>channels with one track with multiple channels:</b> One track with two channels
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>auto </code> <b>channels with one track with one channel:</b> One track with one channel
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>auto </code> <b>channels with one track with multiple channels:</b> One track with multiple channels
     * </p>
     * </li>
     * </ul>
     * <p>
     * When you specify <code>OneChannelPerTrack</code>, Elastic Transcoder creates a new track for every channel in
     * your output. Your output can have up to eight single-channel tracks.
     * </p>
     * <p>
     * The outputs of <code>OneChannelPerTrack</code> for a specific channel value and inputs are as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>0 </code> <b>channels with any input:</b> Audio omitted from the output
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>1, 2, or auto </code> <b>channels with no audio input:</b> Audio omitted from the output
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>1 </code> <b>channel with any input with audio:</b> One track with one channel, downmixed if necessary
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>2 </code> <b>channels with one track with one channel:</b> Two tracks with one identical channel each
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>2 or auto </code> <b>channels with two tracks with one channel each:</b> Two tracks with one channel each
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>2 or auto </code> <b>channels with one track with two channels:</b> Two tracks with one channel each
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>2 </code> <b>channels with one track with multiple channels:</b> Two tracks with one channel each
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>auto </code> <b>channels with one track with one channel:</b> One track with one channel
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>auto </code> <b>channels with one track with multiple channels:</b> Up to eight tracks with one channel
     * each
     * </p>
     * </li>
     * </ul>
     * <p>
     * When you specify <code>OneChannelPerTrackWithMosTo8Tracks</code>, Elastic Transcoder creates eight single-channel
     * tracks for your output. All tracks that do not contain audio data from an input channel are MOS, or Mit Out
     * Sound, tracks.
     * </p>
     * <p>
     * The outputs of <code>OneChannelPerTrackWithMosTo8Tracks</code> for a specific channel value and inputs are as
     * follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>0 </code> <b>channels with any input:</b> Audio omitted from the output
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>1, 2, or auto </code> <b>channels with no audio input:</b> Audio omitted from the output
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>1 </code> <b>channel with any input with audio:</b> One track with one channel, downmixed if necessary,
     * plus six MOS tracks
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>2 </code> <b>channels with one track with one channel:</b> Two tracks with one identical channel each, plus
     * six MOS tracks
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>2 or auto </code> <b>channels with two tracks with one channel each:</b> Two tracks with one channel each,
     * plus six MOS tracks
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>2 or auto </code> <b>channels with one track with two channels:</b> Two tracks with one channel each, plus
     * six MOS tracks
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>2 </code> <b>channels with one track with multiple channels:</b> Two tracks with one channel each, plus six
     * MOS tracks
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>auto </code> <b>channels with one track with one channel:</b> One track with one channel, plus seven MOS
     * tracks
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>auto </code> <b>channels with one track with multiple channels:</b> Up to eight tracks with one channel
     * each, plus MOS tracks until there are eight tracks in all
     * </p>
     * </li>
     * </ul>
     * 
     * @return The method of organizing audio channels and tracks. Use <code>Audio:Channels</code> to specify the number
     *         of channels in your output, and <code>Audio:AudioPackingMode</code> to specify the number of tracks and
     *         their relation to the channels. If you do not specify an <code>Audio:AudioPackingMode</code>, Elastic
     *         Transcoder uses <code>SingleTrack</code>.</p>
     *         <p>
     *         The following values are valid:
     *         </p>
     *         <p>
     *         <code>SingleTrack</code>, <code>OneChannelPerTrack</code>, and
     *         <code>OneChannelPerTrackWithMosTo8Tracks</code>
     *         </p>
     *         <p>
     *         When you specify <code>SingleTrack</code>, Elastic Transcoder creates a single track for your output. The
     *         track can have up to eight channels. Use <code>SingleTrack</code> for all non-<code>mxf</code>
     *         containers.
     *         </p>
     *         <p>
     *         The outputs of <code>SingleTrack</code> for a specific channel value and inputs are as follows:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>0</code> <b> channels with any input:</b> Audio omitted from the output
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>1, 2, or auto </code> <b>channels with no audio input:</b> Audio omitted from the output
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>1 </code> <b>channel with any input with audio:</b> One track with one channel, downmixed if
     *         necessary
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>2 </code> <b>channels with one track with one channel:</b> One track with two identical channels
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>2 or auto </code> <b>channels with two tracks with one channel each:</b> One track with two
     *         channels
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>2 or auto </code> <b>channels with one track with two channels:</b> One track with two channels
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>2 </code> <b>channels with one track with multiple channels:</b> One track with two channels
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>auto </code> <b>channels with one track with one channel:</b> One track with one channel
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>auto </code> <b>channels with one track with multiple channels:</b> One track with multiple
     *         channels
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         When you specify <code>OneChannelPerTrack</code>, Elastic Transcoder creates a new track for every
     *         channel in your output. Your output can have up to eight single-channel tracks.
     *         </p>
     *         <p>
     *         The outputs of <code>OneChannelPerTrack</code> for a specific channel value and inputs are as follows:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>0 </code> <b>channels with any input:</b> Audio omitted from the output
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>1, 2, or auto </code> <b>channels with no audio input:</b> Audio omitted from the output
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>1 </code> <b>channel with any input with audio:</b> One track with one channel, downmixed if
     *         necessary
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>2 </code> <b>channels with one track with one channel:</b> Two tracks with one identical channel
     *         each
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>2 or auto </code> <b>channels with two tracks with one channel each:</b> Two tracks with one
     *         channel each
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>2 or auto </code> <b>channels with one track with two channels:</b> Two tracks with one channel
     *         each
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>2 </code> <b>channels with one track with multiple channels:</b> Two tracks with one channel each
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>auto </code> <b>channels with one track with one channel:</b> One track with one channel
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>auto </code> <b>channels with one track with multiple channels:</b> Up to eight tracks with one
     *         channel each
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         When you specify <code>OneChannelPerTrackWithMosTo8Tracks</code>, Elastic Transcoder creates eight
     *         single-channel tracks for your output. All tracks that do not contain audio data from an input channel
     *         are MOS, or Mit Out Sound, tracks.
     *         </p>
     *         <p>
     *         The outputs of <code>OneChannelPerTrackWithMosTo8Tracks</code> for a specific channel value and inputs
     *         are as follows:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>0 </code> <b>channels with any input:</b> Audio omitted from the output
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>1, 2, or auto </code> <b>channels with no audio input:</b> Audio omitted from the output
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>1 </code> <b>channel with any input with audio:</b> One track with one channel, downmixed if
     *         necessary, plus six MOS tracks
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>2 </code> <b>channels with one track with one channel:</b> Two tracks with one identical channel
     *         each, plus six MOS tracks
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>2 or auto </code> <b>channels with two tracks with one channel each:</b> Two tracks with one
     *         channel each, plus six MOS tracks
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>2 or auto </code> <b>channels with one track with two channels:</b> Two tracks with one channel
     *         each, plus six MOS tracks
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>2 </code> <b>channels with one track with multiple channels:</b> Two tracks with one channel each,
     *         plus six MOS tracks
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>auto </code> <b>channels with one track with one channel:</b> One track with one channel, plus
     *         seven MOS tracks
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>auto </code> <b>channels with one track with multiple channels:</b> Up to eight tracks with one
     *         channel each, plus MOS tracks until there are eight tracks in all
     *         </p>
     *         </li>
     */
    public String audioPackingMode() {
        return audioPackingMode;
    }

    /**
     * <p>
     * If you specified <code>AAC</code> for <code>Audio:Codec</code>, this is the <code>AAC</code> compression profile
     * to use. Valid values include:
     * </p>
     * <p>
     * <code>auto</code>, <code>AAC-LC</code>, <code>HE-AAC</code>, <code>HE-AACv2</code>
     * </p>
     * <p>
     * If you specify <code>auto</code>, Elastic Transcoder chooses a profile based on the bit rate of the output file.
     * </p>
     * 
     * @return If you specified <code>AAC</code> for <code>Audio:Codec</code>, this is the <code>AAC</code> compression
     *         profile to use. Valid values include:</p>
     *         <p>
     *         <code>auto</code>, <code>AAC-LC</code>, <code>HE-AAC</code>, <code>HE-AACv2</code>
     *         </p>
     *         <p>
     *         If you specify <code>auto</code>, Elastic Transcoder chooses a profile based on the bit rate of the
     *         output file.
     */
    public AudioCodecOptions codecOptions() {
        return codecOptions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(codec());
        hashCode = 31 * hashCode + Objects.hashCode(sampleRate());
        hashCode = 31 * hashCode + Objects.hashCode(bitRate());
        hashCode = 31 * hashCode + Objects.hashCode(channels());
        hashCode = 31 * hashCode + Objects.hashCode(audioPackingMode());
        hashCode = 31 * hashCode + Objects.hashCode(codecOptions());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AudioParameters)) {
            return false;
        }
        AudioParameters other = (AudioParameters) obj;
        return Objects.equals(codec(), other.codec()) && Objects.equals(sampleRate(), other.sampleRate())
                && Objects.equals(bitRate(), other.bitRate()) && Objects.equals(channels(), other.channels())
                && Objects.equals(audioPackingMode(), other.audioPackingMode())
                && Objects.equals(codecOptions(), other.codecOptions());
    }

    @Override
    public String toString() {
        return ToString.builder("AudioParameters").add("Codec", codec()).add("SampleRate", sampleRate())
                .add("BitRate", bitRate()).add("Channels", channels()).add("AudioPackingMode", audioPackingMode())
                .add("CodecOptions", codecOptions()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Codec":
            return Optional.ofNullable(clazz.cast(codec()));
        case "SampleRate":
            return Optional.ofNullable(clazz.cast(sampleRate()));
        case "BitRate":
            return Optional.ofNullable(clazz.cast(bitRate()));
        case "Channels":
            return Optional.ofNullable(clazz.cast(channels()));
        case "AudioPackingMode":
            return Optional.ofNullable(clazz.cast(audioPackingMode()));
        case "CodecOptions":
            return Optional.ofNullable(clazz.cast(codecOptions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<AudioParameters, T> g) {
        return obj -> g.apply((AudioParameters) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AudioParameters> {
        /**
         * <p>
         * The audio codec for the output file. Valid values include <code>aac</code>, <code>flac</code>,
         * <code>mp2</code>, <code>mp3</code>, <code>pcm</code>, and <code>vorbis</code>.
         * </p>
         * 
         * @param codec
         *        The audio codec for the output file. Valid values include <code>aac</code>, <code>flac</code>,
         *        <code>mp2</code>, <code>mp3</code>, <code>pcm</code>, and <code>vorbis</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder codec(String codec);

        /**
         * <p>
         * The sample rate of the audio stream in the output file, in Hertz. Valid values include:
         * </p>
         * <p>
         * <code>auto</code>, <code>22050</code>, <code>32000</code>, <code>44100</code>, <code>48000</code>,
         * <code>96000</code>
         * </p>
         * <p>
         * If you specify <code>auto</code>, Elastic Transcoder automatically detects the sample rate.
         * </p>
         * 
         * @param sampleRate
         *        The sample rate of the audio stream in the output file, in Hertz. Valid values include:</p>
         *        <p>
         *        <code>auto</code>, <code>22050</code>, <code>32000</code>, <code>44100</code>, <code>48000</code>,
         *        <code>96000</code>
         *        </p>
         *        <p>
         *        If you specify <code>auto</code>, Elastic Transcoder automatically detects the sample rate.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sampleRate(String sampleRate);

        /**
         * <p>
         * The bit rate of the audio stream in the output file, in kilobits/second. Enter an integer between 64 and 320,
         * inclusive.
         * </p>
         * 
         * @param bitRate
         *        The bit rate of the audio stream in the output file, in kilobits/second. Enter an integer between 64
         *        and 320, inclusive.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bitRate(String bitRate);

        /**
         * <p>
         * The number of audio channels in the output file. The following values are valid:
         * </p>
         * <p>
         * <code>auto</code>, <code>0</code>, <code>1</code>, <code>2</code>
         * </p>
         * <p>
         * One channel carries the information played by a single speaker. For example, a stereo track with two channels
         * sends one channel to the left speaker, and the other channel to the right speaker. The output channels are
         * organized into tracks. If you want Elastic Transcoder to automatically detect the number of audio channels in
         * the input file and use that value for the output file, select <code>auto</code>.
         * </p>
         * <p>
         * The output of a specific channel value and inputs are as follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>auto</code> <b> channel specified, with any input:</b> Pass through up to eight input channels.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>0</code> <b> channels specified, with any input:</b> Audio omitted from the output.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>1</code> <b> channel specified, with at least one input channel:</b> Mono sound.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>2</code> <b> channels specified, with any input:</b> Two identical mono channels or stereo. For more
         * information about tracks, see <code>Audio:AudioPackingMode.</code>
         * </p>
         * </li>
         * </ul>
         * <p>
         * For more information about how Elastic Transcoder organizes channels and tracks, see
         * <code>Audio:AudioPackingMode</code>.
         * </p>
         * 
         * @param channels
         *        The number of audio channels in the output file. The following values are valid:</p>
         *        <p>
         *        <code>auto</code>, <code>0</code>, <code>1</code>, <code>2</code>
         *        </p>
         *        <p>
         *        One channel carries the information played by a single speaker. For example, a stereo track with two
         *        channels sends one channel to the left speaker, and the other channel to the right speaker. The output
         *        channels are organized into tracks. If you want Elastic Transcoder to automatically detect the number
         *        of audio channels in the input file and use that value for the output file, select <code>auto</code>.
         *        </p>
         *        <p>
         *        The output of a specific channel value and inputs are as follows:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>auto</code> <b> channel specified, with any input:</b> Pass through up to eight input channels.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>0</code> <b> channels specified, with any input:</b> Audio omitted from the output.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>1</code> <b> channel specified, with at least one input channel:</b> Mono sound.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>2</code> <b> channels specified, with any input:</b> Two identical mono channels or stereo. For
         *        more information about tracks, see <code>Audio:AudioPackingMode.</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For more information about how Elastic Transcoder organizes channels and tracks, see
         *        <code>Audio:AudioPackingMode</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder channels(String channels);

        /**
         * <p>
         * The method of organizing audio channels and tracks. Use <code>Audio:Channels</code> to specify the number of
         * channels in your output, and <code>Audio:AudioPackingMode</code> to specify the number of tracks and their
         * relation to the channels. If you do not specify an <code>Audio:AudioPackingMode</code>, Elastic Transcoder
         * uses <code>SingleTrack</code>.
         * </p>
         * <p>
         * The following values are valid:
         * </p>
         * <p>
         * <code>SingleTrack</code>, <code>OneChannelPerTrack</code>, and
         * <code>OneChannelPerTrackWithMosTo8Tracks</code>
         * </p>
         * <p>
         * When you specify <code>SingleTrack</code>, Elastic Transcoder creates a single track for your output. The
         * track can have up to eight channels. Use <code>SingleTrack</code> for all non-<code>mxf</code> containers.
         * </p>
         * <p>
         * The outputs of <code>SingleTrack</code> for a specific channel value and inputs are as follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>0</code> <b> channels with any input:</b> Audio omitted from the output
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>1, 2, or auto </code> <b>channels with no audio input:</b> Audio omitted from the output
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>1 </code> <b>channel with any input with audio:</b> One track with one channel, downmixed if necessary
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>2 </code> <b>channels with one track with one channel:</b> One track with two identical channels
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>2 or auto </code> <b>channels with two tracks with one channel each:</b> One track with two channels
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>2 or auto </code> <b>channels with one track with two channels:</b> One track with two channels
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>2 </code> <b>channels with one track with multiple channels:</b> One track with two channels
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>auto </code> <b>channels with one track with one channel:</b> One track with one channel
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>auto </code> <b>channels with one track with multiple channels:</b> One track with multiple channels
         * </p>
         * </li>
         * </ul>
         * <p>
         * When you specify <code>OneChannelPerTrack</code>, Elastic Transcoder creates a new track for every channel in
         * your output. Your output can have up to eight single-channel tracks.
         * </p>
         * <p>
         * The outputs of <code>OneChannelPerTrack</code> for a specific channel value and inputs are as follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>0 </code> <b>channels with any input:</b> Audio omitted from the output
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>1, 2, or auto </code> <b>channels with no audio input:</b> Audio omitted from the output
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>1 </code> <b>channel with any input with audio:</b> One track with one channel, downmixed if necessary
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>2 </code> <b>channels with one track with one channel:</b> Two tracks with one identical channel each
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>2 or auto </code> <b>channels with two tracks with one channel each:</b> Two tracks with one channel
         * each
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>2 or auto </code> <b>channels with one track with two channels:</b> Two tracks with one channel each
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>2 </code> <b>channels with one track with multiple channels:</b> Two tracks with one channel each
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>auto </code> <b>channels with one track with one channel:</b> One track with one channel
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>auto </code> <b>channels with one track with multiple channels:</b> Up to eight tracks with one channel
         * each
         * </p>
         * </li>
         * </ul>
         * <p>
         * When you specify <code>OneChannelPerTrackWithMosTo8Tracks</code>, Elastic Transcoder creates eight
         * single-channel tracks for your output. All tracks that do not contain audio data from an input channel are
         * MOS, or Mit Out Sound, tracks.
         * </p>
         * <p>
         * The outputs of <code>OneChannelPerTrackWithMosTo8Tracks</code> for a specific channel value and inputs are as
         * follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>0 </code> <b>channels with any input:</b> Audio omitted from the output
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>1, 2, or auto </code> <b>channels with no audio input:</b> Audio omitted from the output
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>1 </code> <b>channel with any input with audio:</b> One track with one channel, downmixed if necessary,
         * plus six MOS tracks
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>2 </code> <b>channels with one track with one channel:</b> Two tracks with one identical channel each,
         * plus six MOS tracks
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>2 or auto </code> <b>channels with two tracks with one channel each:</b> Two tracks with one channel
         * each, plus six MOS tracks
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>2 or auto </code> <b>channels with one track with two channels:</b> Two tracks with one channel each,
         * plus six MOS tracks
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>2 </code> <b>channels with one track with multiple channels:</b> Two tracks with one channel each, plus
         * six MOS tracks
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>auto </code> <b>channels with one track with one channel:</b> One track with one channel, plus seven
         * MOS tracks
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>auto </code> <b>channels with one track with multiple channels:</b> Up to eight tracks with one channel
         * each, plus MOS tracks until there are eight tracks in all
         * </p>
         * </li>
         * </ul>
         * 
         * @param audioPackingMode
         *        The method of organizing audio channels and tracks. Use <code>Audio:Channels</code> to specify the
         *        number of channels in your output, and <code>Audio:AudioPackingMode</code> to specify the number of
         *        tracks and their relation to the channels. If you do not specify an
         *        <code>Audio:AudioPackingMode</code>, Elastic Transcoder uses <code>SingleTrack</code>.</p>
         *        <p>
         *        The following values are valid:
         *        </p>
         *        <p>
         *        <code>SingleTrack</code>, <code>OneChannelPerTrack</code>, and
         *        <code>OneChannelPerTrackWithMosTo8Tracks</code>
         *        </p>
         *        <p>
         *        When you specify <code>SingleTrack</code>, Elastic Transcoder creates a single track for your output.
         *        The track can have up to eight channels. Use <code>SingleTrack</code> for all non-<code>mxf</code>
         *        containers.
         *        </p>
         *        <p>
         *        The outputs of <code>SingleTrack</code> for a specific channel value and inputs are as follows:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>0</code> <b> channels with any input:</b> Audio omitted from the output
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>1, 2, or auto </code> <b>channels with no audio input:</b> Audio omitted from the output
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>1 </code> <b>channel with any input with audio:</b> One track with one channel, downmixed if
         *        necessary
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>2 </code> <b>channels with one track with one channel:</b> One track with two identical channels
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>2 or auto </code> <b>channels with two tracks with one channel each:</b> One track with two
         *        channels
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>2 or auto </code> <b>channels with one track with two channels:</b> One track with two channels
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>2 </code> <b>channels with one track with multiple channels:</b> One track with two channels
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>auto </code> <b>channels with one track with one channel:</b> One track with one channel
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>auto </code> <b>channels with one track with multiple channels:</b> One track with multiple
         *        channels
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        When you specify <code>OneChannelPerTrack</code>, Elastic Transcoder creates a new track for every
         *        channel in your output. Your output can have up to eight single-channel tracks.
         *        </p>
         *        <p>
         *        The outputs of <code>OneChannelPerTrack</code> for a specific channel value and inputs are as follows:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>0 </code> <b>channels with any input:</b> Audio omitted from the output
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>1, 2, or auto </code> <b>channels with no audio input:</b> Audio omitted from the output
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>1 </code> <b>channel with any input with audio:</b> One track with one channel, downmixed if
         *        necessary
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>2 </code> <b>channels with one track with one channel:</b> Two tracks with one identical channel
         *        each
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>2 or auto </code> <b>channels with two tracks with one channel each:</b> Two tracks with one
         *        channel each
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>2 or auto </code> <b>channels with one track with two channels:</b> Two tracks with one channel
         *        each
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>2 </code> <b>channels with one track with multiple channels:</b> Two tracks with one channel
         *        each
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>auto </code> <b>channels with one track with one channel:</b> One track with one channel
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>auto </code> <b>channels with one track with multiple channels:</b> Up to eight tracks with one
         *        channel each
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        When you specify <code>OneChannelPerTrackWithMosTo8Tracks</code>, Elastic Transcoder creates eight
         *        single-channel tracks for your output. All tracks that do not contain audio data from an input channel
         *        are MOS, or Mit Out Sound, tracks.
         *        </p>
         *        <p>
         *        The outputs of <code>OneChannelPerTrackWithMosTo8Tracks</code> for a specific channel value and inputs
         *        are as follows:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>0 </code> <b>channels with any input:</b> Audio omitted from the output
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>1, 2, or auto </code> <b>channels with no audio input:</b> Audio omitted from the output
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>1 </code> <b>channel with any input with audio:</b> One track with one channel, downmixed if
         *        necessary, plus six MOS tracks
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>2 </code> <b>channels with one track with one channel:</b> Two tracks with one identical channel
         *        each, plus six MOS tracks
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>2 or auto </code> <b>channels with two tracks with one channel each:</b> Two tracks with one
         *        channel each, plus six MOS tracks
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>2 or auto </code> <b>channels with one track with two channels:</b> Two tracks with one channel
         *        each, plus six MOS tracks
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>2 </code> <b>channels with one track with multiple channels:</b> Two tracks with one channel
         *        each, plus six MOS tracks
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>auto </code> <b>channels with one track with one channel:</b> One track with one channel, plus
         *        seven MOS tracks
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>auto </code> <b>channels with one track with multiple channels:</b> Up to eight tracks with one
         *        channel each, plus MOS tracks until there are eight tracks in all
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder audioPackingMode(String audioPackingMode);

        /**
         * <p>
         * If you specified <code>AAC</code> for <code>Audio:Codec</code>, this is the <code>AAC</code> compression
         * profile to use. Valid values include:
         * </p>
         * <p>
         * <code>auto</code>, <code>AAC-LC</code>, <code>HE-AAC</code>, <code>HE-AACv2</code>
         * </p>
         * <p>
         * If you specify <code>auto</code>, Elastic Transcoder chooses a profile based on the bit rate of the output
         * file.
         * </p>
         * 
         * @param codecOptions
         *        If you specified <code>AAC</code> for <code>Audio:Codec</code>, this is the <code>AAC</code>
         *        compression profile to use. Valid values include:</p>
         *        <p>
         *        <code>auto</code>, <code>AAC-LC</code>, <code>HE-AAC</code>, <code>HE-AACv2</code>
         *        </p>
         *        <p>
         *        If you specify <code>auto</code>, Elastic Transcoder chooses a profile based on the bit rate of the
         *        output file.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder codecOptions(AudioCodecOptions codecOptions);

        /**
         * <p>
         * If you specified <code>AAC</code> for <code>Audio:Codec</code>, this is the <code>AAC</code> compression
         * profile to use. Valid values include:
         * </p>
         * <p>
         * <code>auto</code>, <code>AAC-LC</code>, <code>HE-AAC</code>, <code>HE-AACv2</code>
         * </p>
         * <p>
         * If you specify <code>auto</code>, Elastic Transcoder chooses a profile based on the bit rate of the output
         * file.
         * </p>
         * This is a convenience that creates an instance of the {@link AudioCodecOptions.Builder} avoiding the need to
         * create one manually via {@link AudioCodecOptions#builder()}.
         *
         * When the {@link Consumer} completes, {@link AudioCodecOptions.Builder#build()} is called immediately and its
         * result is passed to {@link #codecOptions(AudioCodecOptions)}.
         * 
         * @param codecOptions
         *        a consumer that will call methods on {@link AudioCodecOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #codecOptions(AudioCodecOptions)
         */
        default Builder codecOptions(Consumer<AudioCodecOptions.Builder> codecOptions) {
            return codecOptions(AudioCodecOptions.builder().applyMutation(codecOptions).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String codec;

        private String sampleRate;

        private String bitRate;

        private String channels;

        private String audioPackingMode;

        private AudioCodecOptions codecOptions;

        private BuilderImpl() {
        }

        private BuilderImpl(AudioParameters model) {
            codec(model.codec);
            sampleRate(model.sampleRate);
            bitRate(model.bitRate);
            channels(model.channels);
            audioPackingMode(model.audioPackingMode);
            codecOptions(model.codecOptions);
        }

        public final String getCodec() {
            return codec;
        }

        @Override
        public final Builder codec(String codec) {
            this.codec = codec;
            return this;
        }

        public final void setCodec(String codec) {
            this.codec = codec;
        }

        public final String getSampleRate() {
            return sampleRate;
        }

        @Override
        public final Builder sampleRate(String sampleRate) {
            this.sampleRate = sampleRate;
            return this;
        }

        public final void setSampleRate(String sampleRate) {
            this.sampleRate = sampleRate;
        }

        public final String getBitRate() {
            return bitRate;
        }

        @Override
        public final Builder bitRate(String bitRate) {
            this.bitRate = bitRate;
            return this;
        }

        public final void setBitRate(String bitRate) {
            this.bitRate = bitRate;
        }

        public final String getChannels() {
            return channels;
        }

        @Override
        public final Builder channels(String channels) {
            this.channels = channels;
            return this;
        }

        public final void setChannels(String channels) {
            this.channels = channels;
        }

        public final String getAudioPackingMode() {
            return audioPackingMode;
        }

        @Override
        public final Builder audioPackingMode(String audioPackingMode) {
            this.audioPackingMode = audioPackingMode;
            return this;
        }

        public final void setAudioPackingMode(String audioPackingMode) {
            this.audioPackingMode = audioPackingMode;
        }

        public final AudioCodecOptions.Builder getCodecOptions() {
            return codecOptions != null ? codecOptions.toBuilder() : null;
        }

        @Override
        public final Builder codecOptions(AudioCodecOptions codecOptions) {
            this.codecOptions = codecOptions;
            return this;
        }

        public final void setCodecOptions(AudioCodecOptions.BuilderImpl codecOptions) {
            this.codecOptions = codecOptions != null ? codecOptions.build() : null;
        }

        @Override
        public AudioParameters build() {
            return new AudioParameters(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
