/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.elastictranscoder.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The .jpg or .png file associated with an audio file.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class JobAlbumArt implements SdkPojo, Serializable, ToCopyableBuilder<JobAlbumArt.Builder, JobAlbumArt> {
    private static final SdkField<String> MERGE_POLICY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(JobAlbumArt::mergePolicy)).setter(setter(Builder::mergePolicy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MergePolicy").build()).build();

    private static final SdkField<List<Artwork>> ARTWORK_FIELD = SdkField
            .<List<Artwork>> builder(MarshallingType.LIST)
            .getter(getter(JobAlbumArt::artwork))
            .setter(setter(Builder::artwork))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Artwork").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Artwork> builder(MarshallingType.SDK_POJO)
                                            .constructor(Artwork::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(MERGE_POLICY_FIELD,
            ARTWORK_FIELD));

    private static final long serialVersionUID = 1L;

    private final String mergePolicy;

    private final List<Artwork> artwork;

    private JobAlbumArt(BuilderImpl builder) {
        this.mergePolicy = builder.mergePolicy;
        this.artwork = builder.artwork;
    }

    /**
     * <p>
     * A policy that determines how Elastic Transcoder handles the existence of multiple album artwork files.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Replace:</code> The specified album art replaces any existing album art.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Prepend:</code> The specified album art is placed in front of any existing album art.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Append:</code> The specified album art is placed after any existing album art.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Fallback:</code> If the original input file contains artwork, Elastic Transcoder uses that artwork for the
     * output. If the original input does not contain artwork, Elastic Transcoder uses the specified album art file.
     * </p>
     * </li>
     * </ul>
     * 
     * @return A policy that determines how Elastic Transcoder handles the existence of multiple album artwork
     *         files.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Replace:</code> The specified album art replaces any existing album art.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Prepend:</code> The specified album art is placed in front of any existing album art.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Append:</code> The specified album art is placed after any existing album art.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Fallback:</code> If the original input file contains artwork, Elastic Transcoder uses that artwork
     *         for the output. If the original input does not contain artwork, Elastic Transcoder uses the specified
     *         album art file.
     *         </p>
     *         </li>
     */
    public String mergePolicy() {
        return mergePolicy;
    }

    /**
     * <p>
     * The file to be used as album art. There can be multiple artworks associated with an audio file, to a maximum of
     * 20. Valid formats are <code>.jpg</code> and <code>.png</code>
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The file to be used as album art. There can be multiple artworks associated with an audio file, to a
     *         maximum of 20. Valid formats are <code>.jpg</code> and <code>.png</code>
     */
    public List<Artwork> artwork() {
        return artwork;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(mergePolicy());
        hashCode = 31 * hashCode + Objects.hashCode(artwork());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof JobAlbumArt)) {
            return false;
        }
        JobAlbumArt other = (JobAlbumArt) obj;
        return Objects.equals(mergePolicy(), other.mergePolicy()) && Objects.equals(artwork(), other.artwork());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("JobAlbumArt").add("MergePolicy", mergePolicy()).add("Artwork", artwork()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "MergePolicy":
            return Optional.ofNullable(clazz.cast(mergePolicy()));
        case "Artwork":
            return Optional.ofNullable(clazz.cast(artwork()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<JobAlbumArt, T> g) {
        return obj -> g.apply((JobAlbumArt) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, JobAlbumArt> {
        /**
         * <p>
         * A policy that determines how Elastic Transcoder handles the existence of multiple album artwork files.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Replace:</code> The specified album art replaces any existing album art.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Prepend:</code> The specified album art is placed in front of any existing album art.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Append:</code> The specified album art is placed after any existing album art.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Fallback:</code> If the original input file contains artwork, Elastic Transcoder uses that artwork for
         * the output. If the original input does not contain artwork, Elastic Transcoder uses the specified album art
         * file.
         * </p>
         * </li>
         * </ul>
         * 
         * @param mergePolicy
         *        A policy that determines how Elastic Transcoder handles the existence of multiple album artwork
         *        files.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Replace:</code> The specified album art replaces any existing album art.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Prepend:</code> The specified album art is placed in front of any existing album art.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Append:</code> The specified album art is placed after any existing album art.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Fallback:</code> If the original input file contains artwork, Elastic Transcoder uses that
         *        artwork for the output. If the original input does not contain artwork, Elastic Transcoder uses the
         *        specified album art file.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder mergePolicy(String mergePolicy);

        /**
         * <p>
         * The file to be used as album art. There can be multiple artworks associated with an audio file, to a maximum
         * of 20. Valid formats are <code>.jpg</code> and <code>.png</code>
         * </p>
         * 
         * @param artwork
         *        The file to be used as album art. There can be multiple artworks associated with an audio file, to a
         *        maximum of 20. Valid formats are <code>.jpg</code> and <code>.png</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder artwork(Collection<Artwork> artwork);

        /**
         * <p>
         * The file to be used as album art. There can be multiple artworks associated with an audio file, to a maximum
         * of 20. Valid formats are <code>.jpg</code> and <code>.png</code>
         * </p>
         * 
         * @param artwork
         *        The file to be used as album art. There can be multiple artworks associated with an audio file, to a
         *        maximum of 20. Valid formats are <code>.jpg</code> and <code>.png</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder artwork(Artwork... artwork);

        /**
         * <p>
         * The file to be used as album art. There can be multiple artworks associated with an audio file, to a maximum
         * of 20. Valid formats are <code>.jpg</code> and <code>.png</code>
         * </p>
         * This is a convenience that creates an instance of the {@link List<Artwork>.Builder} avoiding the need to
         * create one manually via {@link List<Artwork>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Artwork>.Builder#build()} is called immediately and its
         * result is passed to {@link #artwork(List<Artwork>)}.
         * 
         * @param artwork
         *        a consumer that will call methods on {@link List<Artwork>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #artwork(List<Artwork>)
         */
        Builder artwork(Consumer<Artwork.Builder>... artwork);
    }

    static final class BuilderImpl implements Builder {
        private String mergePolicy;

        private List<Artwork> artwork = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(JobAlbumArt model) {
            mergePolicy(model.mergePolicy);
            artwork(model.artwork);
        }

        public final String getMergePolicy() {
            return mergePolicy;
        }

        @Override
        public final Builder mergePolicy(String mergePolicy) {
            this.mergePolicy = mergePolicy;
            return this;
        }

        public final void setMergePolicy(String mergePolicy) {
            this.mergePolicy = mergePolicy;
        }

        public final Collection<Artwork.Builder> getArtwork() {
            return artwork != null ? artwork.stream().map(Artwork::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder artwork(Collection<Artwork> artwork) {
            this.artwork = ArtworksCopier.copy(artwork);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder artwork(Artwork... artwork) {
            artwork(Arrays.asList(artwork));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder artwork(Consumer<Artwork.Builder>... artwork) {
            artwork(Stream.of(artwork).map(c -> Artwork.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setArtwork(Collection<Artwork.BuilderImpl> artwork) {
            this.artwork = ArtworksCopier.copyFromBuilder(artwork);
        }

        @Override
        public JobAlbumArt build() {
            return new JobAlbumArt(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
