/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.elastictranscoder;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.elastictranscoder.model.AccessDeniedException;
import software.amazon.awssdk.services.elastictranscoder.model.CancelJobRequest;
import software.amazon.awssdk.services.elastictranscoder.model.CancelJobResponse;
import software.amazon.awssdk.services.elastictranscoder.model.CreateJobRequest;
import software.amazon.awssdk.services.elastictranscoder.model.CreateJobResponse;
import software.amazon.awssdk.services.elastictranscoder.model.CreatePipelineRequest;
import software.amazon.awssdk.services.elastictranscoder.model.CreatePipelineResponse;
import software.amazon.awssdk.services.elastictranscoder.model.CreatePresetRequest;
import software.amazon.awssdk.services.elastictranscoder.model.CreatePresetResponse;
import software.amazon.awssdk.services.elastictranscoder.model.DeletePipelineRequest;
import software.amazon.awssdk.services.elastictranscoder.model.DeletePipelineResponse;
import software.amazon.awssdk.services.elastictranscoder.model.DeletePresetRequest;
import software.amazon.awssdk.services.elastictranscoder.model.DeletePresetResponse;
import software.amazon.awssdk.services.elastictranscoder.model.ElasticTranscoderException;
import software.amazon.awssdk.services.elastictranscoder.model.ElasticTranscoderRequest;
import software.amazon.awssdk.services.elastictranscoder.model.IncompatibleVersionException;
import software.amazon.awssdk.services.elastictranscoder.model.InternalServiceException;
import software.amazon.awssdk.services.elastictranscoder.model.LimitExceededException;
import software.amazon.awssdk.services.elastictranscoder.model.ListJobsByPipelineRequest;
import software.amazon.awssdk.services.elastictranscoder.model.ListJobsByPipelineResponse;
import software.amazon.awssdk.services.elastictranscoder.model.ListJobsByStatusRequest;
import software.amazon.awssdk.services.elastictranscoder.model.ListJobsByStatusResponse;
import software.amazon.awssdk.services.elastictranscoder.model.ListPipelinesRequest;
import software.amazon.awssdk.services.elastictranscoder.model.ListPipelinesResponse;
import software.amazon.awssdk.services.elastictranscoder.model.ListPresetsRequest;
import software.amazon.awssdk.services.elastictranscoder.model.ListPresetsResponse;
import software.amazon.awssdk.services.elastictranscoder.model.ReadJobRequest;
import software.amazon.awssdk.services.elastictranscoder.model.ReadJobResponse;
import software.amazon.awssdk.services.elastictranscoder.model.ReadPipelineRequest;
import software.amazon.awssdk.services.elastictranscoder.model.ReadPipelineResponse;
import software.amazon.awssdk.services.elastictranscoder.model.ReadPresetRequest;
import software.amazon.awssdk.services.elastictranscoder.model.ReadPresetResponse;
import software.amazon.awssdk.services.elastictranscoder.model.ResourceInUseException;
import software.amazon.awssdk.services.elastictranscoder.model.ResourceNotFoundException;
import software.amazon.awssdk.services.elastictranscoder.model.UpdatePipelineNotificationsRequest;
import software.amazon.awssdk.services.elastictranscoder.model.UpdatePipelineNotificationsResponse;
import software.amazon.awssdk.services.elastictranscoder.model.UpdatePipelineRequest;
import software.amazon.awssdk.services.elastictranscoder.model.UpdatePipelineResponse;
import software.amazon.awssdk.services.elastictranscoder.model.UpdatePipelineStatusRequest;
import software.amazon.awssdk.services.elastictranscoder.model.UpdatePipelineStatusResponse;
import software.amazon.awssdk.services.elastictranscoder.model.ValidationException;
import software.amazon.awssdk.services.elastictranscoder.paginators.ListJobsByPipelinePublisher;
import software.amazon.awssdk.services.elastictranscoder.paginators.ListJobsByStatusPublisher;
import software.amazon.awssdk.services.elastictranscoder.paginators.ListPipelinesPublisher;
import software.amazon.awssdk.services.elastictranscoder.paginators.ListPresetsPublisher;
import software.amazon.awssdk.services.elastictranscoder.transform.CancelJobRequestMarshaller;
import software.amazon.awssdk.services.elastictranscoder.transform.CreateJobRequestMarshaller;
import software.amazon.awssdk.services.elastictranscoder.transform.CreatePipelineRequestMarshaller;
import software.amazon.awssdk.services.elastictranscoder.transform.CreatePresetRequestMarshaller;
import software.amazon.awssdk.services.elastictranscoder.transform.DeletePipelineRequestMarshaller;
import software.amazon.awssdk.services.elastictranscoder.transform.DeletePresetRequestMarshaller;
import software.amazon.awssdk.services.elastictranscoder.transform.ListJobsByPipelineRequestMarshaller;
import software.amazon.awssdk.services.elastictranscoder.transform.ListJobsByStatusRequestMarshaller;
import software.amazon.awssdk.services.elastictranscoder.transform.ListPipelinesRequestMarshaller;
import software.amazon.awssdk.services.elastictranscoder.transform.ListPresetsRequestMarshaller;
import software.amazon.awssdk.services.elastictranscoder.transform.ReadJobRequestMarshaller;
import software.amazon.awssdk.services.elastictranscoder.transform.ReadPipelineRequestMarshaller;
import software.amazon.awssdk.services.elastictranscoder.transform.ReadPresetRequestMarshaller;
import software.amazon.awssdk.services.elastictranscoder.transform.UpdatePipelineNotificationsRequestMarshaller;
import software.amazon.awssdk.services.elastictranscoder.transform.UpdatePipelineRequestMarshaller;
import software.amazon.awssdk.services.elastictranscoder.transform.UpdatePipelineStatusRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link ElasticTranscoderAsyncClient}.
 *
 * @see ElasticTranscoderAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultElasticTranscoderAsyncClient implements ElasticTranscoderAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultElasticTranscoderAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultElasticTranscoderAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * The CancelJob operation cancels an unfinished job.
     * </p>
     * <note>
     * <p>
     * You can only cancel a job that has a status of <code>Submitted</code>. To prevent a pipeline from starting to
     * process a job while you're getting the job identifier, use <a>UpdatePipelineStatus</a> to temporarily pause the
     * pipeline.
     * </p>
     * </note>
     *
     * @param cancelJobRequest
     *        The <code>CancelJobRequest</code> structure.
     * @return A Java Future containing the result of the CancelJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One or more required parameter values were not provided in the request.</li>
     *         <li>IncompatibleVersionException</li>
     *         <li>ResourceNotFoundException The requested resource does not exist or is not available. For example, the
     *         pipeline to which you're trying to add a job doesn't exist or is still being created.</li>
     *         <li>ResourceInUseException The resource you are attempting to change is in use. For example, you are
     *         attempting to delete a pipeline that is currently in use.</li>
     *         <li>AccessDeniedException General authentication failure. The request was not signed correctly.</li>
     *         <li>InternalServiceException Elastic Transcoder encountered an unexpected exception while trying to
     *         fulfill the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ElasticTranscoderException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ElasticTranscoderAsyncClient.CancelJob
     */
    @Override
    public CompletableFuture<CancelJobResponse> cancelJob(CancelJobRequest cancelJobRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CancelJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CancelJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CancelJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CancelJobRequest, CancelJobResponse>().withOperationName("CancelJob")
                            .withMarshaller(new CancelJobRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(cancelJobRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * When you create a job, Elastic Transcoder returns JSON data that includes the values that you specified plus
     * information about the job that is created.
     * </p>
     * <p>
     * If you have specified more than one output for your jobs (for example, one output for the Kindle Fire and another
     * output for the Apple iPhone 4s), you currently must use the Elastic Transcoder API to list the jobs (as opposed
     * to the AWS Console).
     * </p>
     *
     * @param createJobRequest
     *        The <code>CreateJobRequest</code> structure.
     * @return A Java Future containing the result of the CreateJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One or more required parameter values were not provided in the request.</li>
     *         <li>IncompatibleVersionException</li>
     *         <li>ResourceNotFoundException The requested resource does not exist or is not available. For example, the
     *         pipeline to which you're trying to add a job doesn't exist or is still being created.</li>
     *         <li>AccessDeniedException General authentication failure. The request was not signed correctly.</li>
     *         <li>LimitExceededException Too many operations for a given AWS account. For example, the number of
     *         pipelines exceeds the maximum allowed.</li>
     *         <li>InternalServiceException Elastic Transcoder encountered an unexpected exception while trying to
     *         fulfill the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ElasticTranscoderException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ElasticTranscoderAsyncClient.CreateJob
     */
    @Override
    public CompletableFuture<CreateJobResponse> createJob(CreateJobRequest createJobRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateJobRequest, CreateJobResponse>().withOperationName("CreateJob")
                            .withMarshaller(new CreateJobRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(createJobRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The CreatePipeline operation creates a pipeline with settings that you specify.
     * </p>
     *
     * @param createPipelineRequest
     *        The <code>CreatePipelineRequest</code> structure.
     * @return A Java Future containing the result of the CreatePipeline operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One or more required parameter values were not provided in the request.</li>
     *         <li>IncompatibleVersionException</li>
     *         <li>AccessDeniedException General authentication failure. The request was not signed correctly.</li>
     *         <li>ResourceNotFoundException The requested resource does not exist or is not available. For example, the
     *         pipeline to which you're trying to add a job doesn't exist or is still being created.</li>
     *         <li>LimitExceededException Too many operations for a given AWS account. For example, the number of
     *         pipelines exceeds the maximum allowed.</li>
     *         <li>InternalServiceException Elastic Transcoder encountered an unexpected exception while trying to
     *         fulfill the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ElasticTranscoderException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ElasticTranscoderAsyncClient.CreatePipeline
     */
    @Override
    public CompletableFuture<CreatePipelineResponse> createPipeline(CreatePipelineRequest createPipelineRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreatePipelineResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreatePipelineResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreatePipelineResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreatePipelineRequest, CreatePipelineResponse>()
                            .withOperationName("CreatePipeline")
                            .withMarshaller(new CreatePipelineRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createPipelineRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The CreatePreset operation creates a preset with settings that you specify.
     * </p>
     * <important>
     * <p>
     * Elastic Transcoder checks the CreatePreset settings to ensure that they meet Elastic Transcoder requirements and
     * to determine whether they comply with H.264 standards. If your settings are not valid for Elastic Transcoder,
     * Elastic Transcoder returns an HTTP 400 response (<code>ValidationException</code>) and does not create the
     * preset. If the settings are valid for Elastic Transcoder but aren't strictly compliant with the H.264 standard,
     * Elastic Transcoder creates the preset and returns a warning message in the response. This helps you determine
     * whether your settings comply with the H.264 standard while giving you greater flexibility with respect to the
     * video that Elastic Transcoder produces.
     * </p>
     * </important>
     * <p>
     * Elastic Transcoder uses the H.264 video-compression format. For more information, see the International
     * Telecommunication Union publication <i>Recommendation ITU-T H.264: Advanced video coding for generic audiovisual
     * services</i>.
     * </p>
     *
     * @param createPresetRequest
     *        The <code>CreatePresetRequest</code> structure.
     * @return A Java Future containing the result of the CreatePreset operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One or more required parameter values were not provided in the request.</li>
     *         <li>IncompatibleVersionException</li>
     *         <li>AccessDeniedException General authentication failure. The request was not signed correctly.</li>
     *         <li>LimitExceededException Too many operations for a given AWS account. For example, the number of
     *         pipelines exceeds the maximum allowed.</li>
     *         <li>InternalServiceException Elastic Transcoder encountered an unexpected exception while trying to
     *         fulfill the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ElasticTranscoderException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ElasticTranscoderAsyncClient.CreatePreset
     */
    @Override
    public CompletableFuture<CreatePresetResponse> createPreset(CreatePresetRequest createPresetRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreatePresetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreatePresetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreatePresetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreatePresetRequest, CreatePresetResponse>()
                            .withOperationName("CreatePreset").withMarshaller(new CreatePresetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createPresetRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The DeletePipeline operation removes a pipeline.
     * </p>
     * <p>
     * You can only delete a pipeline that has never been used or that is not currently in use (doesn't contain any
     * active jobs). If the pipeline is currently in use, <code>DeletePipeline</code> returns an error.
     * </p>
     *
     * @param deletePipelineRequest
     *        The <code>DeletePipelineRequest</code> structure.
     * @return A Java Future containing the result of the DeletePipeline operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One or more required parameter values were not provided in the request.</li>
     *         <li>IncompatibleVersionException</li>
     *         <li>ResourceNotFoundException The requested resource does not exist or is not available. For example, the
     *         pipeline to which you're trying to add a job doesn't exist or is still being created.</li>
     *         <li>ResourceInUseException The resource you are attempting to change is in use. For example, you are
     *         attempting to delete a pipeline that is currently in use.</li>
     *         <li>AccessDeniedException General authentication failure. The request was not signed correctly.</li>
     *         <li>InternalServiceException Elastic Transcoder encountered an unexpected exception while trying to
     *         fulfill the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ElasticTranscoderException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ElasticTranscoderAsyncClient.DeletePipeline
     */
    @Override
    public CompletableFuture<DeletePipelineResponse> deletePipeline(DeletePipelineRequest deletePipelineRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeletePipelineResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeletePipelineResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeletePipelineResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeletePipelineRequest, DeletePipelineResponse>()
                            .withOperationName("DeletePipeline")
                            .withMarshaller(new DeletePipelineRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deletePipelineRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The DeletePreset operation removes a preset that you've added in an AWS region.
     * </p>
     * <note>
     * <p>
     * You can't delete the default presets that are included with Elastic Transcoder.
     * </p>
     * </note>
     *
     * @param deletePresetRequest
     *        The <code>DeletePresetRequest</code> structure.
     * @return A Java Future containing the result of the DeletePreset operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One or more required parameter values were not provided in the request.</li>
     *         <li>IncompatibleVersionException</li>
     *         <li>ResourceNotFoundException The requested resource does not exist or is not available. For example, the
     *         pipeline to which you're trying to add a job doesn't exist or is still being created.</li>
     *         <li>AccessDeniedException General authentication failure. The request was not signed correctly.</li>
     *         <li>InternalServiceException Elastic Transcoder encountered an unexpected exception while trying to
     *         fulfill the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ElasticTranscoderException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ElasticTranscoderAsyncClient.DeletePreset
     */
    @Override
    public CompletableFuture<DeletePresetResponse> deletePreset(DeletePresetRequest deletePresetRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeletePresetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeletePresetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeletePresetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeletePresetRequest, DeletePresetResponse>()
                            .withOperationName("DeletePreset").withMarshaller(new DeletePresetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deletePresetRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The ListJobsByPipeline operation gets a list of the jobs currently in a pipeline.
     * </p>
     * <p>
     * Elastic Transcoder returns all of the jobs currently in the specified pipeline. The response body contains one
     * element for each job that satisfies the search criteria.
     * </p>
     *
     * @param listJobsByPipelineRequest
     *        The <code>ListJobsByPipelineRequest</code> structure.
     * @return A Java Future containing the result of the ListJobsByPipeline operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One or more required parameter values were not provided in the request.</li>
     *         <li>IncompatibleVersionException</li>
     *         <li>ResourceNotFoundException The requested resource does not exist or is not available. For example, the
     *         pipeline to which you're trying to add a job doesn't exist or is still being created.</li>
     *         <li>AccessDeniedException General authentication failure. The request was not signed correctly.</li>
     *         <li>InternalServiceException Elastic Transcoder encountered an unexpected exception while trying to
     *         fulfill the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ElasticTranscoderException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ElasticTranscoderAsyncClient.ListJobsByPipeline
     */
    @Override
    public CompletableFuture<ListJobsByPipelineResponse> listJobsByPipeline(ListJobsByPipelineRequest listJobsByPipelineRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListJobsByPipelineResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListJobsByPipelineResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListJobsByPipelineResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListJobsByPipelineRequest, ListJobsByPipelineResponse>()
                            .withOperationName("ListJobsByPipeline")
                            .withMarshaller(new ListJobsByPipelineRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listJobsByPipelineRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The ListJobsByPipeline operation gets a list of the jobs currently in a pipeline.
     * </p>
     * <p>
     * Elastic Transcoder returns all of the jobs currently in the specified pipeline. The response body contains one
     * element for each job that satisfies the search criteria.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listJobsByPipeline(software.amazon.awssdk.services.elastictranscoder.model.ListJobsByPipelineRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.elastictranscoder.paginators.ListJobsByPipelinePublisher publisher = client.listJobsByPipelinePaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.elastictranscoder.paginators.ListJobsByPipelinePublisher publisher = client.listJobsByPipelinePaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.elastictranscoder.model.ListJobsByPipelineResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.elastictranscoder.model.ListJobsByPipelineResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listJobsByPipeline(software.amazon.awssdk.services.elastictranscoder.model.ListJobsByPipelineRequest)}
     * operation.</b>
     * </p>
     *
     * @param listJobsByPipelineRequest
     *        The <code>ListJobsByPipelineRequest</code> structure.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One or more required parameter values were not provided in the request.</li>
     *         <li>IncompatibleVersionException</li>
     *         <li>ResourceNotFoundException The requested resource does not exist or is not available. For example, the
     *         pipeline to which you're trying to add a job doesn't exist or is still being created.</li>
     *         <li>AccessDeniedException General authentication failure. The request was not signed correctly.</li>
     *         <li>InternalServiceException Elastic Transcoder encountered an unexpected exception while trying to
     *         fulfill the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ElasticTranscoderException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ElasticTranscoderAsyncClient.ListJobsByPipeline
     */
    public ListJobsByPipelinePublisher listJobsByPipelinePaginator(ListJobsByPipelineRequest listJobsByPipelineRequest) {
        return new ListJobsByPipelinePublisher(this, applyPaginatorUserAgent(listJobsByPipelineRequest));
    }

    /**
     * <p>
     * The ListJobsByStatus operation gets a list of jobs that have a specified status. The response body contains one
     * element for each job that satisfies the search criteria.
     * </p>
     *
     * @param listJobsByStatusRequest
     *        The <code>ListJobsByStatusRequest</code> structure.
     * @return A Java Future containing the result of the ListJobsByStatus operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One or more required parameter values were not provided in the request.</li>
     *         <li>IncompatibleVersionException</li>
     *         <li>ResourceNotFoundException The requested resource does not exist or is not available. For example, the
     *         pipeline to which you're trying to add a job doesn't exist or is still being created.</li>
     *         <li>AccessDeniedException General authentication failure. The request was not signed correctly.</li>
     *         <li>InternalServiceException Elastic Transcoder encountered an unexpected exception while trying to
     *         fulfill the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ElasticTranscoderException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ElasticTranscoderAsyncClient.ListJobsByStatus
     */
    @Override
    public CompletableFuture<ListJobsByStatusResponse> listJobsByStatus(ListJobsByStatusRequest listJobsByStatusRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListJobsByStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListJobsByStatusResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListJobsByStatusResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListJobsByStatusRequest, ListJobsByStatusResponse>()
                            .withOperationName("ListJobsByStatus")
                            .withMarshaller(new ListJobsByStatusRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listJobsByStatusRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The ListJobsByStatus operation gets a list of jobs that have a specified status. The response body contains one
     * element for each job that satisfies the search criteria.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listJobsByStatus(software.amazon.awssdk.services.elastictranscoder.model.ListJobsByStatusRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.elastictranscoder.paginators.ListJobsByStatusPublisher publisher = client.listJobsByStatusPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.elastictranscoder.paginators.ListJobsByStatusPublisher publisher = client.listJobsByStatusPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.elastictranscoder.model.ListJobsByStatusResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.elastictranscoder.model.ListJobsByStatusResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listJobsByStatus(software.amazon.awssdk.services.elastictranscoder.model.ListJobsByStatusRequest)}
     * operation.</b>
     * </p>
     *
     * @param listJobsByStatusRequest
     *        The <code>ListJobsByStatusRequest</code> structure.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One or more required parameter values were not provided in the request.</li>
     *         <li>IncompatibleVersionException</li>
     *         <li>ResourceNotFoundException The requested resource does not exist or is not available. For example, the
     *         pipeline to which you're trying to add a job doesn't exist or is still being created.</li>
     *         <li>AccessDeniedException General authentication failure. The request was not signed correctly.</li>
     *         <li>InternalServiceException Elastic Transcoder encountered an unexpected exception while trying to
     *         fulfill the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ElasticTranscoderException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ElasticTranscoderAsyncClient.ListJobsByStatus
     */
    public ListJobsByStatusPublisher listJobsByStatusPaginator(ListJobsByStatusRequest listJobsByStatusRequest) {
        return new ListJobsByStatusPublisher(this, applyPaginatorUserAgent(listJobsByStatusRequest));
    }

    /**
     * <p>
     * The ListPipelines operation gets a list of the pipelines associated with the current AWS account.
     * </p>
     *
     * @param listPipelinesRequest
     *        The <code>ListPipelineRequest</code> structure.
     * @return A Java Future containing the result of the ListPipelines operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One or more required parameter values were not provided in the request.</li>
     *         <li>IncompatibleVersionException</li>
     *         <li>AccessDeniedException General authentication failure. The request was not signed correctly.</li>
     *         <li>InternalServiceException Elastic Transcoder encountered an unexpected exception while trying to
     *         fulfill the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ElasticTranscoderException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ElasticTranscoderAsyncClient.ListPipelines
     */
    @Override
    public CompletableFuture<ListPipelinesResponse> listPipelines(ListPipelinesRequest listPipelinesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPipelinesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListPipelinesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPipelinesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPipelinesRequest, ListPipelinesResponse>()
                            .withOperationName("ListPipelines")
                            .withMarshaller(new ListPipelinesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listPipelinesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The ListPipelines operation gets a list of the pipelines associated with the current AWS account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listPipelines(software.amazon.awssdk.services.elastictranscoder.model.ListPipelinesRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.elastictranscoder.paginators.ListPipelinesPublisher publisher = client.listPipelinesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.elastictranscoder.paginators.ListPipelinesPublisher publisher = client.listPipelinesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.elastictranscoder.model.ListPipelinesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.elastictranscoder.model.ListPipelinesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPipelines(software.amazon.awssdk.services.elastictranscoder.model.ListPipelinesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listPipelinesRequest
     *        The <code>ListPipelineRequest</code> structure.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One or more required parameter values were not provided in the request.</li>
     *         <li>IncompatibleVersionException</li>
     *         <li>AccessDeniedException General authentication failure. The request was not signed correctly.</li>
     *         <li>InternalServiceException Elastic Transcoder encountered an unexpected exception while trying to
     *         fulfill the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ElasticTranscoderException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ElasticTranscoderAsyncClient.ListPipelines
     */
    public ListPipelinesPublisher listPipelinesPaginator(ListPipelinesRequest listPipelinesRequest) {
        return new ListPipelinesPublisher(this, applyPaginatorUserAgent(listPipelinesRequest));
    }

    /**
     * <p>
     * The ListPresets operation gets a list of the default presets included with Elastic Transcoder and the presets
     * that you've added in an AWS region.
     * </p>
     *
     * @param listPresetsRequest
     *        The <code>ListPresetsRequest</code> structure.
     * @return A Java Future containing the result of the ListPresets operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One or more required parameter values were not provided in the request.</li>
     *         <li>IncompatibleVersionException</li>
     *         <li>AccessDeniedException General authentication failure. The request was not signed correctly.</li>
     *         <li>InternalServiceException Elastic Transcoder encountered an unexpected exception while trying to
     *         fulfill the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ElasticTranscoderException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ElasticTranscoderAsyncClient.ListPresets
     */
    @Override
    public CompletableFuture<ListPresetsResponse> listPresets(ListPresetsRequest listPresetsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPresetsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListPresetsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPresetsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPresetsRequest, ListPresetsResponse>()
                            .withOperationName("ListPresets").withMarshaller(new ListPresetsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listPresetsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The ListPresets operation gets a list of the default presets included with Elastic Transcoder and the presets
     * that you've added in an AWS region.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listPresets(software.amazon.awssdk.services.elastictranscoder.model.ListPresetsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.elastictranscoder.paginators.ListPresetsPublisher publisher = client.listPresetsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.elastictranscoder.paginators.ListPresetsPublisher publisher = client.listPresetsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.elastictranscoder.model.ListPresetsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.elastictranscoder.model.ListPresetsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPresets(software.amazon.awssdk.services.elastictranscoder.model.ListPresetsRequest)} operation.</b>
     * </p>
     *
     * @param listPresetsRequest
     *        The <code>ListPresetsRequest</code> structure.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One or more required parameter values were not provided in the request.</li>
     *         <li>IncompatibleVersionException</li>
     *         <li>AccessDeniedException General authentication failure. The request was not signed correctly.</li>
     *         <li>InternalServiceException Elastic Transcoder encountered an unexpected exception while trying to
     *         fulfill the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ElasticTranscoderException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ElasticTranscoderAsyncClient.ListPresets
     */
    public ListPresetsPublisher listPresetsPaginator(ListPresetsRequest listPresetsRequest) {
        return new ListPresetsPublisher(this, applyPaginatorUserAgent(listPresetsRequest));
    }

    /**
     * <p>
     * The ReadJob operation returns detailed information about a job.
     * </p>
     *
     * @param readJobRequest
     *        The <code>ReadJobRequest</code> structure.
     * @return A Java Future containing the result of the ReadJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One or more required parameter values were not provided in the request.</li>
     *         <li>IncompatibleVersionException</li>
     *         <li>ResourceNotFoundException The requested resource does not exist or is not available. For example, the
     *         pipeline to which you're trying to add a job doesn't exist or is still being created.</li>
     *         <li>AccessDeniedException General authentication failure. The request was not signed correctly.</li>
     *         <li>InternalServiceException Elastic Transcoder encountered an unexpected exception while trying to
     *         fulfill the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ElasticTranscoderException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ElasticTranscoderAsyncClient.ReadJob
     */
    @Override
    public CompletableFuture<ReadJobResponse> readJob(ReadJobRequest readJobRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ReadJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ReadJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ReadJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ReadJobRequest, ReadJobResponse>().withOperationName("ReadJob")
                            .withMarshaller(new ReadJobRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(readJobRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The ReadPipeline operation gets detailed information about a pipeline.
     * </p>
     *
     * @param readPipelineRequest
     *        The <code>ReadPipelineRequest</code> structure.
     * @return A Java Future containing the result of the ReadPipeline operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One or more required parameter values were not provided in the request.</li>
     *         <li>IncompatibleVersionException</li>
     *         <li>ResourceNotFoundException The requested resource does not exist or is not available. For example, the
     *         pipeline to which you're trying to add a job doesn't exist or is still being created.</li>
     *         <li>AccessDeniedException General authentication failure. The request was not signed correctly.</li>
     *         <li>InternalServiceException Elastic Transcoder encountered an unexpected exception while trying to
     *         fulfill the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ElasticTranscoderException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ElasticTranscoderAsyncClient.ReadPipeline
     */
    @Override
    public CompletableFuture<ReadPipelineResponse> readPipeline(ReadPipelineRequest readPipelineRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ReadPipelineResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ReadPipelineResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ReadPipelineResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ReadPipelineRequest, ReadPipelineResponse>()
                            .withOperationName("ReadPipeline").withMarshaller(new ReadPipelineRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(readPipelineRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The ReadPreset operation gets detailed information about a preset.
     * </p>
     *
     * @param readPresetRequest
     *        The <code>ReadPresetRequest</code> structure.
     * @return A Java Future containing the result of the ReadPreset operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One or more required parameter values were not provided in the request.</li>
     *         <li>IncompatibleVersionException</li>
     *         <li>ResourceNotFoundException The requested resource does not exist or is not available. For example, the
     *         pipeline to which you're trying to add a job doesn't exist or is still being created.</li>
     *         <li>AccessDeniedException General authentication failure. The request was not signed correctly.</li>
     *         <li>InternalServiceException Elastic Transcoder encountered an unexpected exception while trying to
     *         fulfill the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ElasticTranscoderException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ElasticTranscoderAsyncClient.ReadPreset
     */
    @Override
    public CompletableFuture<ReadPresetResponse> readPreset(ReadPresetRequest readPresetRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ReadPresetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ReadPresetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ReadPresetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ReadPresetRequest, ReadPresetResponse>().withOperationName("ReadPreset")
                            .withMarshaller(new ReadPresetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(readPresetRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Use the <code>UpdatePipeline</code> operation to update settings for a pipeline.
     * </p>
     * <important>
     * <p>
     * When you change pipeline settings, your changes take effect immediately. Jobs that you have already submitted and
     * that Elastic Transcoder has not started to process are affected in addition to jobs that you submit after you
     * change settings.
     * </p>
     * </important>
     *
     * @param updatePipelineRequest
     *        The <code>UpdatePipelineRequest</code> structure.
     * @return A Java Future containing the result of the UpdatePipeline operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One or more required parameter values were not provided in the request.</li>
     *         <li>IncompatibleVersionException</li>
     *         <li>AccessDeniedException General authentication failure. The request was not signed correctly.</li>
     *         <li>ResourceInUseException The resource you are attempting to change is in use. For example, you are
     *         attempting to delete a pipeline that is currently in use.</li>
     *         <li>ResourceNotFoundException The requested resource does not exist or is not available. For example, the
     *         pipeline to which you're trying to add a job doesn't exist or is still being created.</li>
     *         <li>InternalServiceException Elastic Transcoder encountered an unexpected exception while trying to
     *         fulfill the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ElasticTranscoderException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ElasticTranscoderAsyncClient.UpdatePipeline
     */
    @Override
    public CompletableFuture<UpdatePipelineResponse> updatePipeline(UpdatePipelineRequest updatePipelineRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdatePipelineResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdatePipelineResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdatePipelineResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdatePipelineRequest, UpdatePipelineResponse>()
                            .withOperationName("UpdatePipeline")
                            .withMarshaller(new UpdatePipelineRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updatePipelineRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * With the UpdatePipelineNotifications operation, you can update Amazon Simple Notification Service (Amazon SNS)
     * notifications for a pipeline.
     * </p>
     * <p>
     * When you update notifications for a pipeline, Elastic Transcoder returns the values that you specified in the
     * request.
     * </p>
     *
     * @param updatePipelineNotificationsRequest
     *        The <code>UpdatePipelineNotificationsRequest</code> structure.
     * @return A Java Future containing the result of the UpdatePipelineNotifications operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One or more required parameter values were not provided in the request.</li>
     *         <li>IncompatibleVersionException</li>
     *         <li>ResourceNotFoundException The requested resource does not exist or is not available. For example, the
     *         pipeline to which you're trying to add a job doesn't exist or is still being created.</li>
     *         <li>ResourceInUseException The resource you are attempting to change is in use. For example, you are
     *         attempting to delete a pipeline that is currently in use.</li>
     *         <li>AccessDeniedException General authentication failure. The request was not signed correctly.</li>
     *         <li>InternalServiceException Elastic Transcoder encountered an unexpected exception while trying to
     *         fulfill the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ElasticTranscoderException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ElasticTranscoderAsyncClient.UpdatePipelineNotifications
     */
    @Override
    public CompletableFuture<UpdatePipelineNotificationsResponse> updatePipelineNotifications(
            UpdatePipelineNotificationsRequest updatePipelineNotificationsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdatePipelineNotificationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdatePipelineNotificationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdatePipelineNotificationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdatePipelineNotificationsRequest, UpdatePipelineNotificationsResponse>()
                            .withOperationName("UpdatePipelineNotifications")
                            .withMarshaller(new UpdatePipelineNotificationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updatePipelineNotificationsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The UpdatePipelineStatus operation pauses or reactivates a pipeline, so that the pipeline stops or restarts the
     * processing of jobs.
     * </p>
     * <p>
     * Changing the pipeline status is useful if you want to cancel one or more jobs. You can't cancel jobs after
     * Elastic Transcoder has started processing them; if you pause the pipeline to which you submitted the jobs, you
     * have more time to get the job IDs for the jobs that you want to cancel, and to send a <a>CancelJob</a> request.
     * </p>
     *
     * @param updatePipelineStatusRequest
     *        The <code>UpdatePipelineStatusRequest</code> structure.
     * @return A Java Future containing the result of the UpdatePipelineStatus operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One or more required parameter values were not provided in the request.</li>
     *         <li>IncompatibleVersionException</li>
     *         <li>ResourceNotFoundException The requested resource does not exist or is not available. For example, the
     *         pipeline to which you're trying to add a job doesn't exist or is still being created.</li>
     *         <li>ResourceInUseException The resource you are attempting to change is in use. For example, you are
     *         attempting to delete a pipeline that is currently in use.</li>
     *         <li>AccessDeniedException General authentication failure. The request was not signed correctly.</li>
     *         <li>InternalServiceException Elastic Transcoder encountered an unexpected exception while trying to
     *         fulfill the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ElasticTranscoderException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ElasticTranscoderAsyncClient.UpdatePipelineStatus
     */
    @Override
    public CompletableFuture<UpdatePipelineStatusResponse> updatePipelineStatus(
            UpdatePipelineStatusRequest updatePipelineStatusRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdatePipelineStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdatePipelineStatusResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdatePipelineStatusResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdatePipelineStatusRequest, UpdatePipelineStatusResponse>()
                            .withOperationName("UpdatePipelineStatus")
                            .withMarshaller(new UpdatePipelineStatusRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updatePipelineStatusRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ElasticTranscoderException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceInUseException")
                                .exceptionBuilderSupplier(ResourceInUseException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("IncompatibleVersionException")
                                .exceptionBuilderSupplier(IncompatibleVersionException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServiceException")
                                .exceptionBuilderSupplier(InternalServiceException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(429).build());
    }

    private <T extends ElasticTranscoderRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
