/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.elastictranscoder.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Use Only for Fragmented MP4 or MPEG-TS Outputs. If you specify a preset for which the value of Container is
 * <code>fmp4</code> (Fragmented MP4) or <code>ts</code> (MPEG-TS), Playlists contains information about the master
 * playlists that you want Elastic Transcoder to create. We recommend that you create only one master playlist per
 * output format. The maximum number of master playlists in a job is 30.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Playlist implements SdkPojo, Serializable, ToCopyableBuilder<Playlist.Builder, Playlist> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Playlist::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> FORMAT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Playlist::format)).setter(setter(Builder::format))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Format").build()).build();

    private static final SdkField<List<String>> OUTPUT_KEYS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(Playlist::outputKeys))
            .setter(setter(Builder::outputKeys))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OutputKeys").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<HlsContentProtection> HLS_CONTENT_PROTECTION_FIELD = SdkField
            .<HlsContentProtection> builder(MarshallingType.SDK_POJO).getter(getter(Playlist::hlsContentProtection))
            .setter(setter(Builder::hlsContentProtection)).constructor(HlsContentProtection::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HlsContentProtection").build())
            .build();

    private static final SdkField<PlayReadyDrm> PLAY_READY_DRM_FIELD = SdkField.<PlayReadyDrm> builder(MarshallingType.SDK_POJO)
            .getter(getter(Playlist::playReadyDrm)).setter(setter(Builder::playReadyDrm)).constructor(PlayReadyDrm::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PlayReadyDrm").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Playlist::status)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> STATUS_DETAIL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Playlist::statusDetail)).setter(setter(Builder::statusDetail))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StatusDetail").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, FORMAT_FIELD,
            OUTPUT_KEYS_FIELD, HLS_CONTENT_PROTECTION_FIELD, PLAY_READY_DRM_FIELD, STATUS_FIELD, STATUS_DETAIL_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String format;

    private final List<String> outputKeys;

    private final HlsContentProtection hlsContentProtection;

    private final PlayReadyDrm playReadyDrm;

    private final String status;

    private final String statusDetail;

    private Playlist(BuilderImpl builder) {
        this.name = builder.name;
        this.format = builder.format;
        this.outputKeys = builder.outputKeys;
        this.hlsContentProtection = builder.hlsContentProtection;
        this.playReadyDrm = builder.playReadyDrm;
        this.status = builder.status;
        this.statusDetail = builder.statusDetail;
    }

    /**
     * <p>
     * The name that you want Elastic Transcoder to assign to the master playlist, for example, nyc-vacation.m3u8. If
     * the name includes a <code>/</code> character, the section of the name before the last <code>/</code> must be
     * identical for all <code>Name</code> objects. If you create more than one master playlist, the values of all
     * <code>Name</code> objects must be unique.
     * </p>
     * <note>
     * <p>
     * Elastic Transcoder automatically appends the relevant file extension to the file name (<code>.m3u8</code> for
     * <code>HLSv3</code> and <code>HLSv4</code> playlists, and <code>.ism</code> and <code>.ismc</code> for
     * <code>Smooth</code> playlists). If you include a file extension in <code>Name</code>, the file name will have two
     * extensions.
     * </p>
     * </note>
     * 
     * @return The name that you want Elastic Transcoder to assign to the master playlist, for example,
     *         nyc-vacation.m3u8. If the name includes a <code>/</code> character, the section of the name before the
     *         last <code>/</code> must be identical for all <code>Name</code> objects. If you create more than one
     *         master playlist, the values of all <code>Name</code> objects must be unique.</p> <note>
     *         <p>
     *         Elastic Transcoder automatically appends the relevant file extension to the file name (<code>.m3u8</code>
     *         for <code>HLSv3</code> and <code>HLSv4</code> playlists, and <code>.ism</code> and <code>.ismc</code> for
     *         <code>Smooth</code> playlists). If you include a file extension in <code>Name</code>, the file name will
     *         have two extensions.
     *         </p>
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * The format of the output playlist. Valid formats include <code>HLSv3</code>, <code>HLSv4</code>, and
     * <code>Smooth</code>.
     * </p>
     * 
     * @return The format of the output playlist. Valid formats include <code>HLSv3</code>, <code>HLSv4</code>, and
     *         <code>Smooth</code>.
     */
    public String format() {
        return format;
    }

    /**
     * <p>
     * For each output in this job that you want to include in a master playlist, the value of the Outputs:Key object.
     * </p>
     * <ul>
     * <li>
     * <p>
     * If your output is not <code>HLS</code> or does not have a segment duration set, the name of the output file is a
     * concatenation of <code>OutputKeyPrefix</code> and <code>Outputs:Key</code>:
     * </p>
     * <p>
     * OutputKeyPrefix<code>Outputs:Key</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * If your output is <code>HLSv3</code> and has a segment duration set, or is not included in a playlist, Elastic
     * Transcoder creates an output playlist file with a file extension of <code>.m3u8</code>, and a series of
     * <code>.ts</code> files that include a five-digit sequential counter beginning with 00000:
     * </p>
     * <p>
     * OutputKeyPrefix<code>Outputs:Key</code>.m3u8
     * </p>
     * <p>
     * OutputKeyPrefix<code>Outputs:Key</code>00000.ts
     * </p>
     * </li>
     * <li>
     * <p>
     * If your output is <code>HLSv4</code>, has a segment duration set, and is included in an <code>HLSv4</code>
     * playlist, Elastic Transcoder creates an output playlist file with a file extension of <code>_v4.m3u8</code>. If
     * the output is video, Elastic Transcoder also creates an output file with an extension of
     * <code>_iframe.m3u8</code>:
     * </p>
     * <p>
     * OutputKeyPrefix<code>Outputs:Key</code>_v4.m3u8
     * </p>
     * <p>
     * OutputKeyPrefix<code>Outputs:Key</code>_iframe.m3u8
     * </p>
     * <p>
     * OutputKeyPrefix<code>Outputs:Key</code>.ts
     * </p>
     * </li>
     * </ul>
     * <p>
     * Elastic Transcoder automatically appends the relevant file extension to the file name. If you include a file
     * extension in Output Key, the file name will have two extensions.
     * </p>
     * <p>
     * If you include more than one output in a playlist, any segment duration settings, clip settings, or caption
     * settings must be the same for all outputs in the playlist. For <code>Smooth</code> playlists, the
     * <code>Audio:Profile</code>, <code>Video:Profile</code>, and <code>Video:FrameRate</code> to
     * <code>Video:KeyframesMaxDist</code> ratio must be the same for all outputs.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return For each output in this job that you want to include in a master playlist, the value of the Outputs:Key
     *         object.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         If your output is not <code>HLS</code> or does not have a segment duration set, the name of the output
     *         file is a concatenation of <code>OutputKeyPrefix</code> and <code>Outputs:Key</code>:
     *         </p>
     *         <p>
     *         OutputKeyPrefix<code>Outputs:Key</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If your output is <code>HLSv3</code> and has a segment duration set, or is not included in a playlist,
     *         Elastic Transcoder creates an output playlist file with a file extension of <code>.m3u8</code>, and a
     *         series of <code>.ts</code> files that include a five-digit sequential counter beginning with 00000:
     *         </p>
     *         <p>
     *         OutputKeyPrefix<code>Outputs:Key</code>.m3u8
     *         </p>
     *         <p>
     *         OutputKeyPrefix<code>Outputs:Key</code>00000.ts
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If your output is <code>HLSv4</code>, has a segment duration set, and is included in an
     *         <code>HLSv4</code> playlist, Elastic Transcoder creates an output playlist file with a file extension of
     *         <code>_v4.m3u8</code>. If the output is video, Elastic Transcoder also creates an output file with an
     *         extension of <code>_iframe.m3u8</code>:
     *         </p>
     *         <p>
     *         OutputKeyPrefix<code>Outputs:Key</code>_v4.m3u8
     *         </p>
     *         <p>
     *         OutputKeyPrefix<code>Outputs:Key</code>_iframe.m3u8
     *         </p>
     *         <p>
     *         OutputKeyPrefix<code>Outputs:Key</code>.ts
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Elastic Transcoder automatically appends the relevant file extension to the file name. If you include a
     *         file extension in Output Key, the file name will have two extensions.
     *         </p>
     *         <p>
     *         If you include more than one output in a playlist, any segment duration settings, clip settings, or
     *         caption settings must be the same for all outputs in the playlist. For <code>Smooth</code> playlists, the
     *         <code>Audio:Profile</code>, <code>Video:Profile</code>, and <code>Video:FrameRate</code> to
     *         <code>Video:KeyframesMaxDist</code> ratio must be the same for all outputs.
     */
    public List<String> outputKeys() {
        return outputKeys;
    }

    /**
     * <p>
     * The HLS content protection settings, if any, that you want Elastic Transcoder to apply to the output files
     * associated with this playlist.
     * </p>
     * 
     * @return The HLS content protection settings, if any, that you want Elastic Transcoder to apply to the output
     *         files associated with this playlist.
     */
    public HlsContentProtection hlsContentProtection() {
        return hlsContentProtection;
    }

    /**
     * <p>
     * The DRM settings, if any, that you want Elastic Transcoder to apply to the output files associated with this
     * playlist.
     * </p>
     * 
     * @return The DRM settings, if any, that you want Elastic Transcoder to apply to the output files associated with
     *         this playlist.
     */
    public PlayReadyDrm playReadyDrm() {
        return playReadyDrm;
    }

    /**
     * <p>
     * The status of the job with which the playlist is associated.
     * </p>
     * 
     * @return The status of the job with which the playlist is associated.
     */
    public String status() {
        return status;
    }

    /**
     * <p>
     * Information that further explains the status.
     * </p>
     * 
     * @return Information that further explains the status.
     */
    public String statusDetail() {
        return statusDetail;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(format());
        hashCode = 31 * hashCode + Objects.hashCode(outputKeys());
        hashCode = 31 * hashCode + Objects.hashCode(hlsContentProtection());
        hashCode = 31 * hashCode + Objects.hashCode(playReadyDrm());
        hashCode = 31 * hashCode + Objects.hashCode(status());
        hashCode = 31 * hashCode + Objects.hashCode(statusDetail());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Playlist)) {
            return false;
        }
        Playlist other = (Playlist) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(format(), other.format())
                && Objects.equals(outputKeys(), other.outputKeys())
                && Objects.equals(hlsContentProtection(), other.hlsContentProtection())
                && Objects.equals(playReadyDrm(), other.playReadyDrm()) && Objects.equals(status(), other.status())
                && Objects.equals(statusDetail(), other.statusDetail());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("Playlist").add("Name", name()).add("Format", format()).add("OutputKeys", outputKeys())
                .add("HlsContentProtection", hlsContentProtection()).add("PlayReadyDrm", playReadyDrm()).add("Status", status())
                .add("StatusDetail", statusDetail()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Format":
            return Optional.ofNullable(clazz.cast(format()));
        case "OutputKeys":
            return Optional.ofNullable(clazz.cast(outputKeys()));
        case "HlsContentProtection":
            return Optional.ofNullable(clazz.cast(hlsContentProtection()));
        case "PlayReadyDrm":
            return Optional.ofNullable(clazz.cast(playReadyDrm()));
        case "Status":
            return Optional.ofNullable(clazz.cast(status()));
        case "StatusDetail":
            return Optional.ofNullable(clazz.cast(statusDetail()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Playlist, T> g) {
        return obj -> g.apply((Playlist) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Playlist> {
        /**
         * <p>
         * The name that you want Elastic Transcoder to assign to the master playlist, for example, nyc-vacation.m3u8.
         * If the name includes a <code>/</code> character, the section of the name before the last <code>/</code> must
         * be identical for all <code>Name</code> objects. If you create more than one master playlist, the values of
         * all <code>Name</code> objects must be unique.
         * </p>
         * <note>
         * <p>
         * Elastic Transcoder automatically appends the relevant file extension to the file name (<code>.m3u8</code> for
         * <code>HLSv3</code> and <code>HLSv4</code> playlists, and <code>.ism</code> and <code>.ismc</code> for
         * <code>Smooth</code> playlists). If you include a file extension in <code>Name</code>, the file name will have
         * two extensions.
         * </p>
         * </note>
         * 
         * @param name
         *        The name that you want Elastic Transcoder to assign to the master playlist, for example,
         *        nyc-vacation.m3u8. If the name includes a <code>/</code> character, the section of the name before the
         *        last <code>/</code> must be identical for all <code>Name</code> objects. If you create more than one
         *        master playlist, the values of all <code>Name</code> objects must be unique.</p> <note>
         *        <p>
         *        Elastic Transcoder automatically appends the relevant file extension to the file name (
         *        <code>.m3u8</code> for <code>HLSv3</code> and <code>HLSv4</code> playlists, and <code>.ism</code> and
         *        <code>.ismc</code> for <code>Smooth</code> playlists). If you include a file extension in
         *        <code>Name</code>, the file name will have two extensions.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The format of the output playlist. Valid formats include <code>HLSv3</code>, <code>HLSv4</code>, and
         * <code>Smooth</code>.
         * </p>
         * 
         * @param format
         *        The format of the output playlist. Valid formats include <code>HLSv3</code>, <code>HLSv4</code>, and
         *        <code>Smooth</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder format(String format);

        /**
         * <p>
         * For each output in this job that you want to include in a master playlist, the value of the Outputs:Key
         * object.
         * </p>
         * <ul>
         * <li>
         * <p>
         * If your output is not <code>HLS</code> or does not have a segment duration set, the name of the output file
         * is a concatenation of <code>OutputKeyPrefix</code> and <code>Outputs:Key</code>:
         * </p>
         * <p>
         * OutputKeyPrefix<code>Outputs:Key</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * If your output is <code>HLSv3</code> and has a segment duration set, or is not included in a playlist,
         * Elastic Transcoder creates an output playlist file with a file extension of <code>.m3u8</code>, and a series
         * of <code>.ts</code> files that include a five-digit sequential counter beginning with 00000:
         * </p>
         * <p>
         * OutputKeyPrefix<code>Outputs:Key</code>.m3u8
         * </p>
         * <p>
         * OutputKeyPrefix<code>Outputs:Key</code>00000.ts
         * </p>
         * </li>
         * <li>
         * <p>
         * If your output is <code>HLSv4</code>, has a segment duration set, and is included in an <code>HLSv4</code>
         * playlist, Elastic Transcoder creates an output playlist file with a file extension of <code>_v4.m3u8</code>.
         * If the output is video, Elastic Transcoder also creates an output file with an extension of
         * <code>_iframe.m3u8</code>:
         * </p>
         * <p>
         * OutputKeyPrefix<code>Outputs:Key</code>_v4.m3u8
         * </p>
         * <p>
         * OutputKeyPrefix<code>Outputs:Key</code>_iframe.m3u8
         * </p>
         * <p>
         * OutputKeyPrefix<code>Outputs:Key</code>.ts
         * </p>
         * </li>
         * </ul>
         * <p>
         * Elastic Transcoder automatically appends the relevant file extension to the file name. If you include a file
         * extension in Output Key, the file name will have two extensions.
         * </p>
         * <p>
         * If you include more than one output in a playlist, any segment duration settings, clip settings, or caption
         * settings must be the same for all outputs in the playlist. For <code>Smooth</code> playlists, the
         * <code>Audio:Profile</code>, <code>Video:Profile</code>, and <code>Video:FrameRate</code> to
         * <code>Video:KeyframesMaxDist</code> ratio must be the same for all outputs.
         * </p>
         * 
         * @param outputKeys
         *        For each output in this job that you want to include in a master playlist, the value of the
         *        Outputs:Key object.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        If your output is not <code>HLS</code> or does not have a segment duration set, the name of the output
         *        file is a concatenation of <code>OutputKeyPrefix</code> and <code>Outputs:Key</code>:
         *        </p>
         *        <p>
         *        OutputKeyPrefix<code>Outputs:Key</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If your output is <code>HLSv3</code> and has a segment duration set, or is not included in a playlist,
         *        Elastic Transcoder creates an output playlist file with a file extension of <code>.m3u8</code>, and a
         *        series of <code>.ts</code> files that include a five-digit sequential counter beginning with 00000:
         *        </p>
         *        <p>
         *        OutputKeyPrefix<code>Outputs:Key</code>.m3u8
         *        </p>
         *        <p>
         *        OutputKeyPrefix<code>Outputs:Key</code>00000.ts
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If your output is <code>HLSv4</code>, has a segment duration set, and is included in an
         *        <code>HLSv4</code> playlist, Elastic Transcoder creates an output playlist file with a file extension
         *        of <code>_v4.m3u8</code>. If the output is video, Elastic Transcoder also creates an output file with
         *        an extension of <code>_iframe.m3u8</code>:
         *        </p>
         *        <p>
         *        OutputKeyPrefix<code>Outputs:Key</code>_v4.m3u8
         *        </p>
         *        <p>
         *        OutputKeyPrefix<code>Outputs:Key</code>_iframe.m3u8
         *        </p>
         *        <p>
         *        OutputKeyPrefix<code>Outputs:Key</code>.ts
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Elastic Transcoder automatically appends the relevant file extension to the file name. If you include
         *        a file extension in Output Key, the file name will have two extensions.
         *        </p>
         *        <p>
         *        If you include more than one output in a playlist, any segment duration settings, clip settings, or
         *        caption settings must be the same for all outputs in the playlist. For <code>Smooth</code> playlists,
         *        the <code>Audio:Profile</code>, <code>Video:Profile</code>, and <code>Video:FrameRate</code> to
         *        <code>Video:KeyframesMaxDist</code> ratio must be the same for all outputs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputKeys(Collection<String> outputKeys);

        /**
         * <p>
         * For each output in this job that you want to include in a master playlist, the value of the Outputs:Key
         * object.
         * </p>
         * <ul>
         * <li>
         * <p>
         * If your output is not <code>HLS</code> or does not have a segment duration set, the name of the output file
         * is a concatenation of <code>OutputKeyPrefix</code> and <code>Outputs:Key</code>:
         * </p>
         * <p>
         * OutputKeyPrefix<code>Outputs:Key</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * If your output is <code>HLSv3</code> and has a segment duration set, or is not included in a playlist,
         * Elastic Transcoder creates an output playlist file with a file extension of <code>.m3u8</code>, and a series
         * of <code>.ts</code> files that include a five-digit sequential counter beginning with 00000:
         * </p>
         * <p>
         * OutputKeyPrefix<code>Outputs:Key</code>.m3u8
         * </p>
         * <p>
         * OutputKeyPrefix<code>Outputs:Key</code>00000.ts
         * </p>
         * </li>
         * <li>
         * <p>
         * If your output is <code>HLSv4</code>, has a segment duration set, and is included in an <code>HLSv4</code>
         * playlist, Elastic Transcoder creates an output playlist file with a file extension of <code>_v4.m3u8</code>.
         * If the output is video, Elastic Transcoder also creates an output file with an extension of
         * <code>_iframe.m3u8</code>:
         * </p>
         * <p>
         * OutputKeyPrefix<code>Outputs:Key</code>_v4.m3u8
         * </p>
         * <p>
         * OutputKeyPrefix<code>Outputs:Key</code>_iframe.m3u8
         * </p>
         * <p>
         * OutputKeyPrefix<code>Outputs:Key</code>.ts
         * </p>
         * </li>
         * </ul>
         * <p>
         * Elastic Transcoder automatically appends the relevant file extension to the file name. If you include a file
         * extension in Output Key, the file name will have two extensions.
         * </p>
         * <p>
         * If you include more than one output in a playlist, any segment duration settings, clip settings, or caption
         * settings must be the same for all outputs in the playlist. For <code>Smooth</code> playlists, the
         * <code>Audio:Profile</code>, <code>Video:Profile</code>, and <code>Video:FrameRate</code> to
         * <code>Video:KeyframesMaxDist</code> ratio must be the same for all outputs.
         * </p>
         * 
         * @param outputKeys
         *        For each output in this job that you want to include in a master playlist, the value of the
         *        Outputs:Key object.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        If your output is not <code>HLS</code> or does not have a segment duration set, the name of the output
         *        file is a concatenation of <code>OutputKeyPrefix</code> and <code>Outputs:Key</code>:
         *        </p>
         *        <p>
         *        OutputKeyPrefix<code>Outputs:Key</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If your output is <code>HLSv3</code> and has a segment duration set, or is not included in a playlist,
         *        Elastic Transcoder creates an output playlist file with a file extension of <code>.m3u8</code>, and a
         *        series of <code>.ts</code> files that include a five-digit sequential counter beginning with 00000:
         *        </p>
         *        <p>
         *        OutputKeyPrefix<code>Outputs:Key</code>.m3u8
         *        </p>
         *        <p>
         *        OutputKeyPrefix<code>Outputs:Key</code>00000.ts
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If your output is <code>HLSv4</code>, has a segment duration set, and is included in an
         *        <code>HLSv4</code> playlist, Elastic Transcoder creates an output playlist file with a file extension
         *        of <code>_v4.m3u8</code>. If the output is video, Elastic Transcoder also creates an output file with
         *        an extension of <code>_iframe.m3u8</code>:
         *        </p>
         *        <p>
         *        OutputKeyPrefix<code>Outputs:Key</code>_v4.m3u8
         *        </p>
         *        <p>
         *        OutputKeyPrefix<code>Outputs:Key</code>_iframe.m3u8
         *        </p>
         *        <p>
         *        OutputKeyPrefix<code>Outputs:Key</code>.ts
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Elastic Transcoder automatically appends the relevant file extension to the file name. If you include
         *        a file extension in Output Key, the file name will have two extensions.
         *        </p>
         *        <p>
         *        If you include more than one output in a playlist, any segment duration settings, clip settings, or
         *        caption settings must be the same for all outputs in the playlist. For <code>Smooth</code> playlists,
         *        the <code>Audio:Profile</code>, <code>Video:Profile</code>, and <code>Video:FrameRate</code> to
         *        <code>Video:KeyframesMaxDist</code> ratio must be the same for all outputs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputKeys(String... outputKeys);

        /**
         * <p>
         * The HLS content protection settings, if any, that you want Elastic Transcoder to apply to the output files
         * associated with this playlist.
         * </p>
         * 
         * @param hlsContentProtection
         *        The HLS content protection settings, if any, that you want Elastic Transcoder to apply to the output
         *        files associated with this playlist.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hlsContentProtection(HlsContentProtection hlsContentProtection);

        /**
         * <p>
         * The HLS content protection settings, if any, that you want Elastic Transcoder to apply to the output files
         * associated with this playlist.
         * </p>
         * This is a convenience that creates an instance of the {@link HlsContentProtection.Builder} avoiding the need
         * to create one manually via {@link HlsContentProtection#builder()}.
         *
         * When the {@link Consumer} completes, {@link HlsContentProtection.Builder#build()} is called immediately and
         * its result is passed to {@link #hlsContentProtection(HlsContentProtection)}.
         * 
         * @param hlsContentProtection
         *        a consumer that will call methods on {@link HlsContentProtection.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #hlsContentProtection(HlsContentProtection)
         */
        default Builder hlsContentProtection(Consumer<HlsContentProtection.Builder> hlsContentProtection) {
            return hlsContentProtection(HlsContentProtection.builder().applyMutation(hlsContentProtection).build());
        }

        /**
         * <p>
         * The DRM settings, if any, that you want Elastic Transcoder to apply to the output files associated with this
         * playlist.
         * </p>
         * 
         * @param playReadyDrm
         *        The DRM settings, if any, that you want Elastic Transcoder to apply to the output files associated
         *        with this playlist.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder playReadyDrm(PlayReadyDrm playReadyDrm);

        /**
         * <p>
         * The DRM settings, if any, that you want Elastic Transcoder to apply to the output files associated with this
         * playlist.
         * </p>
         * This is a convenience that creates an instance of the {@link PlayReadyDrm.Builder} avoiding the need to
         * create one manually via {@link PlayReadyDrm#builder()}.
         *
         * When the {@link Consumer} completes, {@link PlayReadyDrm.Builder#build()} is called immediately and its
         * result is passed to {@link #playReadyDrm(PlayReadyDrm)}.
         * 
         * @param playReadyDrm
         *        a consumer that will call methods on {@link PlayReadyDrm.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #playReadyDrm(PlayReadyDrm)
         */
        default Builder playReadyDrm(Consumer<PlayReadyDrm.Builder> playReadyDrm) {
            return playReadyDrm(PlayReadyDrm.builder().applyMutation(playReadyDrm).build());
        }

        /**
         * <p>
         * The status of the job with which the playlist is associated.
         * </p>
         * 
         * @param status
         *        The status of the job with which the playlist is associated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder status(String status);

        /**
         * <p>
         * Information that further explains the status.
         * </p>
         * 
         * @param statusDetail
         *        Information that further explains the status.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusDetail(String statusDetail);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String format;

        private List<String> outputKeys = DefaultSdkAutoConstructList.getInstance();

        private HlsContentProtection hlsContentProtection;

        private PlayReadyDrm playReadyDrm;

        private String status;

        private String statusDetail;

        private BuilderImpl() {
        }

        private BuilderImpl(Playlist model) {
            name(model.name);
            format(model.format);
            outputKeys(model.outputKeys);
            hlsContentProtection(model.hlsContentProtection);
            playReadyDrm(model.playReadyDrm);
            status(model.status);
            statusDetail(model.statusDetail);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getFormat() {
            return format;
        }

        @Override
        public final Builder format(String format) {
            this.format = format;
            return this;
        }

        public final void setFormat(String format) {
            this.format = format;
        }

        public final Collection<String> getOutputKeys() {
            return outputKeys;
        }

        @Override
        public final Builder outputKeys(Collection<String> outputKeys) {
            this.outputKeys = OutputKeysCopier.copy(outputKeys);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder outputKeys(String... outputKeys) {
            outputKeys(Arrays.asList(outputKeys));
            return this;
        }

        public final void setOutputKeys(Collection<String> outputKeys) {
            this.outputKeys = OutputKeysCopier.copy(outputKeys);
        }

        public final HlsContentProtection.Builder getHlsContentProtection() {
            return hlsContentProtection != null ? hlsContentProtection.toBuilder() : null;
        }

        @Override
        public final Builder hlsContentProtection(HlsContentProtection hlsContentProtection) {
            this.hlsContentProtection = hlsContentProtection;
            return this;
        }

        public final void setHlsContentProtection(HlsContentProtection.BuilderImpl hlsContentProtection) {
            this.hlsContentProtection = hlsContentProtection != null ? hlsContentProtection.build() : null;
        }

        public final PlayReadyDrm.Builder getPlayReadyDrm() {
            return playReadyDrm != null ? playReadyDrm.toBuilder() : null;
        }

        @Override
        public final Builder playReadyDrm(PlayReadyDrm playReadyDrm) {
            this.playReadyDrm = playReadyDrm;
            return this;
        }

        public final void setPlayReadyDrm(PlayReadyDrm.BuilderImpl playReadyDrm) {
            this.playReadyDrm = playReadyDrm != null ? playReadyDrm.build() : null;
        }

        public final String getStatus() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final String getStatusDetail() {
            return statusDetail;
        }

        @Override
        public final Builder statusDetail(String statusDetail) {
            this.statusDetail = statusDetail;
            return this;
        }

        public final void setStatusDetail(String statusDetail) {
            this.statusDetail = statusDetail;
        }

        @Override
        public Playlist build() {
            return new Playlist(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
