/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lambda;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.lambda.model.AddPermissionRequest;
import software.amazon.awssdk.services.lambda.model.AddPermissionResponse;
import software.amazon.awssdk.services.lambda.model.CodeStorageExceededException;
import software.amazon.awssdk.services.lambda.model.CreateAliasRequest;
import software.amazon.awssdk.services.lambda.model.CreateAliasResponse;
import software.amazon.awssdk.services.lambda.model.CreateEventSourceMappingRequest;
import software.amazon.awssdk.services.lambda.model.CreateEventSourceMappingResponse;
import software.amazon.awssdk.services.lambda.model.CreateFunctionRequest;
import software.amazon.awssdk.services.lambda.model.CreateFunctionResponse;
import software.amazon.awssdk.services.lambda.model.DeleteAliasRequest;
import software.amazon.awssdk.services.lambda.model.DeleteAliasResponse;
import software.amazon.awssdk.services.lambda.model.DeleteEventSourceMappingRequest;
import software.amazon.awssdk.services.lambda.model.DeleteEventSourceMappingResponse;
import software.amazon.awssdk.services.lambda.model.DeleteFunctionConcurrencyRequest;
import software.amazon.awssdk.services.lambda.model.DeleteFunctionConcurrencyResponse;
import software.amazon.awssdk.services.lambda.model.DeleteFunctionRequest;
import software.amazon.awssdk.services.lambda.model.DeleteFunctionResponse;
import software.amazon.awssdk.services.lambda.model.Ec2AccessDeniedException;
import software.amazon.awssdk.services.lambda.model.Ec2ThrottledException;
import software.amazon.awssdk.services.lambda.model.Ec2UnexpectedException;
import software.amazon.awssdk.services.lambda.model.EniLimitReachedException;
import software.amazon.awssdk.services.lambda.model.GetAccountSettingsRequest;
import software.amazon.awssdk.services.lambda.model.GetAccountSettingsResponse;
import software.amazon.awssdk.services.lambda.model.GetAliasRequest;
import software.amazon.awssdk.services.lambda.model.GetAliasResponse;
import software.amazon.awssdk.services.lambda.model.GetEventSourceMappingRequest;
import software.amazon.awssdk.services.lambda.model.GetEventSourceMappingResponse;
import software.amazon.awssdk.services.lambda.model.GetFunctionConfigurationRequest;
import software.amazon.awssdk.services.lambda.model.GetFunctionConfigurationResponse;
import software.amazon.awssdk.services.lambda.model.GetFunctionRequest;
import software.amazon.awssdk.services.lambda.model.GetFunctionResponse;
import software.amazon.awssdk.services.lambda.model.GetPolicyRequest;
import software.amazon.awssdk.services.lambda.model.GetPolicyResponse;
import software.amazon.awssdk.services.lambda.model.InvalidParameterValueException;
import software.amazon.awssdk.services.lambda.model.InvalidRequestContentException;
import software.amazon.awssdk.services.lambda.model.InvalidRuntimeException;
import software.amazon.awssdk.services.lambda.model.InvalidSecurityGroupIdException;
import software.amazon.awssdk.services.lambda.model.InvalidSubnetIdException;
import software.amazon.awssdk.services.lambda.model.InvalidZipFileException;
import software.amazon.awssdk.services.lambda.model.InvokeRequest;
import software.amazon.awssdk.services.lambda.model.InvokeResponse;
import software.amazon.awssdk.services.lambda.model.KmsAccessDeniedException;
import software.amazon.awssdk.services.lambda.model.KmsDisabledException;
import software.amazon.awssdk.services.lambda.model.KmsInvalidStateException;
import software.amazon.awssdk.services.lambda.model.KmsNotFoundException;
import software.amazon.awssdk.services.lambda.model.LambdaException;
import software.amazon.awssdk.services.lambda.model.LambdaRequest;
import software.amazon.awssdk.services.lambda.model.ListAliasesRequest;
import software.amazon.awssdk.services.lambda.model.ListAliasesResponse;
import software.amazon.awssdk.services.lambda.model.ListEventSourceMappingsRequest;
import software.amazon.awssdk.services.lambda.model.ListEventSourceMappingsResponse;
import software.amazon.awssdk.services.lambda.model.ListFunctionsRequest;
import software.amazon.awssdk.services.lambda.model.ListFunctionsResponse;
import software.amazon.awssdk.services.lambda.model.ListTagsRequest;
import software.amazon.awssdk.services.lambda.model.ListTagsResponse;
import software.amazon.awssdk.services.lambda.model.ListVersionsByFunctionRequest;
import software.amazon.awssdk.services.lambda.model.ListVersionsByFunctionResponse;
import software.amazon.awssdk.services.lambda.model.PolicyLengthExceededException;
import software.amazon.awssdk.services.lambda.model.PreconditionFailedException;
import software.amazon.awssdk.services.lambda.model.PublishVersionRequest;
import software.amazon.awssdk.services.lambda.model.PublishVersionResponse;
import software.amazon.awssdk.services.lambda.model.PutFunctionConcurrencyRequest;
import software.amazon.awssdk.services.lambda.model.PutFunctionConcurrencyResponse;
import software.amazon.awssdk.services.lambda.model.RemovePermissionRequest;
import software.amazon.awssdk.services.lambda.model.RemovePermissionResponse;
import software.amazon.awssdk.services.lambda.model.RequestTooLargeException;
import software.amazon.awssdk.services.lambda.model.ResourceConflictException;
import software.amazon.awssdk.services.lambda.model.ResourceInUseException;
import software.amazon.awssdk.services.lambda.model.ResourceNotFoundException;
import software.amazon.awssdk.services.lambda.model.ServiceException;
import software.amazon.awssdk.services.lambda.model.SubnetIpAddressLimitReachedException;
import software.amazon.awssdk.services.lambda.model.TagResourceRequest;
import software.amazon.awssdk.services.lambda.model.TagResourceResponse;
import software.amazon.awssdk.services.lambda.model.TooManyRequestsException;
import software.amazon.awssdk.services.lambda.model.UnsupportedMediaTypeException;
import software.amazon.awssdk.services.lambda.model.UntagResourceRequest;
import software.amazon.awssdk.services.lambda.model.UntagResourceResponse;
import software.amazon.awssdk.services.lambda.model.UpdateAliasRequest;
import software.amazon.awssdk.services.lambda.model.UpdateAliasResponse;
import software.amazon.awssdk.services.lambda.model.UpdateEventSourceMappingRequest;
import software.amazon.awssdk.services.lambda.model.UpdateEventSourceMappingResponse;
import software.amazon.awssdk.services.lambda.model.UpdateFunctionCodeRequest;
import software.amazon.awssdk.services.lambda.model.UpdateFunctionCodeResponse;
import software.amazon.awssdk.services.lambda.model.UpdateFunctionConfigurationRequest;
import software.amazon.awssdk.services.lambda.model.UpdateFunctionConfigurationResponse;
import software.amazon.awssdk.services.lambda.paginators.ListEventSourceMappingsIterable;
import software.amazon.awssdk.services.lambda.paginators.ListFunctionsIterable;
import software.amazon.awssdk.services.lambda.transform.AddPermissionRequestMarshaller;
import software.amazon.awssdk.services.lambda.transform.CreateAliasRequestMarshaller;
import software.amazon.awssdk.services.lambda.transform.CreateEventSourceMappingRequestMarshaller;
import software.amazon.awssdk.services.lambda.transform.CreateFunctionRequestMarshaller;
import software.amazon.awssdk.services.lambda.transform.DeleteAliasRequestMarshaller;
import software.amazon.awssdk.services.lambda.transform.DeleteEventSourceMappingRequestMarshaller;
import software.amazon.awssdk.services.lambda.transform.DeleteFunctionConcurrencyRequestMarshaller;
import software.amazon.awssdk.services.lambda.transform.DeleteFunctionRequestMarshaller;
import software.amazon.awssdk.services.lambda.transform.GetAccountSettingsRequestMarshaller;
import software.amazon.awssdk.services.lambda.transform.GetAliasRequestMarshaller;
import software.amazon.awssdk.services.lambda.transform.GetEventSourceMappingRequestMarshaller;
import software.amazon.awssdk.services.lambda.transform.GetFunctionConfigurationRequestMarshaller;
import software.amazon.awssdk.services.lambda.transform.GetFunctionRequestMarshaller;
import software.amazon.awssdk.services.lambda.transform.GetPolicyRequestMarshaller;
import software.amazon.awssdk.services.lambda.transform.InvokeRequestMarshaller;
import software.amazon.awssdk.services.lambda.transform.ListAliasesRequestMarshaller;
import software.amazon.awssdk.services.lambda.transform.ListEventSourceMappingsRequestMarshaller;
import software.amazon.awssdk.services.lambda.transform.ListFunctionsRequestMarshaller;
import software.amazon.awssdk.services.lambda.transform.ListTagsRequestMarshaller;
import software.amazon.awssdk.services.lambda.transform.ListVersionsByFunctionRequestMarshaller;
import software.amazon.awssdk.services.lambda.transform.PublishVersionRequestMarshaller;
import software.amazon.awssdk.services.lambda.transform.PutFunctionConcurrencyRequestMarshaller;
import software.amazon.awssdk.services.lambda.transform.RemovePermissionRequestMarshaller;
import software.amazon.awssdk.services.lambda.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.lambda.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.lambda.transform.UpdateAliasRequestMarshaller;
import software.amazon.awssdk.services.lambda.transform.UpdateEventSourceMappingRequestMarshaller;
import software.amazon.awssdk.services.lambda.transform.UpdateFunctionCodeRequestMarshaller;
import software.amazon.awssdk.services.lambda.transform.UpdateFunctionConfigurationRequestMarshaller;

/**
 * Internal implementation of {@link LambdaClient}.
 *
 * @see LambdaClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultLambdaClient implements LambdaClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultLambdaClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Adds a permission to the resource policy associated with the specified AWS Lambda function. You use resource
     * policies to grant permissions to event sources that use the <i>push</i> model. In a <i>push</i> model, event
     * sources (such as Amazon S3 and custom applications) invoke your Lambda function. Each permission you add to the
     * resource policy allows an event source permission to invoke the Lambda function.
     * </p>
     * <p>
     * Permissions apply to the Amazon Resource Name (ARN) used to invoke the function, which can be unqualified (the
     * unpublished version of the function), or include a version or alias. If a client uses a version or alias to
     * invoke a function, use the <code>Qualifier</code> parameter to apply permissions to that ARN. For more
     * information about versioning, see <a
     * href="http://docs.aws.amazon.com/lambda/latest/dg/versioning-aliases.html">AWS Lambda Function Versioning and
     * Aliases</a>.
     * </p>
     * <p>
     * This operation requires permission for the <code>lambda:AddPermission</code> action.
     * </p>
     *
     * @param addPermissionRequest
     * @return Result of the AddPermission operation returned by the service.
     * @throws ServiceException
     *         The AWS Lambda service encountered an internal error.
     * @throws ResourceNotFoundException
     *         The resource (for example, a Lambda function or access policy statement) specified in the request does
     *         not exist.
     * @throws ResourceConflictException
     *         The resource already exists.
     * @throws InvalidParameterValueException
     *         One of the parameters in the request is invalid. For example, if you provided an IAM role for AWS Lambda
     *         to assume in the <code>CreateFunction</code> or the <code>UpdateFunctionConfiguration</code> API, that
     *         AWS Lambda is unable to assume you will get this exception.
     * @throws PolicyLengthExceededException
     *         Lambda function access policy is limited to 20 KB.
     * @throws TooManyRequestsException
     *         Request throughput limit exceeded
     * @throws PreconditionFailedException
     *         The RevisionId provided does not match the latest RevisionId for the Lambda function or alias. Call the
     *         <code>GetFunction</code> or the <code>GetAlias</code> API to retrieve the latest RevisionId for your
     *         resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LambdaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LambdaClient.AddPermission
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/lambda-2015-03-31/AddPermission" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public AddPermissionResponse addPermission(AddPermissionRequest addPermissionRequest) throws ServiceException,
            ResourceNotFoundException, ResourceConflictException, InvalidParameterValueException, PolicyLengthExceededException,
            TooManyRequestsException, PreconditionFailedException, AwsServiceException, SdkClientException, LambdaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AddPermissionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                AddPermissionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<AddPermissionRequest, AddPermissionResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(addPermissionRequest).withMarshaller(new AddPermissionRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Creates an alias that points to the specified Lambda function version. For more information, see <a
     * href="http://docs.aws.amazon.com/lambda/latest/dg/aliases-intro.html">Introduction to AWS Lambda Aliases</a>.
     * </p>
     * <p>
     * Alias names are unique for a given function. This requires permission for the lambda:CreateAlias action.
     * </p>
     *
     * @param createAliasRequest
     * @return Result of the CreateAlias operation returned by the service.
     * @throws ServiceException
     *         The AWS Lambda service encountered an internal error.
     * @throws ResourceNotFoundException
     *         The resource (for example, a Lambda function or access policy statement) specified in the request does
     *         not exist.
     * @throws ResourceConflictException
     *         The resource already exists.
     * @throws InvalidParameterValueException
     *         One of the parameters in the request is invalid. For example, if you provided an IAM role for AWS Lambda
     *         to assume in the <code>CreateFunction</code> or the <code>UpdateFunctionConfiguration</code> API, that
     *         AWS Lambda is unable to assume you will get this exception.
     * @throws TooManyRequestsException
     *         Request throughput limit exceeded
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LambdaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LambdaClient.CreateAlias
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/lambda-2015-03-31/CreateAlias" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateAliasResponse createAlias(CreateAliasRequest createAliasRequest) throws ServiceException,
            ResourceNotFoundException, ResourceConflictException, InvalidParameterValueException, TooManyRequestsException,
            AwsServiceException, SdkClientException, LambdaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateAliasResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateAliasResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<CreateAliasRequest, CreateAliasResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(createAliasRequest).withMarshaller(new CreateAliasRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Identifies a poll-based event source for a Lambda function. It can be either an Amazon Kinesis or DynamoDB
     * stream. AWS Lambda invokes the specified function when records are posted to the event source.
     * </p>
     * <p>
     * This association between a poll-based source and a Lambda function is called the event source mapping.
     * </p>
     * <p>
     * You provide mapping information (for example, which stream or SQS queue to read from and which Lambda function to
     * invoke) in the request body.
     * </p>
     * <p>
     * Amazon Kinesis or DynamoDB stream event sources can be associated with multiple AWS Lambda functions and a given
     * Lambda function can be associated with multiple AWS event sources. For Amazon SQS, you can configure multiple
     * queues as event sources for a single Lambda function, but an SQS queue can be mapped only to a single Lambda
     * function.
     * </p>
     * <p>
     * You can configure an SQS queue in an account separate from your Lambda function's account. Also the queue needs
     * to reside in the same AWS region as your function.
     * </p>
     * <p>
     * If you are using versioning, you can specify a specific function version or an alias via the function name
     * parameter. For more information about versioning, see <a
     * href="http://docs.aws.amazon.com/lambda/latest/dg/versioning-aliases.html">AWS Lambda Function Versioning and
     * Aliases</a>.
     * </p>
     * <p>
     * This operation requires permission for the <code>lambda:CreateEventSourceMapping</code> action.
     * </p>
     *
     * @param createEventSourceMappingRequest
     * @return Result of the CreateEventSourceMapping operation returned by the service.
     * @throws ServiceException
     *         The AWS Lambda service encountered an internal error.
     * @throws InvalidParameterValueException
     *         One of the parameters in the request is invalid. For example, if you provided an IAM role for AWS Lambda
     *         to assume in the <code>CreateFunction</code> or the <code>UpdateFunctionConfiguration</code> API, that
     *         AWS Lambda is unable to assume you will get this exception.
     * @throws ResourceConflictException
     *         The resource already exists.
     * @throws TooManyRequestsException
     *         Request throughput limit exceeded
     * @throws ResourceNotFoundException
     *         The resource (for example, a Lambda function or access policy statement) specified in the request does
     *         not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LambdaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LambdaClient.CreateEventSourceMapping
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/lambda-2015-03-31/CreateEventSourceMapping"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateEventSourceMappingResponse createEventSourceMapping(
            CreateEventSourceMappingRequest createEventSourceMappingRequest) throws ServiceException,
            InvalidParameterValueException, ResourceConflictException, TooManyRequestsException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, LambdaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateEventSourceMappingResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateEventSourceMappingResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<CreateEventSourceMappingRequest, CreateEventSourceMappingResponse>()
                        .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                        .withInput(createEventSourceMappingRequest)
                        .withMarshaller(new CreateEventSourceMappingRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Creates a new Lambda function. The function configuration is created from the request parameters, and the code
     * for the function is provided by a .zip file. The function name is case-sensitive.
     * </p>
     * <p>
     * This operation requires permission for the <code>lambda:CreateFunction</code> action.
     * </p>
     *
     * @param createFunctionRequest
     * @return Result of the CreateFunction operation returned by the service.
     * @throws ServiceException
     *         The AWS Lambda service encountered an internal error.
     * @throws InvalidParameterValueException
     *         One of the parameters in the request is invalid. For example, if you provided an IAM role for AWS Lambda
     *         to assume in the <code>CreateFunction</code> or the <code>UpdateFunctionConfiguration</code> API, that
     *         AWS Lambda is unable to assume you will get this exception.
     * @throws ResourceNotFoundException
     *         The resource (for example, a Lambda function or access policy statement) specified in the request does
     *         not exist.
     * @throws ResourceConflictException
     *         The resource already exists.
     * @throws TooManyRequestsException
     *         Request throughput limit exceeded
     * @throws CodeStorageExceededException
     *         You have exceeded your maximum total code size per account. <a
     *         href="http://docs.aws.amazon.com/lambda/latest/dg/limits.html">Limits</a>
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LambdaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LambdaClient.CreateFunction
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/lambda-2015-03-31/CreateFunction" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateFunctionResponse createFunction(CreateFunctionRequest createFunctionRequest) throws ServiceException,
            InvalidParameterValueException, ResourceNotFoundException, ResourceConflictException, TooManyRequestsException,
            CodeStorageExceededException, AwsServiceException, SdkClientException, LambdaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateFunctionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateFunctionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<CreateFunctionRequest, CreateFunctionResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(createFunctionRequest).withMarshaller(new CreateFunctionRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes the specified Lambda function alias. For more information, see <a
     * href="http://docs.aws.amazon.com/lambda/latest/dg/aliases-intro.html">Introduction to AWS Lambda Aliases</a>.
     * </p>
     * <p>
     * This requires permission for the lambda:DeleteAlias action.
     * </p>
     *
     * @param deleteAliasRequest
     * @return Result of the DeleteAlias operation returned by the service.
     * @throws ServiceException
     *         The AWS Lambda service encountered an internal error.
     * @throws InvalidParameterValueException
     *         One of the parameters in the request is invalid. For example, if you provided an IAM role for AWS Lambda
     *         to assume in the <code>CreateFunction</code> or the <code>UpdateFunctionConfiguration</code> API, that
     *         AWS Lambda is unable to assume you will get this exception.
     * @throws TooManyRequestsException
     *         Request throughput limit exceeded
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LambdaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LambdaClient.DeleteAlias
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/lambda-2015-03-31/DeleteAlias" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteAliasResponse deleteAlias(DeleteAliasRequest deleteAliasRequest) throws ServiceException,
            InvalidParameterValueException, TooManyRequestsException, AwsServiceException, SdkClientException, LambdaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteAliasResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteAliasResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteAliasRequest, DeleteAliasResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(deleteAliasRequest).withMarshaller(new DeleteAliasRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Removes an event source mapping. This means AWS Lambda will no longer invoke the function for events in the
     * associated source.
     * </p>
     * <p>
     * This operation requires permission for the <code>lambda:DeleteEventSourceMapping</code> action.
     * </p>
     *
     * @param deleteEventSourceMappingRequest
     * @return Result of the DeleteEventSourceMapping operation returned by the service.
     * @throws ServiceException
     *         The AWS Lambda service encountered an internal error.
     * @throws ResourceNotFoundException
     *         The resource (for example, a Lambda function or access policy statement) specified in the request does
     *         not exist.
     * @throws InvalidParameterValueException
     *         One of the parameters in the request is invalid. For example, if you provided an IAM role for AWS Lambda
     *         to assume in the <code>CreateFunction</code> or the <code>UpdateFunctionConfiguration</code> API, that
     *         AWS Lambda is unable to assume you will get this exception.
     * @throws TooManyRequestsException
     *         Request throughput limit exceeded
     * @throws ResourceInUseException
     *         The operation conflicts with the resource's availability. For example, you attempted to update an
     *         EventSoure Mapping in CREATING, or tried to delete a EventSoure mapping currently in the UPDATING state.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LambdaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LambdaClient.DeleteEventSourceMapping
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/lambda-2015-03-31/DeleteEventSourceMapping"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteEventSourceMappingResponse deleteEventSourceMapping(
            DeleteEventSourceMappingRequest deleteEventSourceMappingRequest) throws ServiceException, ResourceNotFoundException,
            InvalidParameterValueException, TooManyRequestsException, ResourceInUseException, AwsServiceException,
            SdkClientException, LambdaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteEventSourceMappingResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteEventSourceMappingResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<DeleteEventSourceMappingRequest, DeleteEventSourceMappingResponse>()
                        .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                        .withInput(deleteEventSourceMappingRequest)
                        .withMarshaller(new DeleteEventSourceMappingRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes a Lambda function. To delete a specific function version, use the <code>Qualifier</code> parameter.
     * Otherwise, all versions and aliases are deleted. Event source mappings are not deleted.
     * </p>
     * <p>
     * This operation requires permission for the <code>lambda:DeleteFunction</code> action.
     * </p>
     *
     * @param deleteFunctionRequest
     * @return Result of the DeleteFunction operation returned by the service.
     * @throws ServiceException
     *         The AWS Lambda service encountered an internal error.
     * @throws ResourceNotFoundException
     *         The resource (for example, a Lambda function or access policy statement) specified in the request does
     *         not exist.
     * @throws TooManyRequestsException
     *         Request throughput limit exceeded
     * @throws InvalidParameterValueException
     *         One of the parameters in the request is invalid. For example, if you provided an IAM role for AWS Lambda
     *         to assume in the <code>CreateFunction</code> or the <code>UpdateFunctionConfiguration</code> API, that
     *         AWS Lambda is unable to assume you will get this exception.
     * @throws ResourceConflictException
     *         The resource already exists.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LambdaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LambdaClient.DeleteFunction
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/lambda-2015-03-31/DeleteFunction" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteFunctionResponse deleteFunction(DeleteFunctionRequest deleteFunctionRequest) throws ServiceException,
            ResourceNotFoundException, TooManyRequestsException, InvalidParameterValueException, ResourceConflictException,
            AwsServiceException, SdkClientException, LambdaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteFunctionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteFunctionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteFunctionRequest, DeleteFunctionResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(deleteFunctionRequest).withMarshaller(new DeleteFunctionRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Removes concurrent execution limits from this function. For more information, see <a
     * href="http://docs.aws.amazon.com/lambda/latest/dg/concurrent-executions.html">Managing Concurrency</a>.
     * </p>
     *
     * @param deleteFunctionConcurrencyRequest
     * @return Result of the DeleteFunctionConcurrency operation returned by the service.
     * @throws ServiceException
     *         The AWS Lambda service encountered an internal error.
     * @throws ResourceNotFoundException
     *         The resource (for example, a Lambda function or access policy statement) specified in the request does
     *         not exist.
     * @throws TooManyRequestsException
     *         Request throughput limit exceeded
     * @throws InvalidParameterValueException
     *         One of the parameters in the request is invalid. For example, if you provided an IAM role for AWS Lambda
     *         to assume in the <code>CreateFunction</code> or the <code>UpdateFunctionConfiguration</code> API, that
     *         AWS Lambda is unable to assume you will get this exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LambdaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LambdaClient.DeleteFunctionConcurrency
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/lambda-2015-03-31/DeleteFunctionConcurrency"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteFunctionConcurrencyResponse deleteFunctionConcurrency(
            DeleteFunctionConcurrencyRequest deleteFunctionConcurrencyRequest) throws ServiceException,
            ResourceNotFoundException, TooManyRequestsException, InvalidParameterValueException, AwsServiceException,
            SdkClientException, LambdaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteFunctionConcurrencyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteFunctionConcurrencyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<DeleteFunctionConcurrencyRequest, DeleteFunctionConcurrencyResponse>()
                        .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                        .withInput(deleteFunctionConcurrencyRequest)
                        .withMarshaller(new DeleteFunctionConcurrencyRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Retrieves details about your account's <a
     * href="http://docs.aws.amazon.com/lambda/latest/dg/limits.html">limits</a> and usage in a region.
     * </p>
     *
     * @param getAccountSettingsRequest
     * @return Result of the GetAccountSettings operation returned by the service.
     * @throws TooManyRequestsException
     *         Request throughput limit exceeded
     * @throws ServiceException
     *         The AWS Lambda service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LambdaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LambdaClient.GetAccountSettings
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/lambda-2015-03-31/GetAccountSettings" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetAccountSettingsResponse getAccountSettings(GetAccountSettingsRequest getAccountSettingsRequest)
            throws TooManyRequestsException, ServiceException, AwsServiceException, SdkClientException, LambdaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetAccountSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetAccountSettingsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetAccountSettingsRequest, GetAccountSettingsResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(getAccountSettingsRequest).withMarshaller(new GetAccountSettingsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns the specified alias information such as the alias ARN, description, and function version it is pointing
     * to. For more information, see <a
     * href="http://docs.aws.amazon.com/lambda/latest/dg/aliases-intro.html">Introduction to AWS Lambda Aliases</a>.
     * </p>
     * <p>
     * This requires permission for the <code>lambda:GetAlias</code> action.
     * </p>
     *
     * @param getAliasRequest
     * @return Result of the GetAlias operation returned by the service.
     * @throws ServiceException
     *         The AWS Lambda service encountered an internal error.
     * @throws ResourceNotFoundException
     *         The resource (for example, a Lambda function or access policy statement) specified in the request does
     *         not exist.
     * @throws InvalidParameterValueException
     *         One of the parameters in the request is invalid. For example, if you provided an IAM role for AWS Lambda
     *         to assume in the <code>CreateFunction</code> or the <code>UpdateFunctionConfiguration</code> API, that
     *         AWS Lambda is unable to assume you will get this exception.
     * @throws TooManyRequestsException
     *         Request throughput limit exceeded
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LambdaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LambdaClient.GetAlias
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/lambda-2015-03-31/GetAlias" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetAliasResponse getAlias(GetAliasRequest getAliasRequest) throws ServiceException, ResourceNotFoundException,
            InvalidParameterValueException, TooManyRequestsException, AwsServiceException, SdkClientException, LambdaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetAliasResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetAliasResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetAliasRequest, GetAliasResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(getAliasRequest)
                .withMarshaller(new GetAliasRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns configuration information for the specified event source mapping (see <a>CreateEventSourceMapping</a>).
     * </p>
     * <p>
     * This operation requires permission for the <code>lambda:GetEventSourceMapping</code> action.
     * </p>
     *
     * @param getEventSourceMappingRequest
     * @return Result of the GetEventSourceMapping operation returned by the service.
     * @throws ServiceException
     *         The AWS Lambda service encountered an internal error.
     * @throws ResourceNotFoundException
     *         The resource (for example, a Lambda function or access policy statement) specified in the request does
     *         not exist.
     * @throws InvalidParameterValueException
     *         One of the parameters in the request is invalid. For example, if you provided an IAM role for AWS Lambda
     *         to assume in the <code>CreateFunction</code> or the <code>UpdateFunctionConfiguration</code> API, that
     *         AWS Lambda is unable to assume you will get this exception.
     * @throws TooManyRequestsException
     *         Request throughput limit exceeded
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LambdaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LambdaClient.GetEventSourceMapping
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/lambda-2015-03-31/GetEventSourceMapping" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetEventSourceMappingResponse getEventSourceMapping(GetEventSourceMappingRequest getEventSourceMappingRequest)
            throws ServiceException, ResourceNotFoundException, InvalidParameterValueException, TooManyRequestsException,
            AwsServiceException, SdkClientException, LambdaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetEventSourceMappingResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetEventSourceMappingResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetEventSourceMappingRequest, GetEventSourceMappingResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(getEventSourceMappingRequest)
                .withMarshaller(new GetEventSourceMappingRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns the configuration information of the Lambda function and a presigned URL link to the .zip file you
     * uploaded with <a>CreateFunction</a> so you can download the .zip file. Note that the URL is valid for up to 10
     * minutes. The configuration information is the same information you provided as parameters when uploading the
     * function.
     * </p>
     * <p>
     * Use the <code>Qualifier</code> parameter to retrieve a published version of the function. Otherwise, returns the
     * unpublished version (<code>&#36LATEST</code>). For more information, see <a
     * href="http://docs.aws.amazon.com/lambda/latest/dg/versioning-aliases.html">AWS Lambda Function Versioning and
     * Aliases</a>.
     * </p>
     * <p>
     * This operation requires permission for the <code>lambda:GetFunction</code> action.
     * </p>
     *
     * @param getFunctionRequest
     * @return Result of the GetFunction operation returned by the service.
     * @throws ServiceException
     *         The AWS Lambda service encountered an internal error.
     * @throws ResourceNotFoundException
     *         The resource (for example, a Lambda function or access policy statement) specified in the request does
     *         not exist.
     * @throws TooManyRequestsException
     *         Request throughput limit exceeded
     * @throws InvalidParameterValueException
     *         One of the parameters in the request is invalid. For example, if you provided an IAM role for AWS Lambda
     *         to assume in the <code>CreateFunction</code> or the <code>UpdateFunctionConfiguration</code> API, that
     *         AWS Lambda is unable to assume you will get this exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LambdaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LambdaClient.GetFunction
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/lambda-2015-03-31/GetFunction" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetFunctionResponse getFunction(GetFunctionRequest getFunctionRequest) throws ServiceException,
            ResourceNotFoundException, TooManyRequestsException, InvalidParameterValueException, AwsServiceException,
            SdkClientException, LambdaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetFunctionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetFunctionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetFunctionRequest, GetFunctionResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(getFunctionRequest).withMarshaller(new GetFunctionRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns the configuration information of the Lambda function. This the same information you provided as
     * parameters when uploading the function by using <a>CreateFunction</a>.
     * </p>
     * <p>
     * If you are using the versioning feature, you can retrieve this information for a specific function version by
     * using the optional <code>Qualifier</code> parameter and specifying the function version or alias that points to
     * it. If you don't provide it, the API returns information about the &#36LATEST version of the function. For more
     * information about versioning, see <a
     * href="http://docs.aws.amazon.com/lambda/latest/dg/versioning-aliases.html">AWS Lambda Function Versioning and
     * Aliases</a>.
     * </p>
     * <p>
     * This operation requires permission for the <code>lambda:GetFunctionConfiguration</code> operation.
     * </p>
     *
     * @param getFunctionConfigurationRequest
     * @return Result of the GetFunctionConfiguration operation returned by the service.
     * @throws ServiceException
     *         The AWS Lambda service encountered an internal error.
     * @throws ResourceNotFoundException
     *         The resource (for example, a Lambda function or access policy statement) specified in the request does
     *         not exist.
     * @throws TooManyRequestsException
     *         Request throughput limit exceeded
     * @throws InvalidParameterValueException
     *         One of the parameters in the request is invalid. For example, if you provided an IAM role for AWS Lambda
     *         to assume in the <code>CreateFunction</code> or the <code>UpdateFunctionConfiguration</code> API, that
     *         AWS Lambda is unable to assume you will get this exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LambdaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LambdaClient.GetFunctionConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/lambda-2015-03-31/GetFunctionConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetFunctionConfigurationResponse getFunctionConfiguration(
            GetFunctionConfigurationRequest getFunctionConfigurationRequest) throws ServiceException, ResourceNotFoundException,
            TooManyRequestsException, InvalidParameterValueException, AwsServiceException, SdkClientException, LambdaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetFunctionConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetFunctionConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<GetFunctionConfigurationRequest, GetFunctionConfigurationResponse>()
                        .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                        .withInput(getFunctionConfigurationRequest)
                        .withMarshaller(new GetFunctionConfigurationRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns the resource policy associated with the specified Lambda function.
     * </p>
     * <p>
     * This action requires permission for the <code>lambda:GetPolicy action.</code>
     * </p>
     *
     * @param getPolicyRequest
     * @return Result of the GetPolicy operation returned by the service.
     * @throws ServiceException
     *         The AWS Lambda service encountered an internal error.
     * @throws ResourceNotFoundException
     *         The resource (for example, a Lambda function or access policy statement) specified in the request does
     *         not exist.
     * @throws TooManyRequestsException
     *         Request throughput limit exceeded
     * @throws InvalidParameterValueException
     *         One of the parameters in the request is invalid. For example, if you provided an IAM role for AWS Lambda
     *         to assume in the <code>CreateFunction</code> or the <code>UpdateFunctionConfiguration</code> API, that
     *         AWS Lambda is unable to assume you will get this exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LambdaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LambdaClient.GetPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/lambda-2015-03-31/GetPolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetPolicyResponse getPolicy(GetPolicyRequest getPolicyRequest) throws ServiceException, ResourceNotFoundException,
            TooManyRequestsException, InvalidParameterValueException, AwsServiceException, SdkClientException, LambdaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetPolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetPolicyRequest, GetPolicyResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(getPolicyRequest)
                .withMarshaller(new GetPolicyRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Invokes a Lambda function. For an example, see <a
     * href="http://docs.aws.amazon.com/lambda/latest/dg/with-dynamodb-create-function.html#with-dbb-invoke-manually"
     * >Create the Lambda Function and Test It Manually</a>.
     * </p>
     * <p>
     * Specify just a function name to invoke the latest version of the function. To invoke a published version, use the
     * <code>Qualifier</code> parameter to specify a <a
     * href="http://docs.aws.amazon.com/lambda/latest/dg/versioning-aliases.html">version or alias</a>.
     * </p>
     * <p>
     * If you use the <code>RequestResponse</code> (synchronous) invocation option, the function will be invoked only
     * once. If you use the <code>Event</code> (asynchronous) invocation option, the function will be invoked at least
     * once in response to an event and the function must be idempotent to handle this.
     * </p>
     * <p>
     * For functions with a long timeout, your client may be disconnected during synchronous invocation while it waits
     * for a response. Configure your HTTP client, SDK, firewall, proxy, or operating system to allow for long
     * connections with timeout or keep-alive settings.
     * </p>
     * <p>
     * This operation requires permission for the <code>lambda:InvokeFunction</code> action.
     * </p>
     * <p>
     * The <code>TooManyRequestsException</code> noted below will return the following:
     * <code>ConcurrentInvocationLimitExceeded</code> will be returned if you have no functions with reserved
     * concurrency and have exceeded your account concurrent limit or if a function without reserved concurrency exceeds
     * the account's unreserved concurrency limit. <code>ReservedFunctionConcurrentInvocationLimitExceeded</code> will
     * be returned when a function with reserved concurrency exceeds its configured concurrency limit.
     * </p>
     *
     * @param invokeRequest
     * @return Result of the Invoke operation returned by the service.
     * @throws ServiceException
     *         The AWS Lambda service encountered an internal error.
     * @throws ResourceNotFoundException
     *         The resource (for example, a Lambda function or access policy statement) specified in the request does
     *         not exist.
     * @throws InvalidRequestContentException
     *         The request body could not be parsed as JSON.
     * @throws RequestTooLargeException
     *         The request payload exceeded the <code>Invoke</code> request body JSON input limit. For more information,
     *         see <a href="http://docs.aws.amazon.com/lambda/latest/dg/limits.html">Limits</a>.
     * @throws UnsupportedMediaTypeException
     *         The content type of the <code>Invoke</code> request body is not JSON.
     * @throws TooManyRequestsException
     *         Request throughput limit exceeded
     * @throws InvalidParameterValueException
     *         One of the parameters in the request is invalid. For example, if you provided an IAM role for AWS Lambda
     *         to assume in the <code>CreateFunction</code> or the <code>UpdateFunctionConfiguration</code> API, that
     *         AWS Lambda is unable to assume you will get this exception.
     * @throws Ec2UnexpectedException
     *         AWS Lambda received an unexpected EC2 client exception while setting up for the Lambda function.
     * @throws SubnetIpAddressLimitReachedException
     *         AWS Lambda was not able to set up VPC access for the Lambda function because one or more configured
     *         subnets has no available IP addresses.
     * @throws EniLimitReachedException
     *         AWS Lambda was not able to create an Elastic Network Interface (ENI) in the VPC, specified as part of
     *         Lambda function configuration, because the limit for network interfaces has been reached.
     * @throws Ec2ThrottledException
     *         AWS Lambda was throttled by Amazon EC2 during Lambda function initialization using the execution role
     *         provided for the Lambda function.
     * @throws Ec2AccessDeniedException
     *         Need additional permissions to configure VPC settings.
     * @throws InvalidSubnetIdException
     *         The Subnet ID provided in the Lambda function VPC configuration is invalid.
     * @throws InvalidSecurityGroupIdException
     *         The Security Group ID provided in the Lambda function VPC configuration is invalid.
     * @throws InvalidZipFileException
     *         AWS Lambda could not unzip the deployment package.
     * @throws KmsDisabledException
     *         Lambda was unable to decrypt the environment variables because the KMS key used is disabled. Check the
     *         Lambda function's KMS key settings.
     * @throws KmsInvalidStateException
     *         Lambda was unable to decrypt the environment variables because the KMS key used is in an invalid state
     *         for Decrypt. Check the function's KMS key settings.
     * @throws KmsAccessDeniedException
     *         Lambda was unable to decrypt the environment variables because KMS access was denied. Check the Lambda
     *         function's KMS permissions.
     * @throws KmsNotFoundException
     *         Lambda was unable to decrypt the environment variables because the KMS key was not found. Check the
     *         function's KMS key settings.
     * @throws InvalidRuntimeException
     *         The runtime or runtime version specified is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LambdaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LambdaClient.Invoke
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/lambda-2015-03-31/Invoke" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public InvokeResponse invoke(InvokeRequest invokeRequest) throws ServiceException, ResourceNotFoundException,
            InvalidRequestContentException, RequestTooLargeException, UnsupportedMediaTypeException, TooManyRequestsException,
            InvalidParameterValueException, Ec2UnexpectedException, SubnetIpAddressLimitReachedException,
            EniLimitReachedException, Ec2ThrottledException, Ec2AccessDeniedException, InvalidSubnetIdException,
            InvalidSecurityGroupIdException, InvalidZipFileException, KmsDisabledException, KmsInvalidStateException,
            KmsAccessDeniedException, KmsNotFoundException, InvalidRuntimeException, AwsServiceException, SdkClientException,
            LambdaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(false).build();

        HttpResponseHandler<InvokeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                InvokeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<InvokeRequest, InvokeResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(invokeRequest)
                .withMarshaller(new InvokeRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns list of aliases created for a Lambda function. For each alias, the response includes information such as
     * the alias ARN, description, alias name, and the function version to which it points. For more information, see <a
     * href="http://docs.aws.amazon.com/lambda/latest/dg/aliases-intro.html">Introduction to AWS Lambda Aliases</a>.
     * </p>
     * <p>
     * This requires permission for the lambda:ListAliases action.
     * </p>
     *
     * @param listAliasesRequest
     * @return Result of the ListAliases operation returned by the service.
     * @throws ServiceException
     *         The AWS Lambda service encountered an internal error.
     * @throws ResourceNotFoundException
     *         The resource (for example, a Lambda function or access policy statement) specified in the request does
     *         not exist.
     * @throws InvalidParameterValueException
     *         One of the parameters in the request is invalid. For example, if you provided an IAM role for AWS Lambda
     *         to assume in the <code>CreateFunction</code> or the <code>UpdateFunctionConfiguration</code> API, that
     *         AWS Lambda is unable to assume you will get this exception.
     * @throws TooManyRequestsException
     *         Request throughput limit exceeded
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LambdaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LambdaClient.ListAliases
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/lambda-2015-03-31/ListAliases" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListAliasesResponse listAliases(ListAliasesRequest listAliasesRequest) throws ServiceException,
            ResourceNotFoundException, InvalidParameterValueException, TooManyRequestsException, AwsServiceException,
            SdkClientException, LambdaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListAliasesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListAliasesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListAliasesRequest, ListAliasesResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(listAliasesRequest).withMarshaller(new ListAliasesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns a list of event source mappings you created using the <code>CreateEventSourceMapping</code> (see
     * <a>CreateEventSourceMapping</a>).
     * </p>
     * <p>
     * For each mapping, the API returns configuration information. You can optionally specify filters to retrieve
     * specific event source mappings.
     * </p>
     * <p>
     * This operation requires permission for the <code>lambda:ListEventSourceMappings</code> action.
     * </p>
     *
     * @param listEventSourceMappingsRequest
     * @return Result of the ListEventSourceMappings operation returned by the service.
     * @throws ServiceException
     *         The AWS Lambda service encountered an internal error.
     * @throws ResourceNotFoundException
     *         The resource (for example, a Lambda function or access policy statement) specified in the request does
     *         not exist.
     * @throws InvalidParameterValueException
     *         One of the parameters in the request is invalid. For example, if you provided an IAM role for AWS Lambda
     *         to assume in the <code>CreateFunction</code> or the <code>UpdateFunctionConfiguration</code> API, that
     *         AWS Lambda is unable to assume you will get this exception.
     * @throws TooManyRequestsException
     *         Request throughput limit exceeded
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LambdaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LambdaClient.ListEventSourceMappings
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/lambda-2015-03-31/ListEventSourceMappings" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListEventSourceMappingsResponse listEventSourceMappings(ListEventSourceMappingsRequest listEventSourceMappingsRequest)
            throws ServiceException, ResourceNotFoundException, InvalidParameterValueException, TooManyRequestsException,
            AwsServiceException, SdkClientException, LambdaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListEventSourceMappingsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListEventSourceMappingsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListEventSourceMappingsRequest, ListEventSourceMappingsResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(listEventSourceMappingsRequest)
                .withMarshaller(new ListEventSourceMappingsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns a list of event source mappings you created using the <code>CreateEventSourceMapping</code> (see
     * <a>CreateEventSourceMapping</a>).
     * </p>
     * <p>
     * For each mapping, the API returns configuration information. You can optionally specify filters to retrieve
     * specific event source mappings.
     * </p>
     * <p>
     * This operation requires permission for the <code>lambda:ListEventSourceMappings</code> action.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listEventSourceMappings(software.amazon.awssdk.services.lambda.model.ListEventSourceMappingsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.lambda.paginators.ListEventSourceMappingsIterable responses = client.listEventSourceMappingsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.lambda.paginators.ListEventSourceMappingsIterable responses = client
     *             .listEventSourceMappingsPaginator(request);
     *     for (software.amazon.awssdk.services.lambda.model.ListEventSourceMappingsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.lambda.paginators.ListEventSourceMappingsIterable responses = client.listEventSourceMappingsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listEventSourceMappings(software.amazon.awssdk.services.lambda.model.ListEventSourceMappingsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listEventSourceMappingsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ServiceException
     *         The AWS Lambda service encountered an internal error.
     * @throws ResourceNotFoundException
     *         The resource (for example, a Lambda function or access policy statement) specified in the request does
     *         not exist.
     * @throws InvalidParameterValueException
     *         One of the parameters in the request is invalid. For example, if you provided an IAM role for AWS Lambda
     *         to assume in the <code>CreateFunction</code> or the <code>UpdateFunctionConfiguration</code> API, that
     *         AWS Lambda is unable to assume you will get this exception.
     * @throws TooManyRequestsException
     *         Request throughput limit exceeded
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LambdaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LambdaClient.ListEventSourceMappings
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/lambda-2015-03-31/ListEventSourceMappings" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListEventSourceMappingsIterable listEventSourceMappingsPaginator(
            ListEventSourceMappingsRequest listEventSourceMappingsRequest) throws ServiceException, ResourceNotFoundException,
            InvalidParameterValueException, TooManyRequestsException, AwsServiceException, SdkClientException, LambdaException {
        return new ListEventSourceMappingsIterable(this, applyPaginatorUserAgent(listEventSourceMappingsRequest));
    }

    /**
     * <p>
     * Returns a list of your Lambda functions. For each function, the response includes the function configuration
     * information. You must use <a>GetFunction</a> to retrieve the code for your function.
     * </p>
     * <p>
     * This operation requires permission for the <code>lambda:ListFunctions</code> action.
     * </p>
     * <p>
     * If you are using the versioning feature, you can list all of your functions or only <code>&#36LATEST</code>
     * versions. For information about the versioning feature, see <a
     * href="http://docs.aws.amazon.com/lambda/latest/dg/versioning-aliases.html">AWS Lambda Function Versioning and
     * Aliases</a>.
     * </p>
     *
     * @param listFunctionsRequest
     * @return Result of the ListFunctions operation returned by the service.
     * @throws ServiceException
     *         The AWS Lambda service encountered an internal error.
     * @throws TooManyRequestsException
     *         Request throughput limit exceeded
     * @throws InvalidParameterValueException
     *         One of the parameters in the request is invalid. For example, if you provided an IAM role for AWS Lambda
     *         to assume in the <code>CreateFunction</code> or the <code>UpdateFunctionConfiguration</code> API, that
     *         AWS Lambda is unable to assume you will get this exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LambdaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LambdaClient.ListFunctions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/lambda-2015-03-31/ListFunctions" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListFunctionsResponse listFunctions(ListFunctionsRequest listFunctionsRequest) throws ServiceException,
            TooManyRequestsException, InvalidParameterValueException, AwsServiceException, SdkClientException, LambdaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListFunctionsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListFunctionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListFunctionsRequest, ListFunctionsResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(listFunctionsRequest).withMarshaller(new ListFunctionsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns a list of your Lambda functions. For each function, the response includes the function configuration
     * information. You must use <a>GetFunction</a> to retrieve the code for your function.
     * </p>
     * <p>
     * This operation requires permission for the <code>lambda:ListFunctions</code> action.
     * </p>
     * <p>
     * If you are using the versioning feature, you can list all of your functions or only <code>&#36LATEST</code>
     * versions. For information about the versioning feature, see <a
     * href="http://docs.aws.amazon.com/lambda/latest/dg/versioning-aliases.html">AWS Lambda Function Versioning and
     * Aliases</a>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listFunctions(software.amazon.awssdk.services.lambda.model.ListFunctionsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.lambda.paginators.ListFunctionsIterable responses = client.listFunctionsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.lambda.paginators.ListFunctionsIterable responses = client.listFunctionsPaginator(request);
     *     for (software.amazon.awssdk.services.lambda.model.ListFunctionsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.lambda.paginators.ListFunctionsIterable responses = client.listFunctionsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listFunctions(software.amazon.awssdk.services.lambda.model.ListFunctionsRequest)} operation.</b>
     * </p>
     *
     * @param listFunctionsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ServiceException
     *         The AWS Lambda service encountered an internal error.
     * @throws TooManyRequestsException
     *         Request throughput limit exceeded
     * @throws InvalidParameterValueException
     *         One of the parameters in the request is invalid. For example, if you provided an IAM role for AWS Lambda
     *         to assume in the <code>CreateFunction</code> or the <code>UpdateFunctionConfiguration</code> API, that
     *         AWS Lambda is unable to assume you will get this exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LambdaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LambdaClient.ListFunctions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/lambda-2015-03-31/ListFunctions" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListFunctionsIterable listFunctionsPaginator(ListFunctionsRequest listFunctionsRequest) throws ServiceException,
            TooManyRequestsException, InvalidParameterValueException, AwsServiceException, SdkClientException, LambdaException {
        return new ListFunctionsIterable(this, applyPaginatorUserAgent(listFunctionsRequest));
    }

    /**
     * <p>
     * Returns a list of tags assigned to a function when supplied the function ARN (Amazon Resource Name). For more
     * information on Tagging, see <a href="http://docs.aws.amazon.com/lambda/latest/dg/tagging.html">Tagging Lambda
     * Functions</a> in the <b>AWS Lambda Developer Guide</b>.
     * </p>
     *
     * @param listTagsRequest
     * @return Result of the ListTags operation returned by the service.
     * @throws ServiceException
     *         The AWS Lambda service encountered an internal error.
     * @throws ResourceNotFoundException
     *         The resource (for example, a Lambda function or access policy statement) specified in the request does
     *         not exist.
     * @throws InvalidParameterValueException
     *         One of the parameters in the request is invalid. For example, if you provided an IAM role for AWS Lambda
     *         to assume in the <code>CreateFunction</code> or the <code>UpdateFunctionConfiguration</code> API, that
     *         AWS Lambda is unable to assume you will get this exception.
     * @throws TooManyRequestsException
     *         Request throughput limit exceeded
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LambdaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LambdaClient.ListTags
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/lambda-2015-03-31/ListTags" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTagsResponse listTags(ListTagsRequest listTagsRequest) throws ServiceException, ResourceNotFoundException,
            InvalidParameterValueException, TooManyRequestsException, AwsServiceException, SdkClientException, LambdaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListTagsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListTagsRequest, ListTagsResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(listTagsRequest)
                .withMarshaller(new ListTagsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists all versions of a function. For information about versioning, see <a
     * href="http://docs.aws.amazon.com/lambda/latest/dg/versioning-aliases.html">AWS Lambda Function Versioning and
     * Aliases</a>.
     * </p>
     *
     * @param listVersionsByFunctionRequest
     * @return Result of the ListVersionsByFunction operation returned by the service.
     * @throws ServiceException
     *         The AWS Lambda service encountered an internal error.
     * @throws ResourceNotFoundException
     *         The resource (for example, a Lambda function or access policy statement) specified in the request does
     *         not exist.
     * @throws InvalidParameterValueException
     *         One of the parameters in the request is invalid. For example, if you provided an IAM role for AWS Lambda
     *         to assume in the <code>CreateFunction</code> or the <code>UpdateFunctionConfiguration</code> API, that
     *         AWS Lambda is unable to assume you will get this exception.
     * @throws TooManyRequestsException
     *         Request throughput limit exceeded
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LambdaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LambdaClient.ListVersionsByFunction
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/lambda-2015-03-31/ListVersionsByFunction" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListVersionsByFunctionResponse listVersionsByFunction(ListVersionsByFunctionRequest listVersionsByFunctionRequest)
            throws ServiceException, ResourceNotFoundException, InvalidParameterValueException, TooManyRequestsException,
            AwsServiceException, SdkClientException, LambdaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListVersionsByFunctionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListVersionsByFunctionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListVersionsByFunctionRequest, ListVersionsByFunctionResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(listVersionsByFunctionRequest)
                .withMarshaller(new ListVersionsByFunctionRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Publishes a version of your function from the current snapshot of &#36LATEST. That is, AWS Lambda takes a
     * snapshot of the function code and configuration information from &#36LATEST and publishes a new version. The code
     * and configuration cannot be modified after publication. For information about the versioning feature, see <a
     * href="http://docs.aws.amazon.com/lambda/latest/dg/versioning-aliases.html">AWS Lambda Function Versioning and
     * Aliases</a>.
     * </p>
     *
     * @param publishVersionRequest
     * @return Result of the PublishVersion operation returned by the service.
     * @throws ServiceException
     *         The AWS Lambda service encountered an internal error.
     * @throws ResourceNotFoundException
     *         The resource (for example, a Lambda function or access policy statement) specified in the request does
     *         not exist.
     * @throws InvalidParameterValueException
     *         One of the parameters in the request is invalid. For example, if you provided an IAM role for AWS Lambda
     *         to assume in the <code>CreateFunction</code> or the <code>UpdateFunctionConfiguration</code> API, that
     *         AWS Lambda is unable to assume you will get this exception.
     * @throws TooManyRequestsException
     *         Request throughput limit exceeded
     * @throws CodeStorageExceededException
     *         You have exceeded your maximum total code size per account. <a
     *         href="http://docs.aws.amazon.com/lambda/latest/dg/limits.html">Limits</a>
     * @throws PreconditionFailedException
     *         The RevisionId provided does not match the latest RevisionId for the Lambda function or alias. Call the
     *         <code>GetFunction</code> or the <code>GetAlias</code> API to retrieve the latest RevisionId for your
     *         resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LambdaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LambdaClient.PublishVersion
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/lambda-2015-03-31/PublishVersion" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PublishVersionResponse publishVersion(PublishVersionRequest publishVersionRequest) throws ServiceException,
            ResourceNotFoundException, InvalidParameterValueException, TooManyRequestsException, CodeStorageExceededException,
            PreconditionFailedException, AwsServiceException, SdkClientException, LambdaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PublishVersionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PublishVersionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<PublishVersionRequest, PublishVersionResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(publishVersionRequest).withMarshaller(new PublishVersionRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Sets a limit on the number of concurrent executions available to this function. It is a subset of your account's
     * total concurrent execution limit per region. Note that Lambda automatically reserves a buffer of 100 concurrent
     * executions for functions without any reserved concurrency limit. This means if your account limit is 1000, you
     * have a total of 900 available to allocate to individual functions. For more information, see <a
     * href="http://docs.aws.amazon.com/lambda/latest/dg/concurrent-executions.html">Managing Concurrency</a>.
     * </p>
     *
     * @param putFunctionConcurrencyRequest
     * @return Result of the PutFunctionConcurrency operation returned by the service.
     * @throws ServiceException
     *         The AWS Lambda service encountered an internal error.
     * @throws InvalidParameterValueException
     *         One of the parameters in the request is invalid. For example, if you provided an IAM role for AWS Lambda
     *         to assume in the <code>CreateFunction</code> or the <code>UpdateFunctionConfiguration</code> API, that
     *         AWS Lambda is unable to assume you will get this exception.
     * @throws ResourceNotFoundException
     *         The resource (for example, a Lambda function or access policy statement) specified in the request does
     *         not exist.
     * @throws TooManyRequestsException
     *         Request throughput limit exceeded
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LambdaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LambdaClient.PutFunctionConcurrency
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/lambda-2015-03-31/PutFunctionConcurrency" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public PutFunctionConcurrencyResponse putFunctionConcurrency(PutFunctionConcurrencyRequest putFunctionConcurrencyRequest)
            throws ServiceException, InvalidParameterValueException, ResourceNotFoundException, TooManyRequestsException,
            AwsServiceException, SdkClientException, LambdaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutFunctionConcurrencyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutFunctionConcurrencyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<PutFunctionConcurrencyRequest, PutFunctionConcurrencyResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(putFunctionConcurrencyRequest)
                .withMarshaller(new PutFunctionConcurrencyRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Removes permissions from a function. You can remove individual permissions from an resource policy associated
     * with a Lambda function by providing a statement ID that you provided when you added the permission. When you
     * remove permissions, disable the event source mapping or trigger configuration first to avoid errors.
     * </p>
     * <p>
     * Permissions apply to the Amazon Resource Name (ARN) used to invoke the function, which can be unqualified (the
     * unpublished version of the function), or include a version or alias. If a client uses a version or alias to
     * invoke a function, use the <code>Qualifier</code> parameter to apply permissions to that ARN. For more
     * information about versioning, see <a
     * href="http://docs.aws.amazon.com/lambda/latest/dg/versioning-aliases.html">AWS Lambda Function Versioning and
     * Aliases</a>.
     * </p>
     * <p>
     * You need permission for the <code>lambda:RemovePermission</code> action.
     * </p>
     *
     * @param removePermissionRequest
     * @return Result of the RemovePermission operation returned by the service.
     * @throws ServiceException
     *         The AWS Lambda service encountered an internal error.
     * @throws ResourceNotFoundException
     *         The resource (for example, a Lambda function or access policy statement) specified in the request does
     *         not exist.
     * @throws InvalidParameterValueException
     *         One of the parameters in the request is invalid. For example, if you provided an IAM role for AWS Lambda
     *         to assume in the <code>CreateFunction</code> or the <code>UpdateFunctionConfiguration</code> API, that
     *         AWS Lambda is unable to assume you will get this exception.
     * @throws TooManyRequestsException
     *         Request throughput limit exceeded
     * @throws PreconditionFailedException
     *         The RevisionId provided does not match the latest RevisionId for the Lambda function or alias. Call the
     *         <code>GetFunction</code> or the <code>GetAlias</code> API to retrieve the latest RevisionId for your
     *         resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LambdaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LambdaClient.RemovePermission
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/lambda-2015-03-31/RemovePermission" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public RemovePermissionResponse removePermission(RemovePermissionRequest removePermissionRequest) throws ServiceException,
            ResourceNotFoundException, InvalidParameterValueException, TooManyRequestsException, PreconditionFailedException,
            AwsServiceException, SdkClientException, LambdaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RemovePermissionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                RemovePermissionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<RemovePermissionRequest, RemovePermissionResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(removePermissionRequest).withMarshaller(new RemovePermissionRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Creates a list of tags (key-value pairs) on the Lambda function. Requires the Lambda function ARN (Amazon
     * Resource Name). If a key is specified without a value, Lambda creates a tag with the specified key and a value of
     * null. For more information, see <a href="http://docs.aws.amazon.com/lambda/latest/dg/tagging.html">Tagging Lambda
     * Functions</a> in the <b>AWS Lambda Developer Guide</b>.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ServiceException
     *         The AWS Lambda service encountered an internal error.
     * @throws ResourceNotFoundException
     *         The resource (for example, a Lambda function or access policy statement) specified in the request does
     *         not exist.
     * @throws InvalidParameterValueException
     *         One of the parameters in the request is invalid. For example, if you provided an IAM role for AWS Lambda
     *         to assume in the <code>CreateFunction</code> or the <code>UpdateFunctionConfiguration</code> API, that
     *         AWS Lambda is unable to assume you will get this exception.
     * @throws TooManyRequestsException
     *         Request throughput limit exceeded
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LambdaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LambdaClient.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/lambda-2015-03-31/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ServiceException,
            ResourceNotFoundException, InvalidParameterValueException, TooManyRequestsException, AwsServiceException,
            SdkClientException, LambdaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(tagResourceRequest).withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Removes tags from a Lambda function. Requires the function ARN (Amazon Resource Name). For more information, see
     * <a href="http://docs.aws.amazon.com/lambda/latest/dg/tagging.html">Tagging Lambda Functions</a> in the <b>AWS
     * Lambda Developer Guide</b>.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ServiceException
     *         The AWS Lambda service encountered an internal error.
     * @throws ResourceNotFoundException
     *         The resource (for example, a Lambda function or access policy statement) specified in the request does
     *         not exist.
     * @throws InvalidParameterValueException
     *         One of the parameters in the request is invalid. For example, if you provided an IAM role for AWS Lambda
     *         to assume in the <code>CreateFunction</code> or the <code>UpdateFunctionConfiguration</code> API, that
     *         AWS Lambda is unable to assume you will get this exception.
     * @throws TooManyRequestsException
     *         Request throughput limit exceeded
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LambdaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LambdaClient.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/lambda-2015-03-31/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ServiceException,
            ResourceNotFoundException, InvalidParameterValueException, TooManyRequestsException, AwsServiceException,
            SdkClientException, LambdaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(untagResourceRequest).withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Using this API you can update the function version to which the alias points and the alias description. For more
     * information, see <a href="http://docs.aws.amazon.com/lambda/latest/dg/aliases-intro.html">Introduction to AWS
     * Lambda Aliases</a>.
     * </p>
     * <p>
     * This requires permission for the lambda:UpdateAlias action.
     * </p>
     *
     * @param updateAliasRequest
     * @return Result of the UpdateAlias operation returned by the service.
     * @throws ServiceException
     *         The AWS Lambda service encountered an internal error.
     * @throws ResourceNotFoundException
     *         The resource (for example, a Lambda function or access policy statement) specified in the request does
     *         not exist.
     * @throws InvalidParameterValueException
     *         One of the parameters in the request is invalid. For example, if you provided an IAM role for AWS Lambda
     *         to assume in the <code>CreateFunction</code> or the <code>UpdateFunctionConfiguration</code> API, that
     *         AWS Lambda is unable to assume you will get this exception.
     * @throws TooManyRequestsException
     *         Request throughput limit exceeded
     * @throws PreconditionFailedException
     *         The RevisionId provided does not match the latest RevisionId for the Lambda function or alias. Call the
     *         <code>GetFunction</code> or the <code>GetAlias</code> API to retrieve the latest RevisionId for your
     *         resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LambdaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LambdaClient.UpdateAlias
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/lambda-2015-03-31/UpdateAlias" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateAliasResponse updateAlias(UpdateAliasRequest updateAliasRequest) throws ServiceException,
            ResourceNotFoundException, InvalidParameterValueException, TooManyRequestsException, PreconditionFailedException,
            AwsServiceException, SdkClientException, LambdaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateAliasResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateAliasResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UpdateAliasRequest, UpdateAliasResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(updateAliasRequest).withMarshaller(new UpdateAliasRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * You can update an event source mapping. This is useful if you want to change the parameters of the existing
     * mapping without losing your position in the stream. You can change which function will receive the stream
     * records, but to change the stream itself, you must create a new mapping.
     * </p>
     * <p>
     * If you disable the event source mapping, AWS Lambda stops polling. If you enable again, it will resume polling
     * from the time it had stopped polling, so you don't lose processing of any records. However, if you delete event
     * source mapping and create it again, it will reset.
     * </p>
     * <p>
     * This operation requires permission for the <code>lambda:UpdateEventSourceMapping</code> action.
     * </p>
     *
     * @param updateEventSourceMappingRequest
     * @return Result of the UpdateEventSourceMapping operation returned by the service.
     * @throws ServiceException
     *         The AWS Lambda service encountered an internal error.
     * @throws ResourceNotFoundException
     *         The resource (for example, a Lambda function or access policy statement) specified in the request does
     *         not exist.
     * @throws InvalidParameterValueException
     *         One of the parameters in the request is invalid. For example, if you provided an IAM role for AWS Lambda
     *         to assume in the <code>CreateFunction</code> or the <code>UpdateFunctionConfiguration</code> API, that
     *         AWS Lambda is unable to assume you will get this exception.
     * @throws TooManyRequestsException
     *         Request throughput limit exceeded
     * @throws ResourceConflictException
     *         The resource already exists.
     * @throws ResourceInUseException
     *         The operation conflicts with the resource's availability. For example, you attempted to update an
     *         EventSoure Mapping in CREATING, or tried to delete a EventSoure mapping currently in the UPDATING state.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LambdaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LambdaClient.UpdateEventSourceMapping
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/lambda-2015-03-31/UpdateEventSourceMapping"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateEventSourceMappingResponse updateEventSourceMapping(
            UpdateEventSourceMappingRequest updateEventSourceMappingRequest) throws ServiceException, ResourceNotFoundException,
            InvalidParameterValueException, TooManyRequestsException, ResourceConflictException, ResourceInUseException,
            AwsServiceException, SdkClientException, LambdaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateEventSourceMappingResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateEventSourceMappingResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<UpdateEventSourceMappingRequest, UpdateEventSourceMappingResponse>()
                        .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                        .withInput(updateEventSourceMappingRequest)
                        .withMarshaller(new UpdateEventSourceMappingRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Updates the code for the specified Lambda function. This operation must only be used on an existing Lambda
     * function and cannot be used to update the function configuration.
     * </p>
     * <p>
     * If you are using the versioning feature, note this API will always update the &#36LATEST version of your Lambda
     * function. For information about the versioning feature, see <a
     * href="http://docs.aws.amazon.com/lambda/latest/dg/versioning-aliases.html">AWS Lambda Function Versioning and
     * Aliases</a>.
     * </p>
     * <p>
     * This operation requires permission for the <code>lambda:UpdateFunctionCode</code> action.
     * </p>
     *
     * @param updateFunctionCodeRequest
     * @return Result of the UpdateFunctionCode operation returned by the service.
     * @throws ServiceException
     *         The AWS Lambda service encountered an internal error.
     * @throws ResourceNotFoundException
     *         The resource (for example, a Lambda function or access policy statement) specified in the request does
     *         not exist.
     * @throws InvalidParameterValueException
     *         One of the parameters in the request is invalid. For example, if you provided an IAM role for AWS Lambda
     *         to assume in the <code>CreateFunction</code> or the <code>UpdateFunctionConfiguration</code> API, that
     *         AWS Lambda is unable to assume you will get this exception.
     * @throws TooManyRequestsException
     *         Request throughput limit exceeded
     * @throws CodeStorageExceededException
     *         You have exceeded your maximum total code size per account. <a
     *         href="http://docs.aws.amazon.com/lambda/latest/dg/limits.html">Limits</a>
     * @throws PreconditionFailedException
     *         The RevisionId provided does not match the latest RevisionId for the Lambda function or alias. Call the
     *         <code>GetFunction</code> or the <code>GetAlias</code> API to retrieve the latest RevisionId for your
     *         resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LambdaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LambdaClient.UpdateFunctionCode
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/lambda-2015-03-31/UpdateFunctionCode" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateFunctionCodeResponse updateFunctionCode(UpdateFunctionCodeRequest updateFunctionCodeRequest)
            throws ServiceException, ResourceNotFoundException, InvalidParameterValueException, TooManyRequestsException,
            CodeStorageExceededException, PreconditionFailedException, AwsServiceException, SdkClientException, LambdaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateFunctionCodeResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateFunctionCodeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UpdateFunctionCodeRequest, UpdateFunctionCodeResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(updateFunctionCodeRequest).withMarshaller(new UpdateFunctionCodeRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Updates the configuration parameters for the specified Lambda function by using the values provided in the
     * request. You provide only the parameters you want to change. This operation must only be used on an existing
     * Lambda function and cannot be used to update the function's code.
     * </p>
     * <p>
     * If you are using the versioning feature, note this API will always update the &#36LATEST version of your Lambda
     * function. For information about the versioning feature, see <a
     * href="http://docs.aws.amazon.com/lambda/latest/dg/versioning-aliases.html">AWS Lambda Function Versioning and
     * Aliases</a>.
     * </p>
     * <p>
     * This operation requires permission for the <code>lambda:UpdateFunctionConfiguration</code> action.
     * </p>
     *
     * @param updateFunctionConfigurationRequest
     * @return Result of the UpdateFunctionConfiguration operation returned by the service.
     * @throws ServiceException
     *         The AWS Lambda service encountered an internal error.
     * @throws ResourceNotFoundException
     *         The resource (for example, a Lambda function or access policy statement) specified in the request does
     *         not exist.
     * @throws InvalidParameterValueException
     *         One of the parameters in the request is invalid. For example, if you provided an IAM role for AWS Lambda
     *         to assume in the <code>CreateFunction</code> or the <code>UpdateFunctionConfiguration</code> API, that
     *         AWS Lambda is unable to assume you will get this exception.
     * @throws TooManyRequestsException
     *         Request throughput limit exceeded
     * @throws ResourceConflictException
     *         The resource already exists.
     * @throws PreconditionFailedException
     *         The RevisionId provided does not match the latest RevisionId for the Lambda function or alias. Call the
     *         <code>GetFunction</code> or the <code>GetAlias</code> API to retrieve the latest RevisionId for your
     *         resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LambdaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LambdaClient.UpdateFunctionConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/lambda-2015-03-31/UpdateFunctionConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateFunctionConfigurationResponse updateFunctionConfiguration(
            UpdateFunctionConfigurationRequest updateFunctionConfigurationRequest) throws ServiceException,
            ResourceNotFoundException, InvalidParameterValueException, TooManyRequestsException, ResourceConflictException,
            PreconditionFailedException, AwsServiceException, SdkClientException, LambdaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateFunctionConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateFunctionConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<UpdateFunctionConfigurationRequest, UpdateFunctionConfigurationResponse>()
                        .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                        .withInput(updateFunctionConfigurationRequest)
                        .withMarshaller(new UpdateFunctionConfigurationRequestMarshaller(protocolFactory)));
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(LambdaException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterValueException")
                                .exceptionBuilderSupplier(InvalidParameterValueException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("EC2AccessDeniedException")
                                .exceptionBuilderSupplier(Ec2AccessDeniedException::builder).httpStatusCode(502).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceInUseException")
                                .exceptionBuilderSupplier(ResourceInUseException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("KMSInvalidStateException")
                                .exceptionBuilderSupplier(KmsInvalidStateException::builder).httpStatusCode(502).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("KMSNotFoundException")
                                .exceptionBuilderSupplier(KmsNotFoundException::builder).httpStatusCode(502).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PolicyLengthExceededException")
                                .exceptionBuilderSupplier(PolicyLengthExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceException")
                                .exceptionBuilderSupplier(ServiceException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("SubnetIPAddressLimitReachedException")
                                .exceptionBuilderSupplier(SubnetIpAddressLimitReachedException::builder).httpStatusCode(502)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidZipFileException")
                                .exceptionBuilderSupplier(InvalidZipFileException::builder).httpStatusCode(502).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestContentException")
                                .exceptionBuilderSupplier(InvalidRequestContentException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRuntimeException")
                                .exceptionBuilderSupplier(InvalidRuntimeException::builder).httpStatusCode(502).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ENILimitReachedException")
                                .exceptionBuilderSupplier(EniLimitReachedException::builder).httpStatusCode(502).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidSubnetIDException")
                                .exceptionBuilderSupplier(InvalidSubnetIdException::builder).httpStatusCode(502).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceConflictException")
                                .exceptionBuilderSupplier(ResourceConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CodeStorageExceededException")
                                .exceptionBuilderSupplier(CodeStorageExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidSecurityGroupIDException")
                                .exceptionBuilderSupplier(InvalidSecurityGroupIdException::builder).httpStatusCode(502).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("EC2ThrottledException")
                                .exceptionBuilderSupplier(Ec2ThrottledException::builder).httpStatusCode(502).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("KMSAccessDeniedException")
                                .exceptionBuilderSupplier(KmsAccessDeniedException::builder).httpStatusCode(502).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("KMSDisabledException")
                                .exceptionBuilderSupplier(KmsDisabledException::builder).httpStatusCode(502).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PreconditionFailedException")
                                .exceptionBuilderSupplier(PreconditionFailedException::builder).httpStatusCode(412).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("EC2UnexpectedException")
                                .exceptionBuilderSupplier(Ec2UnexpectedException::builder).httpStatusCode(502).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RequestTooLargeException")
                                .exceptionBuilderSupplier(RequestTooLargeException::builder).httpStatusCode(413).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequestsException")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnsupportedMediaTypeException")
                                .exceptionBuilderSupplier(UnsupportedMediaTypeException::builder).httpStatusCode(415).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends LambdaRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
