/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lambda.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An object that contains error information.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ErrorObject implements SdkPojo, Serializable, ToCopyableBuilder<ErrorObject.Builder, ErrorObject> {
    private static final SdkField<String> ERROR_MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ErrorMessage").getter(getter(ErrorObject::errorMessage)).setter(setter(Builder::errorMessage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ErrorMessage").build()).build();

    private static final SdkField<String> ERROR_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ErrorType").getter(getter(ErrorObject::errorType)).setter(setter(Builder::errorType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ErrorType").build()).build();

    private static final SdkField<String> ERROR_DATA_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ErrorData").getter(getter(ErrorObject::errorData)).setter(setter(Builder::errorData))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ErrorData").build()).build();

    private static final SdkField<List<String>> STACK_TRACE_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("StackTrace")
            .getter(getter(ErrorObject::stackTrace))
            .setter(setter(Builder::stackTrace))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StackTrace").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ERROR_MESSAGE_FIELD,
            ERROR_TYPE_FIELD, ERROR_DATA_FIELD, STACK_TRACE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String errorMessage;

    private final String errorType;

    private final String errorData;

    private final List<String> stackTrace;

    private ErrorObject(BuilderImpl builder) {
        this.errorMessage = builder.errorMessage;
        this.errorType = builder.errorType;
        this.errorData = builder.errorData;
        this.stackTrace = builder.stackTrace;
    }

    /**
     * <p>
     * A human-readable error message.
     * </p>
     * 
     * @return A human-readable error message.
     */
    public final String errorMessage() {
        return errorMessage;
    }

    /**
     * <p>
     * The error type.
     * </p>
     * 
     * @return The error type.
     */
    public final String errorType() {
        return errorType;
    }

    /**
     * <p>
     * Machine-readable error data.
     * </p>
     * 
     * @return Machine-readable error data.
     */
    public final String errorData() {
        return errorData;
    }

    /**
     * For responses, this returns true if the service returned a value for the StackTrace property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasStackTrace() {
        return stackTrace != null && !(stackTrace instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Stack trace information for the error.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasStackTrace} method.
     * </p>
     * 
     * @return Stack trace information for the error.
     */
    public final List<String> stackTrace() {
        return stackTrace;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(errorMessage());
        hashCode = 31 * hashCode + Objects.hashCode(errorType());
        hashCode = 31 * hashCode + Objects.hashCode(errorData());
        hashCode = 31 * hashCode + Objects.hashCode(hasStackTrace() ? stackTrace() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ErrorObject)) {
            return false;
        }
        ErrorObject other = (ErrorObject) obj;
        return Objects.equals(errorMessage(), other.errorMessage()) && Objects.equals(errorType(), other.errorType())
                && Objects.equals(errorData(), other.errorData()) && hasStackTrace() == other.hasStackTrace()
                && Objects.equals(stackTrace(), other.stackTrace());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ErrorObject")
                .add("ErrorMessage", errorMessage() == null ? null : "*** Sensitive Data Redacted ***")
                .add("ErrorType", errorType() == null ? null : "*** Sensitive Data Redacted ***")
                .add("ErrorData", errorData() == null ? null : "*** Sensitive Data Redacted ***")
                .add("StackTrace", stackTrace() == null ? null : "*** Sensitive Data Redacted ***").build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ErrorMessage":
            return Optional.ofNullable(clazz.cast(errorMessage()));
        case "ErrorType":
            return Optional.ofNullable(clazz.cast(errorType()));
        case "ErrorData":
            return Optional.ofNullable(clazz.cast(errorData()));
        case "StackTrace":
            return Optional.ofNullable(clazz.cast(stackTrace()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("ErrorMessage", ERROR_MESSAGE_FIELD);
        map.put("ErrorType", ERROR_TYPE_FIELD);
        map.put("ErrorData", ERROR_DATA_FIELD);
        map.put("StackTrace", STACK_TRACE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ErrorObject, T> g) {
        return obj -> g.apply((ErrorObject) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ErrorObject> {
        /**
         * <p>
         * A human-readable error message.
         * </p>
         * 
         * @param errorMessage
         *        A human-readable error message.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder errorMessage(String errorMessage);

        /**
         * <p>
         * The error type.
         * </p>
         * 
         * @param errorType
         *        The error type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder errorType(String errorType);

        /**
         * <p>
         * Machine-readable error data.
         * </p>
         * 
         * @param errorData
         *        Machine-readable error data.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder errorData(String errorData);

        /**
         * <p>
         * Stack trace information for the error.
         * </p>
         * 
         * @param stackTrace
         *        Stack trace information for the error.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stackTrace(Collection<String> stackTrace);

        /**
         * <p>
         * Stack trace information for the error.
         * </p>
         * 
         * @param stackTrace
         *        Stack trace information for the error.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stackTrace(String... stackTrace);
    }

    static final class BuilderImpl implements Builder {
        private String errorMessage;

        private String errorType;

        private String errorData;

        private List<String> stackTrace = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ErrorObject model) {
            errorMessage(model.errorMessage);
            errorType(model.errorType);
            errorData(model.errorData);
            stackTrace(model.stackTrace);
        }

        public final String getErrorMessage() {
            return errorMessage;
        }

        public final void setErrorMessage(String errorMessage) {
            this.errorMessage = errorMessage;
        }

        @Override
        public final Builder errorMessage(String errorMessage) {
            this.errorMessage = errorMessage;
            return this;
        }

        public final String getErrorType() {
            return errorType;
        }

        public final void setErrorType(String errorType) {
            this.errorType = errorType;
        }

        @Override
        public final Builder errorType(String errorType) {
            this.errorType = errorType;
            return this;
        }

        public final String getErrorData() {
            return errorData;
        }

        public final void setErrorData(String errorData) {
            this.errorData = errorData;
        }

        @Override
        public final Builder errorData(String errorData) {
            this.errorData = errorData;
            return this;
        }

        public final Collection<String> getStackTrace() {
            if (stackTrace instanceof SdkAutoConstructList) {
                return null;
            }
            return stackTrace;
        }

        public final void setStackTrace(Collection<String> stackTrace) {
            this.stackTrace = StackTraceEntriesCopier.copy(stackTrace);
        }

        @Override
        public final Builder stackTrace(Collection<String> stackTrace) {
            this.stackTrace = StackTraceEntriesCopier.copy(stackTrace);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder stackTrace(String... stackTrace) {
            stackTrace(Arrays.asList(stackTrace));
            return this;
        }

        @Override
        public ErrorObject build() {
            return new ErrorObject(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
