package uk.num.numlib.service;

import lombok.NonNull;
import uk.num.numlib.dns.DNSServices;
import uk.num.numlib.exc.*;
import uk.num.numlib.util.DomainLookupGenerator;
import uk.num.numlib.util.EmailLookupGenerator;
import uk.num.numlib.util.LookupGenerator;

public class HostedRecordLookupService {

    private final LookupServiceHelper helper;

    public HostedRecordLookupService(final DNSServices dnsServices) {
        helper = new LookupServiceHelper(dnsServices);
    }

    /**
     * Loads a record content from the hosted zone of the DNS. A null value is returned when the record is not
     * available.
     *
     * @param numIdName NUM id name for the record (for example domain name)
     * @param path      path of the record in the DNS
     * @param module    module for the record to retrieve
     * @param timeoutMs maximum time allowed to retrieve the record
     * @return a LoadRecordResponse or null
     * @throws NumInvalidParameterException if any of the supplied parameters are out of an acceptable range (catch NumException to handle all exceptions)
     * @throws NumInvalidDNSQueryException  if the parameters cannot be used to generated a valid DNS query (catch NumException to handle all exceptions)
     * @throws RrSetIncompleteException     if the NUM record is invalid (catch NumException to handle all exceptions)
     * @throws RrSetNoHeadersException      if the NUM record is invalid (catch NumException to handle all exceptions)
     * @throws RrSetHeaderFormatException   if the NUM record is invalid (catch NumException to handle all exceptions)
     */
    public LoadRecordResponse loadRecord(@NonNull final String numIdName, @NonNull final String path, final int module, final int timeoutMs) throws
                                                                                                                                             NumInvalidParameterException,
                                                                                                                                             NumInvalidDNSQueryException,
                                                                                                                                             RrSetIncompleteException,
                                                                                                                                             RrSetNoHeadersException,
                                                                                                                                             RrSetHeaderFormatException {
        LookupServiceHelper.validateParameters(numIdName, path, module, timeoutMs);

        final LookupGenerator generator = numIdName.contains("@") ? new EmailLookupGenerator(numIdName + path) : new DomainLookupGenerator(numIdName + path);

        return helper.loadRecordFromZone(timeoutMs, generator.getHostedLocation(module));
    }


}
