package uk.num.numlib.migrate;

import lombok.NonNull;

import java.util.Optional;
import java.util.function.Function;

/**
 * A Function to map old NUM records to the latest new format.
 */
public class NumRecordMigration implements Function<String, String> {

    public static final String OLD_NUM_RECORD_PREFIX = "_n=";

    public static final String NEW_NUM_RECORD_PREFIX = "@n=";

    /**
     * Make sure this is safe to run on records that have already been migrated.
     *
     * @param s a NUM record String
     * @return a possibly updated NUM record String
     */
    private String migrateUnderscoreN_to_atN(final String s) {
        if (s.startsWith(OLD_NUM_RECORD_PREFIX)) {
            return s.replaceFirst(OLD_NUM_RECORD_PREFIX, NEW_NUM_RECORD_PREFIX);
        }
        return s;
    }

    /**
     * Run a series of NUM record migration functions.
     * <p>
     * Using `map` might be overkill but its in case we need to support external clients with old data that has yet to
     * be migrated, since we can chain addition maps and support migration of records at any intermediate state to the
     * latest version.
     *
     * @param numRecord A non-null NUM record String
     * @return a possibly updated NUM record String - non-null
     */
    public String apply(@NonNull final String numRecord) {
        return Optional.of(numRecord)
                .map(this::migrateUnderscoreN_to_atN)
                .orElse(numRecord);
    }

}
