/*
 *    Copyright 2020 NUM Technology Ltd
 *
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

package uk.num.numlib.dns;

import lombok.NonNull;
import lombok.RequiredArgsConstructor;
import lombok.extern.log4j.Log4j2;
import org.xbill.DNS.DClass;
import org.xbill.DNS.Lookup;
import org.xbill.DNS.Message;
import org.xbill.DNS.Name;
import org.xbill.DNS.Rcode;
import org.xbill.DNS.Record;
import org.xbill.DNS.Resolver;
import org.xbill.DNS.Section;
import org.xbill.DNS.TXTRecord;
import org.xbill.DNS.Type;
import uk.num.numlib.exc.NumInvalidDNSQueryException;
import uk.num.numlib.exc.NumNoRecordAvailableException;
import uk.num.numlib.exc.RrSetHeaderFormatException;
import uk.num.numlib.exc.RrSetIncompleteException;

import java.io.IOException;
import java.time.Duration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Predicate;

/**
 * A default implementation of the DNSServices interface.
 *
 * @author tonywalmsley
 */
@RequiredArgsConstructor
@Log4j2
public class DNSServicesDefaultImpl implements DNSServices {

    public static final String MATCH_MULTIPART_RECORD_FRAGMENT = "(^\\d+\\|.*)|(\\d+\\/\\d+\\|@n=\\d+;.*)";

    /**
     * Is a record an SPF or CNAME record?
     */
    private static final Predicate<Record> isCNAMEOrSPFRecord = r -> r.rdataToString()
            .startsWith("v=spf")
            || r.rdataToString()
            .startsWith("\"v=spf") || r.getType() == Type.CNAME || r.getType() == Type.SPF;

    @NonNull
    private final Resolver resolver;

    public DNSServicesDefaultImpl() {
        resolver = Lookup.getDefaultResolver();
    }

    /**
     * Concatenate an array of TXT record values to a single String
     *
     * @param records The array of Records
     * @return The concatenated result.
     * @throws RrSetHeaderFormatException on error
     * @throws RrSetIncompleteException   on error
     */
    @Override
    public String rebuildTXTRecordContent(final List<Record> records)
            throws RrSetHeaderFormatException, RrSetIncompleteException {
        final StringBuilder buffer = new StringBuilder();

        final Map<Integer, String> ordered = new HashMap<>();

        int ignoredRecords = 0;
        int total = records.size();
        for (final Record r : records) {
            TXTRecord rec = (TXTRecord) r;
            List<String> dataParts = rec.getStrings();

            StringBuilder mergedDataParts = new StringBuilder();
            for (Object part : dataParts) {
                mergedDataParts.append(part.toString());
            }

            String data = mergedDataParts.toString();

            if (data.matches(MATCH_MULTIPART_RECORD_FRAGMENT)) {
                final int pipeIndex = data.indexOf("|");
                final String[] parts = new String[2];
                parts[0] = data.substring(0, pipeIndex);
                parts[1] = data.substring(pipeIndex + 1);

                final String substring = data.substring(parts[0].length() + 1);
                if (parts[0].contains("/")) {
                    ordered.put(0, substring);

                    String[] firstParts = parts[0].split("/");

                    if (firstParts.length == 2) {
                        try {
                            total = Integer.parseInt(firstParts[1]);
                        } catch (NumberFormatException ex) {
                            throw new RrSetHeaderFormatException("Could not parse total parts ${firstParts[1]}");
                        }

                    } else {
                        throw new RrSetHeaderFormatException(
                                "First part should only contain 1 \"/\", format is incorrect!");
                    }
                } else {
                    try {
                        int index = Integer.parseInt(parts[0]) - 1;

                        ordered.put(index, substring);

                    } catch (NumberFormatException ex) {
                        throw new RrSetHeaderFormatException("Could not parse index ${parts[0]}");
                    }

                }
            } else {
                if (records.size() == 1) {
                    ordered.put(0, data);
                } else {
                    // Ignore it according to the NUM Spec.
                    ignoredRecords++;
                }
            }
        }

        if (total != (records.size() - ignoredRecords)) {
            // incomplete set
            throw new RrSetIncompleteException("Parts and records length do not match, expected $total - $records");
        }

        for (int i = 0; i < ordered.size(); i++) {
            buffer.append(ordered.get(i));
        }

        final String result = buffer.toString();
        log.debug("Rebuilt DNS records: {}", result);
        return result;
    }

    /**
     * Get a NUM record from DNS without caching.
     *
     * @param query         The NUM formatted DNS query.
     * @param timeoutMillis The number of milliseconds to wait for a response.
     * @return An array of Records
     * @throws NumInvalidDNSQueryException   on error
     * @throws NumNoRecordAvailableException if a CNAME or SPF record is received
     *                                       instead of a TXT record
     */
    @Override
    public GetRecordResponse getRecordFromDnsNoCache(final String query, final int timeoutMillis)
            throws NumInvalidDNSQueryException, NumNoRecordAvailableException {

        // Return the cached value if we have one.
        List<Record> records;

        // No cached value so look it up in DNS
        resolver.setTimeout(Duration.ofMillis(timeoutMillis));
        resolver.setIgnoreTruncation(false);

        boolean signed;
        try {
            final Record queryTxtRecord = Record.newRecord(new Name(query), Type.TXT, DClass.IN);
            final Message queryMessage = Message.newQuery(queryTxtRecord);
            log.debug("Sending DNS Query: {}", queryMessage);

            final Message response = resolver.send(queryMessage);
            log.debug("Received DNS Response: {}", response);

            signed = response.isSigned();

            if (response.getRcode() == Rcode.NOERROR) {
                records = response.getSection(Section.ANSWER);
                if (records.stream()
                        .anyMatch(isCNAMEOrSPFRecord)) {
                    throw new NumNoRecordAvailableException("Received CNAME or SPF record instead of TXT record.");
                }
            } else {
                log.info("Error querying for NUM record: query: {}, rcode: {}", query,
                        Rcode.string(response.getRcode()));
                return null;
            }
        } catch (IOException e) {
            log.error("Error querying for NUM record.", e);
            throw new NumInvalidDNSQueryException("Invalid DNS query: " + query);
        }

        return new GetRecordResponse(signed, records);
    }

}
