/*
 * Copyright (c) 2019. NUM Technology Ltd
 */

package uk.num.numlib.api;

import org.apache.commons.lang3.StringUtils;
import org.xbill.DNS.Lookup;
import org.xbill.DNS.Record;
import org.xbill.DNS.SimpleResolver;
import uk.num.numlib.exc.*;
import uk.num.numlib.internal.ctx.NumAPIContextBase;
import uk.num.numlib.internal.dns.DNSServices;
import uk.num.numlib.internal.dns.DNSServicesDefaultImpl;
import uk.num.numlib.internal.dns.PossibleMultiPartRecordException;
import uk.num.numlib.internal.modl.ModlServices;
import uk.num.numlib.internal.modl.NumLookupRedirect;
import uk.num.numlib.internal.modl.NumQueryRedirect;
import uk.num.numlib.internal.modl.PopulatorResponse;
import uk.num.numlib.internal.module.ModuleConfig;
import uk.num.numlib.internal.module.ModuleDNSQueries;
import uk.num.numlib.internal.module.ModuleFactory;
import uk.num.numlib.internal.util.PopulatorRetryConfig;
import uk.num.numlib.internal.util.StringConstants;

import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;

import static uk.num.numlib.api.NumAPICallbacks.Location.*;

/**
 * This is the main class for using the num-client-library.
 * Use the default constructor to use the DNS servers configured on your local machine,
 * or override these by supplying a specific DNS host domain name using the alternative constructor.
 *
 * @author tonywalmsley
 */
public class NumAPI {

    public static final int MAX_NUM_REDIRECTS = 3;
    private static final int MAX_NUMBER_OF_MULTI_PARTS = 30;
    /**
     * Services for accessing DNS and processing the resulting records.
     */
    private DNSServices dnsServices;
    /**
     * Services for running the MODL Interpreter
     */
    private ModlServices modlServices;

    /**
     * Supports running DNS queries asynchronously.
     */
    private ExecutorService executor = Executors.newSingleThreadExecutor();

    /**
     * Default constructor to initialise the default DNS services and MODL services.
     */
    public NumAPI() {
        dnsServices = new DNSServicesDefaultImpl();
        modlServices = new ModlServices();
    }

    /**
     * Constructor allowing the core services to be overridden, mainly for testing purposes, but could also be useful for other purposes.
     *
     * @param dnsServices  Replacement DNSServices implementation.
     * @param modlServices Replacement ModlServices implementation.
     */
    public NumAPI(final DNSServices dnsServices, final ModlServices modlServices) {
        this.dnsServices = dnsServices;
        this.modlServices = modlServices;
    }

    /**
     * Alternative constructor used to override the default DNS hosts. Unit tests rely on this constructor.
     *
     * @param dnsHost The DNS host to override the defaults configured for the local machine.
     * @throws NumInvalidDNSHostException on error
     */
    public NumAPI(final String dnsHost) throws NumInvalidDNSHostException {
        this();
        try {
            if (!StringUtils.isEmpty(dnsHost)) {
                final SimpleResolver resolver = new SimpleResolver(dnsHost);
                Lookup.setDefaultResolver(resolver);
            }
        } catch (UnknownHostException e) {
            throw new NumInvalidDNSHostException("Invalid DNS host.", e);
        }
    }

    /**
     * Alternative constructor used to override the default DNS host and port.
     *
     * @param dnsHost The DNS host to override the defaults configured for the local machine.
     * @param port    The port to use on the DNS host.
     * @throws NumInvalidDNSHostException on error
     */
    public NumAPI(final String dnsHost, final int port) throws NumInvalidDNSHostException {
        this();
        try {
            if (!StringUtils.isEmpty(dnsHost)) {
                final SimpleResolver resolver = new SimpleResolver(dnsHost);
                resolver.setPort(port);
                Lookup.setDefaultResolver(resolver);
                resolver.setTCP(true);
            }
        } catch (UnknownHostException e) {
            throw new NumInvalidDNSHostException("Invalid DNS host.", e);
        }
    }

    /**
     * Tell dnsjava to use TCP and not UDP.
     *
     * @param flag true to use TCP only.
     */
    public void setTCPOnly(final boolean flag) {
        Lookup.getDefaultResolver().setTCP(flag);
    }

    /**
     * Override the top-level zone from 'num.uk' to 'myzone.com' for example.
     *
     * @param zone The top level zone to use for DNS lookups. Replaces the default of 'num.uk'
     */
    public void setTopLevelZone(final String zone) {
        StringConstants.instance().setTopLevelZone(zone);
    }

    /**
     * Initialise a new NumAPIContextBase object for a specific module/domain combination.
     * The returned context object can be used to obtain the list of required user variables that must be set
     * before moving on to retrieveNumRecord().
     *
     * @param moduleId      E.g. "1" for the Contacts module.
     * @param netString     a domain name, URL, or email address that identifies the location in DNS of a NUM record.
     * @param timeoutMillis the timeout in milliseconds to wait for responses from DNS.
     * @return a new NumAPIContextBase object.
     * @throws NumBadModuleIdException on error
     * @throws NumBadModuleConfigDataException on error
     * @throws NumBadURLException on error
     * @throws NumInvalidParameterException on error
     * @throws NumBadRecordException on error
     * @throws NumDNSQueryException on error
     * @throws NumInvalidDNSQueryException on error
     */
    public NumAPIContext begin(final String moduleId, final String netString, final int timeoutMillis) throws
                                                                                                       NumBadModuleIdException,
                                                                                                       NumBadModuleConfigDataException,
                                                                                                       NumBadURLException,
                                                                                                       NumInvalidParameterException,
                                                                                                       NumBadRecordException,
                                                                                                       NumDNSQueryException,
                                                                                                       NumInvalidDNSQueryException {
        assert moduleId != null && moduleId.trim().length() > 0;
        assert netString != null && netString.trim().length() > 0;
        assert timeoutMillis > 0;

        // Create the context object and the validated ModuleDNSQueries object.
        final NumAPIContextBase ctx = new NumAPIContextBase();

        final ModuleDNSQueries moduleDNSQueries = ModuleFactory.getInstance(moduleId, netString);
        ctx.setModuleDNSQueries(moduleDNSQueries);

        // Get the moduleDNSQueries config data
        final Record[] records = dnsServices.getConfigFileTXTRecords(moduleId, timeoutMillis);

        if (records == null || records.length == 0) {
            throw new NumBadModuleIdException("No configuration moduleDNSQueries file available. Check that the moduleDNSQueries ID is correct.: " + moduleId);
        }

        final String configTxt = dnsServices.rebuildTXTRecordContent(records).replaceAll("\\\\", "");

        final ModuleConfig moduleConfig = modlServices.interpretModuleConfig(configTxt);
        if (!moduleConfig.isValid()) {
            throw new NumBadModuleConfigDataException("Invalid module config data: " + configTxt);
        }
        ctx.setModuleConfig(moduleConfig);

        return ctx;
    }

    /**
     * This method uses the module context and the supplied Required User Variable values to obtain a fully expanded
     * JSON object from DNS. The supplied handler will be notified when the results are available or an error occurs.
     *
     * @param ctx           The context object returned by the begin() method.
     * @param handler       a handler object to receive the JSON results or processing errors.
     * @param timeoutMillis the maximum duration of each DNS request, the total wait time could be up to 4 times this value.
     * @return A Future object
     */
    public Future<String> retrieveNumRecord(final NumAPIContext ctx, final NumAPICallbacks handler, final int timeoutMillis) {
        assert ctx != null;
        assert handler != null;

        // 1. Re-interpret the module config now that we have the user variables.
        // 2. Get the NUM record from DNS (either independent, managed, prepopulated, or populator)
        // 2. Prepend the RCF to the NUM record from DNS as a *LOAD entry (i.e. the module URL)
        // 3. Run the resulting record through the Java MODL Interpreter and make the results available to the client via the handler.

        // Do the rest of the operation asynchronously.
        // This submits a Callable object, so exceptions should be reported to the user when they call the get() method on the Future object.
        return executor.submit(() -> {
            do {
                final NumAPIContextBase context = (NumAPIContextBase) ctx;
                try {

                    final ModuleConfig moduleConfig = context.getModuleConfig();

                    // Attempt to get the independent record from DNS
                    String jsonResult = getNumRecordFromIndependentLocation(timeoutMillis, context);

                    // If that failed then try the managed record.
                    if (jsonResult == null) {
                        jsonResult = getNumRecordFromManagedLocation(timeoutMillis, context);
                        if (jsonResult != null) {
                            handler.setLocation(MANAGED);
                        }
                        // If we still don't have a result then try the prepopulated and populator queries if the module config flags allow it.
                        // There are different flags for root queries and branch queries.
                        if (context.getModuleDNSQueries().isRootQuery()) {
                            if (jsonResult == null && moduleConfig.getModule().isRprq()) {
                                jsonResult = getNumRecordFromPrepopulatedLocation(timeoutMillis, context);
                                if (jsonResult != null) {
                                    handler.setLocation(POPULATED);
                                }
                            }
                            if (jsonResult == null && moduleConfig.getModule().isRpsq()) {
                                jsonResult = getNumRecordFromPopulator(timeoutMillis, context);
                            }
                        } else {
                            // Assume its a branch query
                            if (jsonResult == null && moduleConfig.getModule().isBprq()) {
                                jsonResult = getNumRecordFromPrepopulatedLocation(timeoutMillis, context);
                                if (jsonResult != null) {
                                    handler.setLocation(POPULATED);
                                }
                            }
                            if (jsonResult == null && moduleConfig.getModule().isBpsq()) {
                                jsonResult = getNumRecordFromPopulator(timeoutMillis, context);
                            }
                        }
                    } else {
                        handler.setLocation(INDEPENDENT);
                    }

                    // Return the jsonResult if we found something.
                    if (jsonResult != null && jsonResult.trim().length() > 0) {
                        handler.setResult(jsonResult);
                        return jsonResult;
                    } else {
                        handler.setErrorResult("Cannot retrieve the NUM record for module: " + moduleConfig.getModule().getName());
                        break;
                    }
                } catch (final NumException e) {
                    handler.setErrorResult("Cannot retrieve the NUM record for module: " + e.getLocalizedMessage());
                    break;
                } catch (final NumLookupRedirect numLookupRedirect) {
                    //
                    // For l_ redirects we need to retry the current Location with the updated query.
                    //
                    int redirectCount = context.incrementRedirectCount();
                    if (redirectCount >= MAX_NUM_REDIRECTS) {
                        throw new NumMaximumRedirectsExceededException();
                    }
                    context.handleLookupRedirect(numLookupRedirect.getRedirect());
                }
            } while (true);
            return null;
        });
    }

    /**
     * Try retrieving a pre-populated record from DNS
     *
     * @param timeoutMillis The timeout
     * @param context       The context obtained from the NumAPI.begin() method
     * @return The String result or null
     * @throws NumBadMultipartRecordException on error
     * @throws NumNotImplementedException on error
     * @throws NumInvalidDNSQueryException on error
     * @throws NumLookupRedirect on error
     * @throws NumBadRecordException on error
     * @throws NumMaximumRedirectsExceededException on error
     * @throws NumBadURLException on error
     * @throws NumInvalidRedirectException on error
     * @throws NumInvalidParameterException on error
     */
    private String getNumRecordFromPrepopulatedLocation(int timeoutMillis, final NumAPIContextBase context) throws
                                                                                                            NumBadMultipartRecordException,
                                                                                                            NumNotImplementedException,
                                                                                                            NumInvalidDNSQueryException,
                                                                                                            NumLookupRedirect,
                                                                                                            NumBadRecordException,
                                                                                                            NumMaximumRedirectsExceededException,
                                                                                                            NumBadURLException,
                                                                                                            NumInvalidRedirectException,
                                                                                                            NumInvalidParameterException {
        String json = null;
        do {
            final String recordLocation = context.getModuleDNSQueries().getPrepopulatedRecordLocation();
            final String numRecord = getNumRecord(timeoutMillis, context, recordLocation);
            // If we found a NUM record, combine it with the RCF string from the module config file and
            // pass it through the MODL interpreter to obtain the full expanded NUM record as a JSON String.
            if (numRecord != null && numRecord.trim().length() > 0) {
                // Build a MODL object using the required user variables, the RCF, and the NUM record from DNS.
                try {
                    json = interpretNumRecord(numRecord, context);
                    break;
                } catch (final NumQueryRedirect numQueryRedirect) {
                    //
                    // For q_ redirects we need to retry the current Location with the updated query.
                    //
                    int redirectCount = context.incrementRedirectCount();
                    if (redirectCount >= MAX_NUM_REDIRECTS) {
                        throw new NumMaximumRedirectsExceededException();
                    }
                    context.handlePrepopulatedQueryRedirect(numQueryRedirect.getRedirect());
                }
            } else {
                break;
            }
        } while (true);

        return json;
    }

    /**
     * We had a response with the Truncated Flag set.
     *
     * @param timeoutMillis  The number of milliseconds we're prepared to wait per DNS request.
     * @param context        The NumAPIContextBase
     * @param recordLocation The DNS query String.
     * @return An array of Record objects
     * @throws NumBadMultipartRecordException on error
     * @throws NumInvalidDNSQueryException on error
     * @throws NumNotImplementedException on error
     */
    private Record[] getMultiPartRecords(final int timeoutMillis, final NumAPIContextBase context, final String recordLocation) throws
                                                                                                                                NumBadMultipartRecordException,
                                                                                                                                NumInvalidDNSQueryException,
                                                                                                                                NumNotImplementedException {
        Record[] recordFromDns;// First get the number of parts and check it is valid.
        final Record[] numberOfPartsRecord = dnsServices.getRecordFromDns("0." + recordLocation, timeoutMillis, context.getModuleConfig().getModule().isDsr());
        if (numberOfPartsRecord == null || numberOfPartsRecord.length == 0) {
            return null;
        }
        final String numberOfPartsStr = dnsServices.rebuildTXTRecordContent(numberOfPartsRecord);
        if (!numberOfPartsStr.startsWith("parts=")) {
            throw new NumBadMultipartRecordException("Invalid record 0 for multi-part record: " + numberOfPartsStr);
        }
        final int numberOfParts = Integer.parseInt(numberOfPartsStr.substring(6));
        if (numberOfParts > MAX_NUMBER_OF_MULTI_PARTS) {
            throw new NumBadMultipartRecordException("Too many parts for a multi-part record: " + numberOfPartsStr);
        }

        // Now get each part and add them all to a list.
        final List<Record> parts = new ArrayList<>();
        for (int i = 1; i <= numberOfParts; i++) {
            final Record[] partNRecords = dnsServices.getRecordFromDns("" + i + "." + recordLocation, timeoutMillis, context.getModuleConfig().getModule().isDsr());
            if (partNRecords != null && partNRecords.length > 0) {
                parts.addAll(Arrays.asList(partNRecords));
            }
        }
        recordFromDns = parts.toArray(new Record[]{});
        return recordFromDns;
    }

    /**
     * Try retrieving a record from the populator
     *
     * @param timeoutMillis The timeout
     * @param context       The context obtained from the NumAPI.begin() method
     * @return The String result or null
     * @throws NumPopoulatorErrorException on error
     * @throws NumNoRecordAvailableException on error
     * @throws NumInvalidPopulatorResponseCodeException on error
     * @throws NumBadMultipartRecordException on error
     * @throws NumBadRecordException on error
     * @throws NumNotImplementedException on error
     * @throws NumInvalidDNSQueryException on error
     * @throws NumLookupRedirect on error
     * @throws NumMaximumRedirectsExceededException on error
     * @throws NumInvalidParameterException on error
     * @throws NumBadURLException on error
     * @throws NumInvalidRedirectException on error
     */
    private String getNumRecordFromPopulator(int timeoutMillis, final NumAPIContextBase context) throws
                                                                                                 NumPopoulatorErrorException,
                                                                                                 NumNoRecordAvailableException,
                                                                                                 NumInvalidPopulatorResponseCodeException,
                                                                                                 NumBadMultipartRecordException,
                                                                                                 NumBadRecordException,
                                                                                                 NumNotImplementedException,
                                                                                                 NumInvalidDNSQueryException,
                                                                                                 NumLookupRedirect,
                                                                                                 NumMaximumRedirectsExceededException,
                                                                                                 NumInvalidParameterException,
                                                                                                 NumBadURLException,
                                                                                                 NumInvalidRedirectException {
        final String recordLocation = context.getModuleDNSQueries().getPopulatorLocation();
        String numRecord = null;
        while (numRecord == null) {
            numRecord = getNumRecord(timeoutMillis, context, recordLocation);
            if (numRecord == null) {
                // This is unrecoverable, we should get status_ or error_ object.
                break;
            }

            // Parse the MODL response
            final PopulatorResponse response = modlServices.interpretPopulatorResponse(numRecord);
            if (!response.isValid()) {
                throw new NumInvalidPopulatorResponseCodeException("Bad response received from the populator service.");
            }
            // Handle the status_ response codes
            if (response.getStatus_() != null) {
                numRecord = handlePopulatorStatusCodes(timeoutMillis, context, response);
            }
            // Handle the error_ response codes
            if (response.getError_() != null) {
                if (response.getError_().getCode() == 100) {// Enter the populated zone retry loop
                    try {
                        int i = 0;
                        while (i < PopulatorRetryConfig.ERROR_RETRIES) {
                            TimeUnit.SECONDS.sleep(PopulatorRetryConfig.ERROR_RETRY_DELAYS[i]);
                            numRecord = getNumRecord(timeoutMillis, context, recordLocation);

                            final PopulatorResponse retryResponse = modlServices.interpretPopulatorResponse(numRecord);
                            if (retryResponse.getStatus_() != null) {
                                return handlePopulatorStatusCodes(timeoutMillis, context, retryResponse);
                            }
                            i++;
                        }
                    } catch (InterruptedException e) {
                        // Ignore, just throw the error below
                    }
                    throw new NumNoRecordAvailableException("Cannot retrieve NUM record from any location.");
                } else {
                    throw new NumPopoulatorErrorException(response.getError_().getDescription());
                }
            }
        }
        return numRecord;
    }

    /**
     * Populator status codes tell us how to retry the queries while the populator works in the background to get the necessary data
     *
     * @param timeoutMillis the timeout
     * @param context       the NumAPIContextBase object.
     * @param response      the response from the populator
     * @return null or a valid NUM record
     * @throws NumNoRecordAvailableException on error
     * @throws NumInvalidPopulatorResponseCodeException on error
     * @throws NumBadMultipartRecordException on error
     * @throws NumInvalidDNSQueryException on error
     * @throws NumNotImplementedException on error
     * @throws NumLookupRedirect on error
     * @throws NumBadRecordException on error
     * @throws NumMaximumRedirectsExceededException on error
     * @throws NumBadURLException on error
     * @throws NumInvalidParameterException on error
     * @throws NumInvalidRedirectException on error
     */
    private String handlePopulatorStatusCodes(int timeoutMillis, NumAPIContextBase context, PopulatorResponse response) throws
                                                                                                                        NumNoRecordAvailableException,
                                                                                                                        NumInvalidPopulatorResponseCodeException,
                                                                                                                        NumBadMultipartRecordException,
                                                                                                                        NumInvalidDNSQueryException,
                                                                                                                        NumNotImplementedException,
                                                                                                                        NumLookupRedirect,
                                                                                                                        NumBadRecordException,
                                                                                                                        NumMaximumRedirectsExceededException,
                                                                                                                        NumBadURLException,
                                                                                                                        NumInvalidParameterException,
                                                                                                                        NumInvalidRedirectException {
        String numRecord = null;
        switch (response.getStatus_().getCode()) {
            case 1:
                // Enter the populated zone retry loop
                try {
                    int i = 0;
                    while (i < PopulatorRetryConfig.RETRIES) {
                        TimeUnit.SECONDS.sleep(PopulatorRetryConfig.RETRY_DELAYS[i]);
                        numRecord = getNumRecordFromPrepopulatedLocation(timeoutMillis, context);
                        if (numRecord != null) {
                            break;
                        }
                        i++;
                    }
                    throw new NumNoRecordAvailableException("Cannot retrieve NUM record from any location.");
                } catch (InterruptedException e) {
                    break;
                }
            case 2:
                // The record is available at the authoritative server
                numRecord = getNumRecordFromIndependentLocation(timeoutMillis, context);
                if (numRecord == null) {
                    throw new NumNoRecordAvailableException("Cannot retrieve NUM record from any location.");
                }
                break;
            case 3:
                // The record exists in the managed zone.
                numRecord = getNumRecordFromManagedLocation(timeoutMillis, context);
                if (numRecord == null) {
                    throw new NumNoRecordAvailableException("Cannot retrieve NUM record from any location.");
                }
                break;
            default:
                throw new NumInvalidPopulatorResponseCodeException("Invalid response code from DNS populator service.");
        }
        return numRecord;
    }

    /**
     * Try retrieving a managed record from DNS
     *
     * @param timeoutMillis The timeout
     * @param context       The context obtained from the NumAPI.begin() method
     * @return The String result or null
     * @throws NumBadMultipartRecordException on error
     * @throws NumNotImplementedException on error
     * @throws NumInvalidDNSQueryException on error
     * @throws NumLookupRedirect on error
     * @throws NumBadRecordException on error
     * @throws NumMaximumRedirectsExceededException on error
     * @throws NumBadURLException on error
     * @throws NumInvalidRedirectException on error
     * @throws NumInvalidParameterException on error
     */
    private String getNumRecordFromManagedLocation(final int timeoutMillis, final NumAPIContextBase context) throws
                                                                                                             NumBadMultipartRecordException,
                                                                                                             NumNotImplementedException,
                                                                                                             NumInvalidDNSQueryException,
                                                                                                             NumLookupRedirect,
                                                                                                             NumBadRecordException,
                                                                                                             NumMaximumRedirectsExceededException,
                                                                                                             NumBadURLException,
                                                                                                             NumInvalidRedirectException,
                                                                                                             NumInvalidParameterException {
        String json = null;
        do {
            final String recordLocation = context.getModuleDNSQueries().getManagedRecordLocation();
            final String numRecord = getNumRecord(timeoutMillis, context, recordLocation);
            // If we found a NUM record, combine it with the RCF string from the module config file and
            // pass it through the MODL interpreter to obtain the full expanded NUM record as a JSON String.
            if (numRecord != null && numRecord.trim().length() > 0) {
                // Build a MODL object using the required user variables, the RCF, and the NUM record from DNS.
                try {
                    json = interpretNumRecord(numRecord, context);
                    break;
                } catch (final NumQueryRedirect numQueryRedirect) {
                    //
                    // For q_ redirects we need to retry the current Location with the updated query.
                    //
                    int redirectCount = context.incrementRedirectCount();
                    if (redirectCount >= MAX_NUM_REDIRECTS) {
                        throw new NumMaximumRedirectsExceededException();
                    }
                    context.handleManagedQueryRedirect(numQueryRedirect.getRedirect());
                }
            } else {
                break;
            }
        } while (true);

        return json;
    }

    /**
     * Interpret the supplied NUM record from DNS using the RCF value from the ModuleConfig object.
     *
     * @param moduleNumber The module number
     * @param moduleConfig The ModuleConfig
     * @param numRecord    The NUM record from DNS
     * @return The JSON String result of the fully expanded NUM record.
     * @throws NumBadRecordException on error
     * @throws NumQueryRedirect on error
     * @throws NumLookupRedirect on error
     */
    private String getInterpretedNumRecordAsJson(final String moduleNumber, final ModuleConfig moduleConfig, final String numRecord) throws
                                                                                                                                     NumBadRecordException,
                                                                                                                                     NumQueryRedirect,
                                                                                                                                     NumLookupRedirect {
        final StringBuilder numRecordBuffer = new StringBuilder();

        final RequiredUserVariable[] ruv = moduleConfig.getModule().getRuv();
        if (ruv != null) {
            for (RequiredUserVariable v : ruv) {
                numRecordBuffer.append(v.getKey());
                numRecordBuffer.append("=");
                numRecordBuffer.append(v.getValue());
                numRecordBuffer.append(";");
            }
        }
        numRecordBuffer.append("*load=\"http://modules.num.uk/");
        numRecordBuffer.append(moduleNumber);
        numRecordBuffer.append("/rcf.txt!\";");
        numRecordBuffer.append(numRecord);

        return modlServices.interpretNumRecord(numRecordBuffer.toString());
    }

    /**
     * Try retrieving an independent record from DNS
     *
     * @param timeoutMillis The timeout
     * @param context       The context obtained from the NumAPI.begin() method
     * @return The String result or null
     * @throws NumBadMultipartRecordException on error
     * @throws NumNotImplementedException on error
     * @throws NumInvalidDNSQueryException on error
     * @throws NumLookupRedirect on error
     * @throws NumBadRecordException on error
     * @throws NumMaximumRedirectsExceededException on error
     * @throws NumInvalidParameterException on error
     * @throws NumBadURLException on error
     * @throws NumInvalidRedirectException on error
     */
    private String getNumRecordFromIndependentLocation(final int timeoutMillis, final NumAPIContextBase context) throws
                                                                                                                 NumBadMultipartRecordException,
                                                                                                                 NumNotImplementedException,
                                                                                                                 NumInvalidDNSQueryException,
                                                                                                                 NumLookupRedirect,
                                                                                                                 NumBadRecordException,
                                                                                                                 NumMaximumRedirectsExceededException,
                                                                                                                 NumInvalidParameterException,
                                                                                                                 NumBadURLException,
                                                                                                                 NumInvalidRedirectException {
        String json = null;
        do {
            final String recordLocation = context.getModuleDNSQueries().getIndependentRecordLocation();
            final String numRecord = getNumRecord(timeoutMillis, context, recordLocation);
            // If we found a NUM record, combine it with the RCF string from the module config file and
            // pass it through the MODL interpreter to obtain the full expanded NUM record as a JSON String.
            if (numRecord != null && numRecord.trim().length() > 0) {
                // Build a MODL object using the required user variables, the RCF, and the NUM record from DNS.
                try {
                    json = interpretNumRecord(numRecord, context);
                    break;
                } catch (final NumQueryRedirect numQueryRedirect) {
                    //
                    // For q_ redirects we need to retry the current Location with the updated query.
                    //
                    int redirectCount = context.incrementRedirectCount();
                    if (redirectCount >= MAX_NUM_REDIRECTS) {
                        throw new NumMaximumRedirectsExceededException();
                    }
                    context.handleIndependentQueryRedirect(numQueryRedirect.getRedirect());
                }
            } else {
                break;
            }
        } while (true);

        return json;
    }

    /**
     * Convert a NUM record String to an interpreted JSON String. Handle any redirect instructions in the interpreted MODL record
     *
     * @param numRecord the uninterpreted NUM record.
     * @param context   the NumAPIContext
     * @return the interpreted NUM record as a JSON string.
     * @throws NumLookupRedirect on error
     * @throws NumBadRecordException on error
     * @throws NumQueryRedirect on error
     */
    private String interpretNumRecord(final String numRecord, final NumAPIContextBase context) throws NumLookupRedirect,
                                                                                                      NumBadRecordException,
                                                                                                      NumQueryRedirect {
        String json = null;
        if (numRecord != null && numRecord.trim().length() > 0) {
            // Build a MODL object using the required user variables, the RCF, and the NUM record from DNS.
            json = getInterpretedNumRecordAsJson(context.getModuleDNSQueries().getModuleId(), context.getModuleConfig(), numRecord);
        }
        return json;
    }

    /**
     * Get a NUM record for the given query string. Try multi-part queries if necessary.
     *
     * @param timeoutMillis  The timeout
     * @param context        The context obtained from the NumAPI.begin() method
     * @param recordLocation The DNS query String.
     * @return The raw NUM record from DNS.
     * @throws NumBadMultipartRecordException on error
     * @throws NumInvalidDNSQueryException on error
     * @throws NumNotImplementedException on error
     */
    private String getNumRecord(int timeoutMillis, NumAPIContextBase context, String recordLocation) throws
                                                                                                     NumBadMultipartRecordException,
                                                                                                     NumInvalidDNSQueryException,
                                                                                                     NumNotImplementedException {
        Record[] recordFromDns;
        try {
            recordFromDns = dnsServices.getRecordFromDns(recordLocation, timeoutMillis, context.getModuleConfig().getModule().isDsr());
        } catch (final PossibleMultiPartRecordException e) {
            recordFromDns = getMultiPartRecords(timeoutMillis, context, recordLocation);
        }
        if (recordFromDns == null || recordFromDns.length == 0) {
            return null;
        }
        return dnsServices.rebuildTXTRecordContent(recordFromDns);
    }

    /**
     * Stop any outstanding DNS queries still in the Executor.
     */
    public void shutdown() {
        try {
            executor.shutdown();
            executor.awaitTermination(1, TimeUnit.SECONDS);
        } catch (InterruptedException e) {
            // Ignore
        } finally {
            if (!executor.isTerminated()) {
                executor.shutdownNow();
            }
        }
    }
}

