package uk.num.numlib.internal.dns;

import org.xbill.DNS.*;
import uk.num.numlib.exc.NumDNSQueryException;
import uk.num.numlib.exc.NumInvalidDNSQueryException;
import uk.num.numlib.exc.NumNotImplementedException;
import uk.num.numlib.internal.util.SimpleCache;
import uk.num.numlib.internal.util.StringConstants;

import java.io.IOException;
import java.util.List;

/**
 * A default implementation of the DNSServices interface.
 *
 * @author tonywalmsley
 */
public class DNSServicesDefaultImpl implements DNSServices {

    /**
     * A cache for DNS query results.
     */
    private final SimpleCache<String, Record[]> cache = new SimpleCache<>();

    /**
     * Get the Module Configuration from DNS as an array of Records
     *
     * @param moduleId      The module Id String
     * @param timeoutMillis The number of milliseconds to wait for a response.
     * @return An array of Records
     * @throws NumInvalidDNSQueryException on error
     * @throws NumDNSQueryException on error
     */
    public Record[] getConfigFileTXTRecords(final String moduleId, final int timeoutMillis) throws
                                                                                            NumInvalidDNSQueryException,
                                                                                            NumDNSQueryException {
        assert moduleId != null && moduleId.trim().length() > 0;
        assert timeoutMillis > 0;

        // Return the cached value if we have one.
        final String key = moduleId + "_config";
        Record[] records;
        records = cache.get(key);
        if (records != null) {
            return records;
        }

        // No cached value so look it up in DNS
        final Resolver resolver = Lookup.getDefaultResolver();
        resolver.setTimeout(timeoutMillis / 1000, timeoutMillis % 1000);
        resolver.setIgnoreTruncation(false);

        final String query = moduleId + StringConstants.instance().CONFIG_FILE_SUFFIX();
        try {
            final Lookup lookup = new Lookup(query, Type.TXT);
            records = lookup.run();

            if (lookup.getResult() != Lookup.SUCCESSFUL) {
                throw new NumDNSQueryException("Error retrieving configuration file from DNS: " + lookup.getErrorString() + " for query: " + query);
            }
        } catch (TextParseException e) {
            throw new NumInvalidDNSQueryException(query);
        }

        // Cache it for next time.
        cache.put(key, records);
        return records;
    }

    /**
     * Concatenate an array of TXT record values to a single String
     *
     * @param records The array of Records
     * @return The concatenated result.
     */
    public String rebuildTXTRecordContent(final Record[] records) {
        assert records != null && records.length > 0;

        final StringBuilder buffer = new StringBuilder();

        for (Record record : records) {
            final TXTRecord txt = (TXTRecord) record;
            final List strings = txt.getStrings();

            // Append all of the String parts of the TXT record to a single String via the StringBuffer
            for (Object string : strings) {
                buffer.append(string);
            }
        }

        return buffer.toString();
    }

    /**
     * Get a NUM record from DNS.
     *
     * @param query               The NUM formatted DNS query.
     * @param timeoutMillis       The number of milliseconds to wait for a response.
     * @param checkDnsSecValidity true if the result should be checked for DNSSEC validity.
     * @return An array of Records
     * @throws NumNotImplementedException on error
     * @throws NumInvalidDNSQueryException on error
     */
    @Override
    public Record[] getRecordFromDns(final String query, final int timeoutMillis, final boolean checkDnsSecValidity) throws
                                                                                                                     NumNotImplementedException,
                                                                                                                     NumInvalidDNSQueryException {
        assert query != null && query.trim().length() > 0;
        assert timeoutMillis > 0;

        // Return the cached value if we have one.
        Record[] records;
        records = cache.get(query);
        if (records != null) {
            return records;
        }

        // No cached value so look it up in DNS
        final Resolver resolver = Lookup.getDefaultResolver();
        resolver.setTimeout(timeoutMillis / 1000, timeoutMillis % 1000);
        resolver.setIgnoreTruncation(false);

        try {
            final Record queryTxtRecord = Record.newRecord(new Name(query), Type.TXT, DClass.IN);
            final Message queryMessage = Message.newQuery(queryTxtRecord);
            final Message response = resolver.send(queryMessage);
            if (response.getRcode() == Rcode.NOERROR) {
                if (checkDnsSecValidity) {
                    throw new NumNotImplementedException("DNSSEC checks not implemented.");
                }
                records = response.getSectionArray(Section.ANSWER);
            } else {
                if (response.getHeader().getFlag(Flags.TC)) {
                    throw new PossibleMultiPartRecordException();
                }
                return null;
            }
        } catch (IOException e) {
            throw new NumInvalidDNSQueryException("Invalid DNS query: " + query);
        }

        // Cache it for next time.
        cache.put(query, records);
        return records;
    }
}
