/*
 * Copyright (c) 2019. NUM Technology Ltd
 */

package uk.num.numlib.internal.util;

import org.apache.commons.lang3.StringUtils;
import uk.num.numlib.exc.NumBadURLException;
import uk.num.numlib.exc.NumInvalidParameterException;
import uk.num.numlib.internal.ctx.AppContext;

import java.net.IDN;
import java.net.MalformedURLException;
import java.net.URL;

/**
 * Convert domain name Strings to normalised domain names as defined in the NUM Protocol Specification.
 *
 * @author tonywalmsley
 */
public class DomainNameUtils {

    /**
     * Constructor.
     */
    public DomainNameUtils() {
    }

    /**
     * Convert a domain name to a query format suitable for use in an independent query.
     *
     * @param appContext the AppContext
     * @param domainName java.lang.String The domain name to be converted.
     * @param moduleId   java.lang.String the module number/id as a String
     * @return java.lang.String The converted domain name result.
     * @throws NumBadURLException           on error
     * @throws NumInvalidParameterException on error
     */
    public String toIndependentRecordQuery(final AppContext appContext, final String domainName, final String moduleId) throws
                                                                                                                        NumBadURLException,
                                                                                                                        NumInvalidParameterException {
        validateParams(domainName, moduleId);

        final String normalisedDomain = normaliseDomainName(domainName);

        return String.format("%s%s%s.", moduleId, appContext.stringConstants
                .UTILITY_MODULE_PREFIX(), normalisedDomain);
    }

    /**
     * Check that the parameters are valid.
     *
     * @param domainName java.lang.String The domain name to be converted.
     * @param moduleId   java.lang.String the module number/id as a String
     * @throws NumInvalidParameterException on error
     */
    private void validateParams(final String domainName, final String moduleId) throws NumInvalidParameterException {
        if (domainName == null) {
            throw new NumInvalidParameterException("Cannot normalise a null domainName");
        }
        if (domainName.trim()
                .isEmpty()) {
            throw new NumInvalidParameterException("The domainName cannot be empty");
        }
        if (moduleId == null) {
            throw new NumInvalidParameterException("The moduleId cannot be null");
        }
        if (moduleId.trim()
                .isEmpty()) {
            throw new NumInvalidParameterException("The moduleId cannot be empty");
        }
    }

    /**
     * Convert a domain name to a query format suitable for use in a managed record query.
     *
     * @param appContext the AppContext
     * @param domainName java.lang.String The domain name to be converted.
     * @param moduleId   java.lang.String the module number/id as a String
     * @return java.lang.String The converted URL result.
     * @throws NumInvalidParameterException on error
     * @throws NumBadURLException           on error
     */
    public String toManagedRecordQuery(final AppContext appContext, final String domainName, final String moduleId) throws
                                                                                                                    NumInvalidParameterException,
                                                                                                                    NumBadURLException {
        validateParams(domainName, moduleId);

        final String normalisedDomain = normaliseDomainName(domainName);

        final String hashValue = HashUtils
                .hash(normalisedDomain);
        return String.format("%s.%s%s%s%s.", moduleId, appContext.stringConstants
                .DOMAIN_NAME_PREFIX(), normalisedDomain, hashValue, appContext.stringConstants
                .MANAGED_RECORD_SUFFIX());
    }

    /**
     * Convert a domain name to a query format suitable for use in a pre-populated record query.
     *
     * @param appContext the AppContext
     * @param domainName java.lang.String The domain name to be converted.
     * @param moduleId   java.lang.String the module number/id as a String
     * @return java.lang.String The converted URL result.
     * @throws NumBadURLException           on error
     * @throws NumInvalidParameterException on error
     */
    public String toPrePopulatedRecordQuery(final AppContext appContext, final String domainName, final String moduleId) throws
                                                                                                                         NumBadURLException,
                                                                                                                         NumInvalidParameterException {
        validateParams(domainName, moduleId);

        final String normalisedDomain = normaliseDomainName(domainName);

        final String hashValue = HashUtils
                .hash(normalisedDomain);
        return String.format("%s.%s%s%s%s.", moduleId, appContext.stringConstants
                .DOMAIN_NAME_PREFIX(), normalisedDomain, hashValue, appContext.stringConstants
                .PREPOPULATED_RECORD_SUFFIX());
    }

    /**
     * Convert a domain name to a query format suitable for use in a populator query.
     *
     * @param appContext the AppContext
     * @param domainName java.lang.String The domain name to be converted.
     * @param moduleId   java.lang.String the module number/id as a String
     * @return java.lang.String The converted URL result.
     * @throws NumBadURLException           on error
     * @throws NumInvalidParameterException on error
     */
    public String toPopulatorQuery(final AppContext appContext, final String domainName, final String moduleId) throws
                                                                                                                NumBadURLException,
                                                                                                                NumInvalidParameterException {
        validateParams(domainName, moduleId);

        final String normalisedDomain = normaliseDomainName(domainName);

        final String hashValue = HashUtils
                .hash(normalisedDomain);
        return String.format("%s.%s%s%s%s.", moduleId, appContext.stringConstants
                .DOMAIN_NAME_PREFIX(), normalisedDomain, hashValue, appContext.stringConstants
                .POPULATOR_SERVICE_SUFFIX());
    }

    /**
     * Convert a domain name or URL into a normalised name by removing 'www.' and any trailing '.'.
     *
     * @param domainName java.lang.String The Not Null domain name string or URL.
     * @return A normalised java.lang.String domain name.
     * @throws NumInvalidParameterException on error
     * @throws NumBadURLException           on error
     */
    public String normaliseDomainName(final String domainName) throws NumInvalidParameterException, NumBadURLException {
        if (domainName == null) {
            throw new NumInvalidParameterException("Null domain name cannot be normalised");
        }
        if (domainName.trim()
                .isEmpty()) {
            throw new NumInvalidParameterException("Empty domain name cannot be normalised");
        }
        if (domainName.startsWith("http")) {
            try {
                final URL url = new URL(domainName);
                final String host = url.getHost();
                return normaliseDomainName(host);
            } catch (MalformedURLException e) {
                throw new NumBadURLException("Invalid URL: " + domainName);
            }
        }

        String result = domainName;
        if (result.startsWith("www.")) {
            result = result.substring(4);
        }

        if (result.startsWith(".")) {
            result = result.substring(1);
        }

        if (result.endsWith(".")) {
            result = result.substring(0, result.length() - 1);
        }

        if (!StringUtils.isAsciiPrintable(result)) {
            final String[] parts = result.split("\\.");
            for (int i = 0; i < parts.length; i++) {
                if (!StringUtils.isAsciiPrintable(parts[i])) {
                    parts[i] = IDN.toASCII(parts[i]);
                }
            }
            result = StringUtils.join(parts, ".");
        }
        return result;
    }
}
