/*
 * Copyright (c) 2019. NUM Technology Ltd
 */

package uk.num.numlib.internal.util;

import uk.num.numlib.exc.NumBadURLException;
import uk.num.numlib.exc.NumInvalidParameterException;
import uk.num.numlib.internal.ctx.AppContext;

/**
 * Methods to convert an email address to a NUM Protocol query.
 *
 * @author tonywalmsley
 */
public class EmailAddressUtils {

    /**
     * Constructor
     */
    public EmailAddressUtils() {
    }

    /**
     * Convert an email address to an independent record query.
     *
     * @param appContext   the AppContext
     * @param emailAddress java.lang.String the email address to be converted.
     * @param moduleId     java.lang.String the num module number/id.
     * @return java.lang.String the transformed email address
     * @throws NumBadURLException           on error
     * @throws NumInvalidParameterException on error
     */
    public String toIndependentRecordQuery(final AppContext appContext, final String emailAddress, final String moduleId) throws
                                                                                                                          NumBadURLException,
                                                                                                                          NumInvalidParameterException {
        validateParams(emailAddress, moduleId);
        final String[] parts = emailAddress.split("@");
        parts[1] = appContext.domainNameUtils
                .normaliseDomainName(parts[1]);
        parts[0] = parts[0].replaceAll(" ", "_");
        return String.format("%s.%s%s%s.", parts[0], moduleId, appContext.stringConstants
                .UTILITY_MODULE_PREFIX(), parts[1]);
    }

    /**
     * Check the method parameters are valid.
     *
     * @param emailAddress java.lang.String the email address to be converted.
     * @param moduleId     java.lang.String the num module number/id.
     * @throws NumInvalidParameterException on error
     */
    private void validateParams(final String emailAddress, final String moduleId) throws NumInvalidParameterException {
        if (emailAddress == null) {
            throw new NumInvalidParameterException("The email address cannot be null");
        }
        if (emailAddress.trim()
                .isEmpty()) {
            throw new NumInvalidParameterException("The email address cannot be empty");
        }
        if (moduleId == null) {
            throw new NumInvalidParameterException("The moduleId cannot be null");
        }
        if (moduleId.trim()
                .isEmpty()) {
            throw new NumInvalidParameterException("The moduleId cannot be empty");
        }
        if (!emailAddress.contains("@")) {
            throw new NumInvalidParameterException("The email address is invalid - missing '@'");
        }
        final String[] emailAddressParts = emailAddress.split("@");
        if (emailAddressParts.length != 2 || emailAddressParts[0].isEmpty()) {
            throw new NumInvalidParameterException("The email address is invalid - the local part or the domain name is empty");
        }
    }

    /**
     * Convert an email address to a managed record query.
     *
     * @param appContext   the AppContext
     * @param emailAddress java.lang.String the email address to be converted.
     * @param moduleId     java.lang.String the num module number/id.
     * @return java.lang.String the transformed email address
     * @throws NumBadURLException           on error
     * @throws NumInvalidParameterException on error
     */
    public String toManagedRecordQuery(final AppContext appContext, final String emailAddress, final String moduleId) throws
                                                                                                                      NumBadURLException,
                                                                                                                      NumInvalidParameterException {
        validateParams(emailAddress, moduleId);
        final String[] parts = emailAddress.split("@");
        parts[1] = appContext.domainNameUtils
                .normaliseDomainName(parts[1]);
        parts[0] = parts[0].replaceAll(" ", "_");
        final String hashValue = HashUtils
                .hash(parts[1]);
        return String.format("%s.%s.%s%s%s%s.", parts[0], moduleId, appContext.stringConstants
                .DOMAIN_NAME_PREFIX(), parts[1], hashValue, appContext.stringConstants
                .MANAGED_RECORD_SUFFIX());
    }

    /**
     * Convert an email address to a pre-populated record query.
     *
     * @param appContext   the AppContext
     * @param emailAddress java.lang.String the email address to be converted.
     * @param moduleId     java.lang.String the num module number/id.
     * @return java.lang.String the transformed email address
     * @throws NumBadURLException           on error
     * @throws NumInvalidParameterException on error
     */
    public String toPrePopulatedRecordQuery(final AppContext appContext, final String emailAddress, final String moduleId) throws
                                                                                                                           NumBadURLException,
                                                                                                                           NumInvalidParameterException {
        validateParams(emailAddress, moduleId);
        final String[] parts = emailAddress.split("@");
        parts[1] = appContext.domainNameUtils
                .normaliseDomainName(parts[1]);
        parts[0] = parts[0].replaceAll(" ", "_");
        final String hashValue = HashUtils
                .hash(parts[1]);
        return String.format("%s.%s.%s%s%s%s.", parts[0], moduleId, appContext.stringConstants
                .DOMAIN_NAME_PREFIX(), parts[1], hashValue, appContext.stringConstants
                .PREPOPULATED_RECORD_SUFFIX());
    }

    /**
     * Convert an email address to a populator query.
     *
     * @param appContext   the AppContext
     * @param emailAddress java.lang.String the email address to be converted.
     * @param moduleId     java.lang.String the num module number/id.
     * @return java.lang.String the transformed email address
     * @throws NumBadURLException           on error
     * @throws NumInvalidParameterException on error
     */
    public String toPopulatorQuery(final AppContext appContext, final String emailAddress, final String moduleId) throws
                                                                                                                  NumBadURLException,
                                                                                                                  NumInvalidParameterException {
        validateParams(emailAddress, moduleId);
        final String[] parts = emailAddress.split("@");
        parts[1] = appContext.domainNameUtils
                .normaliseDomainName(parts[1]);
        parts[0] = parts[0].replaceAll(" ", "_");
        final String hashValue = HashUtils
                .hash(parts[1]);
        return String.format("%s.%s.%s%s%s%s.", parts[0], moduleId, appContext.stringConstants
                .DOMAIN_NAME_PREFIX(), parts[1], hashValue, appContext.stringConstants
                .POPULATOR_SERVICE_SUFFIX());
    }
}
