package uk.num.numlib.internal.ctx;

import lombok.Getter;
import lombok.Setter;
import lombok.extern.log4j.Log4j2;
import uk.num.numlib.api.NumAPICallbacks;
import uk.num.numlib.api.NumAPIContext;
import uk.num.numlib.api.UserVariable;
import uk.num.numlib.exc.*;
import uk.num.numlib.internal.module.ModuleConfig;
import uk.num.numlib.internal.module.ModuleDNSQueries;
import uk.num.numlib.internal.util.NonBlankString;
import uk.num.numlib.internal.util.UrlRelativePathResolver;

/**
 * A base class implementation of NumAPIContext.
 *
 * @author tonywalmsley
 */
@Log4j2
public class NumAPIContextBase implements NumAPIContext {
    private static final int MAX_NUM_REDIRECTS = 3;
    /**
     * The DNS query Strings for the current module and NUM ID.
     */
    @Getter
    @Setter
    private ModuleDNSQueries moduleDNSQueries;
    /**
     * The module configuration data.
     */
    @Setter
    @Getter
    private ModuleConfig moduleConfig;

    /**
     * Count redirects so we don't redirect forever.
     */
    private int redirectCount = 0;
    /**
     * The location currently being checked for a NUM record.
     */
    @Getter
    @Setter
    private NumAPICallbacks.Location location;

    /**
     * Count redirects and return the current number of redirects.
     *
     * @return the current number of redirects
     */
    private int incrementRedirectCount() {
        this.redirectCount++;
        return this.redirectCount;
    }

    /**
     * Modules can have required user variables that must be supplied before the NUM record csn be retrieved.
     *
     * @return An array of objects.
     */
    @Override
    public UserVariable[] getRequiredUserVariables() {
        return moduleConfig.getModule()
                .getUv();
    }

    /**
     * Update the required user variables with values obtained from the client.
     *
     * @param userVariables The RequiredUserVariable array with the value fields populated.
     * @throws NumUserVariablesException on error
     */
    @Override
    public void setRequiredUserVariables(final UserVariable[] userVariables) throws
                                                                             NumUserVariablesException {

        // Validate that the user variables have been entered.
        final StringBuilder errors = new StringBuilder();
        boolean errorsFound = false;

        for (final UserVariable v : userVariables) {
            if (v.isReq() && (v.getValue() == null || v.getValue()
                    .trim()
                    .length() == 0)) {
                errorsFound = true;
                errors.append("Variable '");
                errors.append(v.getKey());
                errors.append("' should not be null or empty.\n");
                log.trace("Required User Variable: key={}, prompt={}, value={}", v.getKey(), v.getName(), v.getValue());
            }
        }

        if (errorsFound) {
            log.error("Error in Required User Variables: {}", errors.toString());
            throw new NumUserVariablesException("Error in Required User Variables: " + errors.toString());
        }

        // All looks fine so store the variables.
        moduleConfig.getModule()
                .setUv(userVariables);
    }

    /**
     * Get the query location based on the current location that is being tried.
     *
     * @return a DNS query string for the current location.
     */
    public String getRecordLocation() {
        switch (location) {
            case INDEPENDENT:
                return moduleDNSQueries.getIndependentRecordLocation();
            case HOSTED:
                return moduleDNSQueries.getHostedRecordLocation();
            case POPULATOR:
                return moduleDNSQueries.getPopulatorLocation();
            default:
                return "STOP";
        }
    }

    /**
     * Update the relevant query for the supplied redirect
     *
     * @param redirect the supplied redirect
     * @param context  the NumAPIContextBase
     * @throws NumMaximumRedirectsExceededException on Error
     */
    public void handleQueryRedirect(final NonBlankString redirect, final NumAPIContextBase context) throws
                                                                                                    NumMaximumRedirectsExceededException,
                                                                                                    NumInvalidDNSQueryException,
                                                                                                    NumInvalidRedirectException {
        log.info("Query Redirected to: {}", redirect);
        int redirectCount = context.incrementRedirectCount();
        if (redirectCount >= MAX_NUM_REDIRECTS) {
            log.error("Maximum Redirects Exceeded. (max={})", MAX_NUM_REDIRECTS);
            throw new NumMaximumRedirectsExceededException();
        }

        switch (location) {
            case INDEPENDENT:
                handleIndependentQueryRedirect(redirect);
            case HOSTED:
                handleHostedQueryRedirect(redirect);
            default:
                break;
        }
    }

    /**
     * Update the hosted query for the supplied redirect
     *
     * @param redirectTo the supplied redirect
     * @throws NumInvalidDNSQueryException on error
     * @throws NumInvalidRedirectException on error
     */
    private void handleHostedQueryRedirect(final NonBlankString redirectTo) throws
                                                                            NumInvalidDNSQueryException,
                                                                            NumInvalidRedirectException {
        final String hostedRecordPath = moduleDNSQueries.getHostedRecordPath();
        try {
            moduleDNSQueries.redirectHostedPath(UrlRelativePathResolver.resolve(hostedRecordPath, redirectTo.value));
        } catch (final RelativePathException e) {
            throw new NumInvalidRedirectException(e);
        }
    }

    /**
     * Update the independent query for the supplied redirect
     *
     * @param redirectTo the supplied redirect
     * @throws NumInvalidDNSQueryException on error
     * @throws NumInvalidRedirectException on error
     */
    private void handleIndependentQueryRedirect(final NonBlankString redirectTo) throws
                                                                                 NumInvalidRedirectException,
                                                                                 NumInvalidDNSQueryException {
        final String independentRecordPath = moduleDNSQueries.getIndependentRecordPath();
        try {
            moduleDNSQueries.redirectIndependentPath(UrlRelativePathResolver.resolve(independentRecordPath, redirectTo.value));
        } catch (final RelativePathException e) {
            throw new NumInvalidRedirectException(e);
        }
    }
}
