/*
 * Copyright (c) 2019. NUM Technology Ltd
 */

package uk.num.numlib.internal.util;

import uk.num.numlib.exc.NumBadURLException;
import uk.num.numlib.exc.NumInvalidParameterException;

/**
 * Methods to convert an email address to a NUM Protocol query.
 *
 * @author tonywalmsley
 */
public class EmailAddressUtils {
    /**
     * Static instance variable for the Singleton object.
     */
    private static EmailAddressUtils _instance;

    /**
     * Private constructor to enforce the Singleton pattern.
     */
    private EmailAddressUtils() {
    }

    /**
     * Static accessor for the Singleton instance - lazy init.
     *
     * @return EmailAddressUtils
     */
    public static EmailAddressUtils instance() {
        if (_instance == null) {
            _instance = new EmailAddressUtils();
        }
        return _instance;
    }

    /**
     * Convert an email address to an independent record query.
     *
     * @param emailAddress java.lang.String the email address to be converted.
     * @param moduleId     java.lang.String the num module number/id.
     * @return java.lang.String the transformed email address
     * @throws NumBadURLException on error
     * @throws NumInvalidParameterException on error
     */
    public String toIndependentRecordQuery(final String emailAddress, final String moduleId) throws NumBadURLException,
                                                                                                    NumInvalidParameterException {
        validateParams(emailAddress, moduleId);
        final String[] parts = emailAddress.split("@");
        parts[1] = DomainNameUtils.instance().normaliseDomainName(parts[1]);
        parts[0] = parts[0].replaceAll(" ", "_");
        return String.format("%s.%s%s%s.", parts[0], moduleId, StringConstants.instance().UTILITY_MODULE_PREFIX(), parts[1]);
    }

    /**
     * Check the method parameters are valid.
     *
     * @param emailAddress java.lang.String the email address to be converted.
     * @param moduleId     java.lang.String the num module number/id.
     * @throws NumInvalidParameterException on error
     */
    private void validateParams(final String emailAddress, final String moduleId) throws NumInvalidParameterException {
        if (emailAddress == null) {
            throw new NumInvalidParameterException("The email address cannot be null");
        }
        if (emailAddress.trim().isEmpty()) {
            throw new NumInvalidParameterException("The email address cannot be empty");
        }
        if (moduleId == null) {
            throw new NumInvalidParameterException("The moduleId cannot be null");
        }
        if (moduleId.trim().isEmpty()) {
            throw new NumInvalidParameterException("The moduleId cannot be empty");
        }
        if (!emailAddress.contains("@")) {
            throw new NumInvalidParameterException("The email address is invalid - missing '@'");
        }
        final String[] emailAddressParts = emailAddress.split("@");
        if (emailAddressParts.length != 2 || emailAddressParts[0].isEmpty()) {
            throw new NumInvalidParameterException("The email address is invalid - the local part or the domain name is empty");
        }
    }

    /**
     * Convert an email address to a managed record query.
     *
     * @param emailAddress java.lang.String the email address to be converted.
     * @param moduleId     java.lang.String the num module number/id.
     * @return java.lang.String the transformed email address
     * @throws NumBadURLException on error
     * @throws NumInvalidParameterException on error
     */
    public String toManagedRecordQuery(final String emailAddress, final String moduleId) throws NumBadURLException,
                                                                                                NumInvalidParameterException {
        validateParams(emailAddress, moduleId);
        final String[] parts = emailAddress.split("@");
        parts[1] = DomainNameUtils.instance().normaliseDomainName(parts[1]);
        parts[0] = parts[0].replaceAll(" ", "_");
        final String hashValue = HashUtils.instance().hash(parts[1]);
        return String.format("%s.%s.%s%s%s%s.", parts[0], moduleId, StringConstants.instance().DOMAIN_NAME_PREFIX(), parts[1], hashValue, StringConstants.instance().MANAGED_RECORD_SUFFIX());
    }

    /**
     * Convert an email address to a pre-populated record query.
     *
     * @param emailAddress java.lang.String the email address to be converted.
     * @param moduleId     java.lang.String the num module number/id.
     * @return java.lang.String the transformed email address
     * @throws NumBadURLException on error
     * @throws NumInvalidParameterException on error
     */
    public String toPrePopulatedRecordQuery(final String emailAddress, final String moduleId) throws NumBadURLException,
                                                                                                     NumInvalidParameterException {
        validateParams(emailAddress, moduleId);
        final String[] parts = emailAddress.split("@");
        parts[1] = DomainNameUtils.instance().normaliseDomainName(parts[1]);
        parts[0] = parts[0].replaceAll(" ", "_");
        final String hashValue = HashUtils.instance().hash(parts[1]);
        return String.format("%s.%s.%s%s%s%s.", parts[0], moduleId, StringConstants.instance().DOMAIN_NAME_PREFIX(), parts[1], hashValue, StringConstants.instance().PREPOPULATED_RECORD_SUFFIX());
    }

    /**
     * Convert an email address to a populator query.
     *
     * @param emailAddress java.lang.String the email address to be converted.
     * @param moduleId     java.lang.String the num module number/id.
     * @return java.lang.String the transformed email address
     * @throws NumBadURLException on error
     * @throws NumInvalidParameterException on error
     */
    public String toPopulatorQuery(final String emailAddress, final String moduleId) throws NumBadURLException,
                                                                                            NumInvalidParameterException {
        validateParams(emailAddress, moduleId);
        final String[] parts = emailAddress.split("@");
        parts[1] = DomainNameUtils.instance().normaliseDomainName(parts[1]);
        parts[0] = parts[0].replaceAll(" ", "_");
        final String hashValue = HashUtils.instance().hash(parts[1]);
        return String.format("%s.%s.%s%s%s%s.", parts[0], moduleId, StringConstants.instance().DOMAIN_NAME_PREFIX(), parts[1], hashValue, StringConstants.instance().POPULATOR_SERVICE_SUFFIX());
    }
}
