/*
 * Copyright (c) 2019. NUM Technology Ltd
 */

package uk.num.numlib.internal.util;

import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import uk.num.numlib.exc.NumBadURLException;
import uk.num.numlib.exc.NumInvalidParameterException;
import uk.num.numlib.internal.ctx.AppContext;

import java.net.URL;

/**
 * Transform a URL to a format suitable for various NUM Protocol queries.
 *
 * @author tonywalmsley
 */
public class URLUtils {

    /**
     * Constructor
     */
    public URLUtils() {
    }

    /**
     * Convert a URL to a query format suitable for use in an independent query.
     *
     * @param url      java.net.URL The URL to be converted.
     * @param moduleId java.lang.String the module number/id as a String
     * @return java.lang.String The converted URL result.
     * @throws NumInvalidParameterException on error
     * @throws NumBadURLException           on error
     */
    public String toIndependentRecordQuery(final URL url, final String moduleId) throws NumInvalidParameterException,
                                                                                        NumBadURLException {
        validateParams(url, moduleId);

        final String normalisedDomain = AppContext.domainNameUtils
                .normaliseDomainName(url.getHost());
        final String normalisedPath = normalisePath(url.getPath());

        return String.format("%s%s%s%s.", normalisedPath, moduleId, AppContext.stringConstants.UTILITY_MODULE_PREFIX(), normalisedDomain);
    }

    /**
     * Check that the parameters are valid.
     *
     * @param url      java.net.URL The URL to be converted.
     * @param moduleId java.lang.String the module number/id as a String
     * @throws NumInvalidParameterException on error
     */
    private void validateParams(final URL url, final String moduleId) throws NumInvalidParameterException {
        if (url == null) {
            throw new NumInvalidParameterException("Cannot normalise a null URL");
        }
        if (moduleId == null) {
            throw new NumInvalidParameterException("The moduleId cannot be null");
        }
        if (moduleId.trim()
                .isEmpty()) {
            throw new NumInvalidParameterException("The moduleId cannot be empty");
        }
    }

    /**
     * Convert a URL to a query format suitable for use in a managed record query.
     *
     * @param url      java.net.URL The URL to be converted.
     * @param moduleId java.lang.String the module number/id as a String
     * @return java.lang.String The converted URL result.
     * @throws NumInvalidParameterException on error
     * @throws NumBadURLException           on error
     */
    public String toManagedRecordQuery(final URL url, final String moduleId) throws NumInvalidParameterException,
                                                                                    NumBadURLException {
        validateParams(url, moduleId);

        final String normalisedDomain = AppContext.domainNameUtils
                .normaliseDomainName(url.getHost());
        final String normalisedPath = normalisePath(url.getPath());

        final String hashValue = HashUtils
                .hash(normalisedDomain);
        return String.format("%s%s.%s%s%s%s.", normalisedPath, moduleId, AppContext.stringConstants.DOMAIN_NAME_PREFIX(), normalisedDomain, hashValue, AppContext.stringConstants.MANAGED_RECORD_SUFFIX());
    }

    /**
     * Convert a URL to a query format suitable for use in a pre-populated record query.
     *
     * @param url      java.net.URL The URL to be converted.
     * @param moduleId java.lang.String the module number/id as a String
     * @return java.lang.String The converted URL result.
     * @throws NumInvalidParameterException on error
     * @throws NumBadURLException           on error
     */
    public String toPrePopulatedRecordQuery(final URL url, final String moduleId) throws NumInvalidParameterException,
                                                                                         NumBadURLException {
        validateParams(url, moduleId);

        final String normalisedDomain = AppContext.domainNameUtils
                .normaliseDomainName(url.getHost());
        final String normalisedPath = normalisePath(url.getPath());

        final String hashValue = HashUtils
                .hash(normalisedDomain);
        return String.format("%s%s.%s%s%s%s.", normalisedPath, moduleId, AppContext.stringConstants.DOMAIN_NAME_PREFIX(), normalisedDomain, hashValue, AppContext.stringConstants.PREPOPULATED_RECORD_SUFFIX());
    }

    /**
     * Convert a URL to a query format suitable for use in a populator query.
     *
     * @param url      java.net.URL The URL to be converted.
     * @param moduleId java.lang.String the module number/id as a String
     * @return java.lang.String The converted URL result.
     * @throws NumInvalidParameterException on error
     * @throws NumBadURLException           on error
     */
    public String toPopulatorQuery(final URL url, final String moduleId) throws NumInvalidParameterException,
                                                                                NumBadURLException {
        validateParams(url, moduleId);

        final String normalisedDomain = AppContext.domainNameUtils
                .normaliseDomainName(url.getHost());
        final String normalisedPath = normalisePath(url.getPath());

        final String hashValue = HashUtils
                .hash(normalisedDomain);
        return String.format("%s%s.%s%s%s%s.", normalisedPath, moduleId, AppContext.stringConstants.DOMAIN_NAME_PREFIX(), normalisedDomain, hashValue, AppContext.stringConstants.POPULATOR_SERVICE_SUFFIX());
    }

    /**
     * Accept the 'path' part of a URL and convert it to a format for use in NUM Protocol Queries.
     * I.e. split by '/' and '.', reverse the results and join with '.', prefix with an underscore and
     * replace all spaces by underscores.
     *
     * @param path java.lang.String the path part of the URL - i.e. the result of URL.getPath()
     * @return java.lang.String the normalised path.
     */
    private String normalisePath(final String path) {
        String result = "";
        if (path != null && !path.isEmpty()) {
            final String[] pathComponents = path.split("[/.]");
            ArrayUtils.reverse(pathComponents);
            result = AppContext.stringConstants.DOMAIN_NAME_PREFIX() + StringUtils.join(pathComponents, ".");
            result = result.replaceAll(" ", "_");
        }
        return result;
    }
}
