/*
 *    Copyright 2020 NUM Technology Ltd
 *
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

package uk.num.net;

import lombok.Getter;
import lombok.NonNull;
import lombok.Setter;
import org.apache.commons.lang3.StringUtils;
import uk.num.numlib.api.*;
import uk.num.numlib.dns.DNSServices;
import uk.num.numlib.exc.NumNoRecordAvailableException;

import java.io.*;
import java.net.ContentHandlerFactory;
import java.net.URL;
import java.net.URLConnection;
import java.nio.charset.StandardCharsets;
import java.security.Permission;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;

public final class NUMURLConnection extends URLConnection {

    public static final String USE_POPULATOR = "NUM_USE_POPULATOR";

    public static final String HIDE_PARAMS = "NUM_HIDE_PARAMS";

    public static final String MODULES_LOCATION = "MODULES_LOCATION";

    @Getter
    @Setter
    private static DNSServices dnsServices = null;

    private NumAPIImpl numAPI = null;

    private NumAPIContext ctx = null;

    @Getter
    private boolean dnsSecSigned;

    @Getter
    private NumAPICallbacks.Location location;

    /**
     * Constructs a URL connection to the specified URL. A connection to
     * the object referenced by the URL is not created.
     *
     * @param url the specified URL.
     */
    public NUMURLConnection(@NonNull final URL url) {
        super(url);
    }

    /**
     * Sets the read timeout to a specified timeout, in
     * milliseconds. A non-zero value specifies the timeout when
     * reading from Input stream when a connection is established to a
     * resource. If the timeout expires before there is data available
     * for read, a java.net.SocketTimeoutException is raised. A
     * timeout of zero is interpreted as an infinite timeout.
     *
     * <p> Some non-standard implementation of this method ignores the
     * specified timeout. To see the read timeout set, please call
     * getReadTimeout().
     *
     * @param timeout an {@code int} that specifies the timeout
     *                value to be used in milliseconds
     * @throws IllegalArgumentException if the timeout parameter is negative
     * @see #getReadTimeout()
     * @see InputStream#read()
     * @since 1.5
     */
    @Override
    public void setReadTimeout(final int timeout) {
        super.setReadTimeout(timeout);
    }

    /**
     * Returns the value of this {@code URLConnection}'s {@code URL}
     * field.
     *
     * @return the value of this {@code URLConnection}'s {@code URL}
     * field.
     * @see URLConnection#url
     */
    @Override
    public URL getURL() {
        return super.getURL();
    }

    /**
     * Returns the value of the {@code content-length} header field.
     * <p>
     * <B>Note</B>: {@link #getContentLengthLong() getContentLengthLong()}
     * should be preferred over this method, since it returns a {@code long}
     * instead and is therefore more portable.</P>
     *
     * @return the content length of the resource that this connection's URL
     * references, {@code -1} if the content length is not known,
     * or if the content length is greater than Integer.MAX_VALUE.
     */
    @Override
    public int getContentLength() {
        return super.getContentLength();
    }

    /**
     * Returns the value of the {@code content-length} header field as a
     * long.
     *
     * @return the content length of the resource that this connection's URL
     * references, or {@code -1} if the content length is
     * not known.
     * @since 7.0
     */
    @Override
    public long getContentLengthLong() {
        return super.getContentLengthLong();
    }

    /**
     * Returns the value of the {@code content-type} header field.
     *
     * @return the content type of the resource that the URL references,
     * or {@code null} if not known.
     * @see URLConnection#getHeaderField(String)
     */
    @Override
    public String getContentType() {
        return super.getContentType();
    }

    /**
     * Returns the value of the {@code content-encoding} header field.
     *
     * @return the content encoding of the resource that the URL references,
     * or {@code null} if not known.
     * @see URLConnection#getHeaderField(String)
     */
    @Override
    public String getContentEncoding() {
        return super.getContentEncoding();
    }

    /**
     * Returns the value of the {@code expires} header field.
     *
     * @return the expiration date of the resource that this URL references,
     * or 0 if not known. The value is the number of milliseconds since
     * January 1, 1970 GMT.
     * @see URLConnection#getHeaderField(String)
     */
    @Override
    public long getExpiration() {
        return super.getExpiration();
    }

    /**
     * Returns the value of the {@code date} header field.
     *
     * @return the sending date of the resource that the URL references,
     * or {@code 0} if not known. The value returned is the
     * number of milliseconds since January 1, 1970 GMT.
     * @see URLConnection#getHeaderField(String)
     */
    @Override
    public long getDate() {
        return super.getDate();
    }

    /**
     * Returns the value of the {@code last-modified} header field.
     * The result is the number of milliseconds since January 1, 1970 GMT.
     *
     * @return the date the resource referenced by this
     * {@code URLConnection} was last modified, or 0 if not known.
     * @see URLConnection#getHeaderField(String)
     */
    @Override
    public long getLastModified() {
        return super.getLastModified();
    }

    /**
     * Returns the value of the named header field.
     * <p>
     * If called on a connection that sets the same header multiple times
     * with possibly different values, only the last value is returned.
     *
     * @param name the name of a header field.
     * @return the value of the named header field, or {@code null}
     * if there is no such field in the header.
     */
    @Override
    public String getHeaderField(final String name) {
        return super.getHeaderField(name);
    }

    /**
     * Returns an unmodifiable Map of the header fields.
     * The Map keys are Strings that represent the
     * response-header field names. Each Map value is an
     * unmodifiable List of Strings that represents
     * the corresponding field values.
     *
     * @return a Map of header fields
     * @since 1.4
     */
    @Override
    public Map<String, List<String>> getHeaderFields() {
        return super.getHeaderFields();
    }

    /**
     * Returns the value of the named field parsed as a number.
     * <p>
     * This form of {@code getHeaderField} exists because some
     * connection types (e.g., {@code http-ng}) have pre-parsed
     * headers. Classes for that connection type can override this method
     * and short-circuit the parsing.
     *
     * @param name    the name of the header field.
     * @param Default the default value.
     * @return the value of the named field, parsed as an integer. The
     * {@code Default} value is returned if the field is
     * missing or malformed.
     */
    @Override
    public int getHeaderFieldInt(final String name, final int Default) {
        return super.getHeaderFieldInt(name, Default);
    }

    /**
     * Returns the value of the named field parsed as a number.
     * <p>
     * This form of {@code getHeaderField} exists because some
     * connection types (e.g., {@code http-ng}) have pre-parsed
     * headers. Classes for that connection type can override this method
     * and short-circuit the parsing.
     *
     * @param name    the name of the header field.
     * @param Default the default value.
     * @return the value of the named field, parsed as a long. The
     * {@code Default} value is returned if the field is
     * missing or malformed.
     * @since 7.0
     */
    @Override
    public long getHeaderFieldLong(final String name, final long Default) {
        return super.getHeaderFieldLong(name, Default);
    }

    /**
     * Returns the value of the named field parsed as date.
     * The result is the number of milliseconds since January 1, 1970 GMT
     * represented by the named field.
     * <p>
     * This form of {@code getHeaderField} exists because some
     * connection types (e.g., {@code http-ng}) have pre-parsed
     * headers. Classes for that connection type can override this method
     * and short-circuit the parsing.
     *
     * @param name    the name of the header field.
     * @param Default a default value.
     * @return the value of the field, parsed as a date. The value of the
     * {@code Default} argument is returned if the field is
     * missing or malformed.
     */
    @Override
    public long getHeaderFieldDate(final String name, final long Default) {
        return super.getHeaderFieldDate(name, Default);
    }

    /**
     * Returns the key for the {@code n}<sup>th</sup> header field.
     * It returns {@code null} if there are fewer than {@code n+1} fields.
     *
     * @param n an index, where {@code n>=0}
     * @return the key for the {@code n}<sup>th</sup> header field,
     * or {@code null} if there are fewer than {@code n+1}
     * fields.
     */
    @Override
    public String getHeaderFieldKey(final int n) {
        return super.getHeaderFieldKey(n);
    }

    /**
     * Returns the value for the {@code n}<sup>th</sup> header field.
     * It returns {@code null} if there are fewer than
     * {@code n+1}fields.
     * <p>
     * This method can be used in conjunction with the
     * {@link #getHeaderFieldKey(int) getHeaderFieldKey} method to iterate through all
     * the headers in the message.
     *
     * @param n an index, where {@code n>=0}
     * @return the value of the {@code n}<sup>th</sup> header field
     * or {@code null} if there are fewer than {@code n+1} fields
     * @see URLConnection#getHeaderFieldKey(int)
     */
    @Override
    public String getHeaderField(final int n) {
        return super.getHeaderField(n);
    }

    /**
     * Retrieves the contents of this URL connection.
     * <p>
     * This method first determines the content type of the object by
     * calling the {@code getContentType} method. If this is
     * the first time that the application has seen that specific content
     * type, a content handler for that content type is created:
     * <ol>
     * <li>If the application has set up a content handler factory instance
     *     using the {@code setContentHandlerFactory} method, the
     *     {@code createContentHandler} method of that instance is called
     *     with the content type as an argument; the result is a content
     *     handler for that content type.
     * <li>If no content handler factory has yet been set up, or if the
     *     factory's {@code createContentHandler} method returns
     *     {@code null}, then the application loads the class named:
     *     <blockquote><pre>
     *         sun.net.www.content.&lt;<i>contentType</i>&gt;
     *     </pre></blockquote>
     *     where &lt;<i>contentType</i>&gt; is formed by taking the
     *     content-type string, replacing all slash characters with a
     *     {@code period} ('.'), and all other non-alphanumeric characters
     *     with the underscore character '{@code _}'. The alphanumeric
     *     characters are specifically the 26 uppercase ASCII letters
     *     '{@code A}' through '{@code Z}', the 26 lowercase ASCII
     *     letters '{@code a}' through '{@code z}', and the 10 ASCII
     *     digits '{@code 0}' through '{@code 9}'. If the specified
     *     class does not exist, or is not a subclass of
     *     {@code ContentHandler}, then an
     *     {@code UnknownServiceException} is thrown.
     * </ol>
     *
     * @return the object fetched. The {@code instanceof} operator
     * should be used to determine the specific kind of object
     * returned.
     * @throws IOException if an I/O error occurs while
     *                     getting the content.
     * @see ContentHandlerFactory#createContentHandler(String)
     * @see URLConnection#getContentType()
     * @see URLConnection#setContentHandlerFactory(ContentHandlerFactory)
     */
    @Override
    public Object getContent() throws IOException {
        return super.getContent();
    }

    /**
     * Retrieves the contents of this URL connection.
     *
     * @param classes the {@code Class} array
     *                indicating the requested types
     * @return the object fetched that is the first match of the type
     * specified in the classes array. null if none of
     * the requested types are supported.
     * The {@code instanceof} operator should be used to
     * determine the specific kind of object returned.
     * @throws IOException if an I/O error occurs while
     *                     getting the content.
     * @see URLConnection#getContent()
     * @see ContentHandlerFactory#createContentHandler(String)
     * @see URLConnection#getContent(Class[])
     * @see URLConnection#setContentHandlerFactory(ContentHandlerFactory)
     * @since 1.3
     */
    @Override
    public Object getContent(final Class[] classes) throws IOException {
        return super.getContent(classes);
    }

    /**
     * Returns a permission object representing the permission
     * necessary to make the connection represented by this
     * object. This method returns null if no permission is
     * required to make the connection. By default, this method
     * returns {@code java.security.AllPermission}. Subclasses
     * should override this method and return the permission
     * that best represents the permission required to make a
     * a connection to the URL. For example, a {@code URLConnection}
     * representing a {@code file:} URL would return a
     * {@code java.io.FilePermission} object.
     *
     * <p>The permission returned may dependent upon the state of the
     * connection. For example, the permission before connecting may be
     * different from that after connecting. For example, an HTTP
     * sever, say foo.com, may redirect the connection to a different
     * host, say bar.com. Before connecting the permission returned by
     * the connection will represent the permission needed to connect
     * to foo.com, while the permission returned after connecting will
     * be to bar.com.
     *
     * <p>Permissions are generally used for two purposes: to protect
     * caches of objects obtained through URLConnections, and to check
     * the right of a recipient to learn about a particular URL. In
     * the first case, the permission should be obtained
     * <em>after</em> the object has been obtained. For example, in an
     * HTTP connection, this will represent the permission to connect
     * to the host from which the data was ultimately fetched. In the
     * second case, the permission should be obtained and tested
     * <em>before</em> connecting.
     *
     * @return the permission object representing the permission
     * necessary to make the connection represented by this
     * URLConnection.
     * @throws IOException if the computation of the permission
     *                     requires network or file I/O and an exception occurs while
     *                     computing it.
     */
    @Override
    public Permission getPermission() throws IOException {
        return super.getPermission();
    }

    /**
     * Returns an input stream that reads from this open connection.
     * <p>
     * A SocketTimeoutException can be thrown when reading from the
     * returned input stream if the read timeout expires before data
     * is available for read.
     *
     * @return an input stream that reads from this open connection.
     * @throws IOException if an I/O error occurs while
     *                     creating the input stream.
     * @see #setReadTimeout(int)
     * @see #getReadTimeout()
     */
    @Override
    public InputStream getInputStream() throws IOException {
        try {
            if (!connected) {
                connect();
            }
            final NumAPICallbacksDefaultHandler handler = new NumAPICallbacksDefaultHandler();
            final Future<String> future = numAPI.retrieveNumRecord(ctx, handler, getReadTimeout());
            final String json = future.get(getReadTimeout(), TimeUnit.MILLISECONDS);
            numAPI.shutdown();
            connected = false;
            if (json != null) {
                dnsSecSigned = handler.isSignedDNSSEC();
                location = handler.getLocation();
                final InputStream stream = new ByteArrayInputStream(json.getBytes(StandardCharsets.UTF_8));
                return new BufferedInputStream(stream);
            }
            throw new IOException(new NumNoRecordAvailableException("No Record"));
        } catch (final Exception e) {
            throw new IOException(e);
        }
    }

    /**
     * Returns an output stream that writes to this connection.
     *
     * @return an output stream that writes to this connection.
     * @throws IOException if an I/O error occurs while
     *                     creating the output stream.
     */
    @Override
    public OutputStream getOutputStream() throws IOException {
        return super.getOutputStream();
    }

    /**
     * Returns a {@code String} representation of this URL connection.
     *
     * @return a string representation of this {@code URLConnection}.
     */
    @Override
    public String toString() {
        return super.toString();
    }

    /**
     * Returns the value of this {@code URLConnection}'s
     * {@code doInput} flag.
     *
     * @return the value of this {@code URLConnection}'s
     * {@code doInput} flag.
     * @see #setDoInput(boolean)
     */
    @Override
    public boolean getDoInput() {
        return super.getDoInput();
    }

    /**
     * Sets the value of the {@code doInput} field for this
     * {@code URLConnection} to the specified value.
     * <p>
     * A URL connection can be used for input and/or output.  Set the DoInput
     * flag to true if you intend to use the URL connection for input,
     * false if not.  The default is true.
     *
     * @param doinput the new value.
     * @throws IllegalStateException if already connected
     * @see URLConnection#doInput
     * @see #getDoInput()
     */
    @Override
    public void setDoInput(final boolean doinput) {
        super.setDoInput(doinput);
    }

    /**
     * Returns the value of this {@code URLConnection}'s
     * {@code doOutput} flag.
     *
     * @return the value of this {@code URLConnection}'s
     * {@code doOutput} flag.
     * @see #setDoOutput(boolean)
     */
    @Override
    public boolean getDoOutput() {
        return super.getDoOutput();
    }

    /**
     * Sets the value of the {@code doOutput} field for this
     * {@code URLConnection} to the specified value.
     * <p>
     * A URL connection can be used for input and/or output.  Set the DoOutput
     * flag to true if you intend to use the URL connection for output,
     * false if not.  The default is false.
     *
     * @param dooutput the new value.
     * @throws IllegalStateException if already connected
     * @see #getDoOutput()
     */
    @Override
    public void setDoOutput(final boolean dooutput) {
        super.setDoOutput(dooutput);
    }

    /**
     * Returns the value of the {@code allowUserInteraction} field for
     * this object.
     *
     * @return the value of the {@code allowUserInteraction} field for
     * this object.
     * @see #setAllowUserInteraction(boolean)
     */
    @Override
    public boolean getAllowUserInteraction() {
        return super.getAllowUserInteraction();
    }

    /**
     * Set the value of the {@code allowUserInteraction} field of
     * this {@code URLConnection}.
     *
     * @param allowuserinteraction the new value.
     * @throws IllegalStateException if already connected
     * @see #getAllowUserInteraction()
     */
    @Override
    public void setAllowUserInteraction(final boolean allowuserinteraction) {
        super.setAllowUserInteraction(allowuserinteraction);
    }

    /**
     * Returns the value of this {@code URLConnection}'s
     * {@code useCaches} field.
     *
     * @return the value of this {@code URLConnection}'s
     * {@code useCaches} field.
     * @see #setUseCaches(boolean)
     */
    @Override
    public boolean getUseCaches() {
        return super.getUseCaches();
    }

    /**
     * Sets the value of the {@code useCaches} field of this
     * {@code URLConnection} to the specified value.
     * <p>
     * Some protocols do caching of documents.  Occasionally, it is important
     * to be able to "tunnel through" and ignore the caches (e.g., the
     * "reload" button in a browser).  If the UseCaches flag on a connection
     * is true, the connection is allowed to use whatever caches it can.
     * If false, caches are to be ignored.
     * The default value comes from DefaultUseCaches, which defaults to
     * true.
     *
     * @param usecaches a {@code boolean} indicating whether
     *                  or not to allow caching
     * @throws IllegalStateException if already connected
     * @see #getUseCaches()
     */
    @Override
    public void setUseCaches(final boolean usecaches) {
        super.setUseCaches(usecaches);
    }

    /**
     * Returns the value of this object's {@code ifModifiedSince} field.
     *
     * @return the value of this object's {@code ifModifiedSince} field.
     * @see #setIfModifiedSince(long)
     */
    @Override
    public long getIfModifiedSince() {
        return super.getIfModifiedSince();
    }

    /**
     * Sets the value of the {@code ifModifiedSince} field of
     * this {@code URLConnection} to the specified value.
     *
     * @param ifmodifiedsince the new value.
     * @throws IllegalStateException if already connected
     * @see #getIfModifiedSince()
     */
    @Override
    public void setIfModifiedSince(final long ifmodifiedsince) {
        super.setIfModifiedSince(ifmodifiedsince);
    }

    /**
     * Returns the default value of a {@code URLConnection}'s
     * {@code useCaches} flag.
     * <p>
     * Ths default is "sticky", being a part of the static state of all
     * URLConnections.  This flag applies to the next, and all following
     * URLConnections that are created.
     *
     * @return the default value of a {@code URLConnection}'s
     * {@code useCaches} flag.
     * @see #setDefaultUseCaches(boolean)
     */
    @Override
    public boolean getDefaultUseCaches() {
        return super.getDefaultUseCaches();
    }

    /**
     * Sets the default value of the {@code useCaches} field to the
     * specified value.
     *
     * @param defaultusecaches the new value.
     * @see #getDefaultUseCaches()
     */
    @Override
    public void setDefaultUseCaches(final boolean defaultusecaches) {
        super.setDefaultUseCaches(defaultusecaches);
    }

    /**
     * Sets the general request property. If a property with the key already
     * exists, overwrite its value with the new value.
     *
     * <p> NOTE: HTTP requires all request properties which can
     * legally have multiple instances with the same key
     * to use a comma-separated list syntax which enables multiple
     * properties to be appended into a single property.
     *
     * @param key   the keyword by which the request is known
     *              (e.g., "{@code Accept}").
     * @param value the value associated with it.
     * @throws IllegalStateException if already connected
     * @throws NullPointerException  if key is <CODE>null</CODE>
     * @see #getRequestProperty(String)
     */
    @Override
    public void setRequestProperty(final String key, final String value) {
        super.setRequestProperty(key, value);
    }

    /**
     * Adds a general request property specified by a
     * key-value pair.  This method will not overwrite
     * existing values associated with the same key.
     *
     * @param key   the keyword by which the request is known
     *              (e.g., "{@code Accept}").
     * @param value the value associated with it.
     * @throws IllegalStateException if already connected
     * @throws NullPointerException  if key is null
     * @see #getRequestProperties()
     * @since 1.4
     */
    @Override
    public void addRequestProperty(final String key, final String value) {
        super.addRequestProperty(key, value);
    }

    /**
     * Returns the value of the named general request property for this
     * connection.
     *
     * @param key the keyword by which the request is known (e.g., "Accept").
     * @return the value of the named general request property for this
     * connection. If key is null, then null is returned.
     * @throws IllegalStateException if already connected
     * @see #setRequestProperty(String, String)
     */
    @Override
    public String getRequestProperty(final String key) {
        return super.getRequestProperty(key);
    }

    /**
     * Returns an unmodifiable Map of general request
     * properties for this connection. The Map keys
     * are Strings that represent the request-header
     * field names. Each Map value is a unmodifiable List
     * of Strings that represents the corresponding
     * field values.
     *
     * @return a Map of the general request properties for this connection.
     * @throws IllegalStateException if already connected
     * @since 1.4
     */
    @Override
    public Map<String, List<String>> getRequestProperties() {
        return super.getRequestProperties();
    }

    /**
     * Opens a communications link to the resource referenced by this
     * URL, if such a connection has not already been established.
     * <p>
     * If the {@code connect} method is called when the connection
     * has already been opened (indicated by the {@code connected}
     * field having the value {@code true}), the call is ignored.
     * <p>
     * URLConnection objects go through two phases: first they are
     * created, then they are connected.  After being created, and
     * before being connected, various options can be specified
     * (e.g., doInput and UseCaches).  After connecting, it is an
     * error to try to set them.  Operations that depend on being
     * connected, like getContentLength, will implicitly perform the
     * connection, if necessary.
     *
     * @throws IOException if an I/O error occurs while opening the
     *                     connection.
     * @see URLConnection#connected
     * @see #getConnectTimeout()
     * @see #setConnectTimeout(int)
     */
    @Override
    public void connect() throws IOException {
        if (!connected) {
            try {
                if (dnsServices != null) {
                    numAPI = new NumAPIImpl(dnsServices, null);
                } else {
                    numAPI = new NumAPIImpl();
                }

                ctx = numAPI.begin(url, 1000);
                final boolean hideParams = !("false".equalsIgnoreCase(getRequestProperty(HIDE_PARAMS)));
                final boolean populatorQueryRequired = "true".equalsIgnoreCase(getRequestProperty(USE_POPULATOR));

                final String modulesLocation = getRequestProperty(MODULES_LOCATION);
                if (modulesLocation != null) {
                    numAPI.setModulesLocation(modulesLocation);
                }
                ctx.setPopulatorQueryRequired(populatorQueryRequired);

                //
                // Get the required user variables - the client must populate values obtained from the user or from its own config data
                //
                final String query = url.getQuery();
                if (StringUtils.isNotEmpty(query)) {
                    final String[] parts = query.split("&");
                    final UserVariable[] userVariables = new UserVariable[parts.length];

                    int i = 0;
                    for (final String part : parts) {
                        final UserVariable uv = new UserVariable();
                        final String[] userVariableParts = part.split("=");

                        if (userVariableParts.length > 0) {
                            final String key = (hideParams) ? StringUtils.prependIfMissing(userVariableParts[0], "_") : userVariableParts[0];
                            uv.setKey(key);
                        }
                        if (userVariableParts.length > 1) {
                            uv.setValue(userVariableParts[1]);
                        }
                        userVariables[i] = uv;
                        i++;
                    }
                    ctx.setRequiredUserVariables(userVariables);
                }

                if (getReadTimeout() == 0) {
                    setReadTimeout(60000);
                }
                connected = true;
            } catch (final Exception e) {
                throw new IOException(e);
            }
        }
    }

}
