/**
*** Copyright (c) 2016-present,
*** Jaguar0625, gimre, BloodyRookie, Tech Bureau, Corp. All rights reserved.
***
*** This file is part of Catapult.
***
*** Catapult is free software: you can redistribute it and/or modify
*** it under the terms of the GNU Lesser General Public License as published by
*** the Free Software Foundation, either version 3 of the License, or
*** (at your option) any later version.
***
*** Catapult is distributed in the hope that it will be useful,
*** but WITHOUT ANY WARRANTY; without even the implied warranty of
*** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
*** GNU Lesser General Public License for more details.
***
*** You should have received a copy of the GNU Lesser General Public License
*** along with Catapult. If not, see <http://www.gnu.org/licenses/>.
**/

package io.nem.symbol.catapult.builders;

import java.io.DataInputStream;
import java.util.List;

/** Binary layout for a mosaic resolution statement. */
public final class MosaicResolutionStatementBuilder extends ReceiptBuilder implements Serializer {
    /** Unresolved mosaic. */
    private final UnresolvedMosaicIdDto unresolved;
    /** Resolution entries. */
    private final List<MosaicResolutionEntryBuilder> resolutionEntries;

    /**
     * Constructor - Creates an object from stream.
     *
     * @param stream Byte stream to use to serialize the object.
     */
    protected MosaicResolutionStatementBuilder(final DataInputStream stream) {
        super(stream);
        try {
            this.unresolved = UnresolvedMosaicIdDto.loadFromBinary(stream);
            this.resolutionEntries = new java.util.ArrayList<>();
            while (stream.available() > 0) {
                resolutionEntries.add(MosaicResolutionEntryBuilder.loadFromBinary(stream));
            }
        } catch(Exception e) {
            throw GeneratorUtils.getExceptionToPropagate(e);
        }
    }

    /**
     * Constructor.
     *
     * @param version Receipt version.
     * @param type Receipt type.
     * @param unresolved Unresolved mosaic.
     * @param resolutionEntries Resolution entries.
     */
    protected MosaicResolutionStatementBuilder(final short version, final ReceiptTypeDto type, final UnresolvedMosaicIdDto unresolved, final List<MosaicResolutionEntryBuilder> resolutionEntries) {
        super(version, type);
        GeneratorUtils.notNull(unresolved, "unresolved is null");
        GeneratorUtils.notNull(resolutionEntries, "resolutionEntries is null");
        this.unresolved = unresolved;
        this.resolutionEntries = resolutionEntries;
    }

    /**
     * Creates an instance of MosaicResolutionStatementBuilder.
     *
     * @param version Receipt version.
     * @param type Receipt type.
     * @param unresolved Unresolved mosaic.
     * @param resolutionEntries Resolution entries.
     * @return Instance of MosaicResolutionStatementBuilder.
     */
    public static MosaicResolutionStatementBuilder create(final short version, final ReceiptTypeDto type, final UnresolvedMosaicIdDto unresolved, final List<MosaicResolutionEntryBuilder> resolutionEntries) {
        return new MosaicResolutionStatementBuilder(version, type, unresolved, resolutionEntries);
    }

    /**
     * Gets unresolved mosaic.
     *
     * @return Unresolved mosaic.
     */
    public UnresolvedMosaicIdDto getUnresolved() {
        return this.unresolved;
    }

    /**
     * Gets resolution entries.
     *
     * @return Resolution entries.
     */
    public List<MosaicResolutionEntryBuilder> getResolutionEntries() {
        return this.resolutionEntries;
    }

    /**
     * Gets the size of the object.
     *
     * @return Size in bytes.
     */
    @Override
    public int getSize() {
        int size = super.getSize();
        size += this.unresolved.getSize();
        size += this.resolutionEntries.stream().mapToInt(o -> o.getSize()).sum();
        return size;
    }

    /**
     * Creates an instance of MosaicResolutionStatementBuilder from a stream.
     *
     * @param stream Byte stream to use to serialize the object.
     * @return Instance of MosaicResolutionStatementBuilder.
     */
    public static MosaicResolutionStatementBuilder loadFromBinary(final DataInputStream stream) {
        return new MosaicResolutionStatementBuilder(stream);
    }

    /**
     * Serializes an object to bytes.
     *
     * @return Serialized bytes.
     */
    public byte[] serialize() {
        return GeneratorUtils.serialize(dataOutputStream -> {
            final byte[] superBytes = super.serialize();
            dataOutputStream.write(superBytes, 0, superBytes.length);
            final byte[] unresolvedBytes = this.unresolved.serialize();
            dataOutputStream.write(unresolvedBytes, 0, unresolvedBytes.length);
            for (int i = 0; i < this.resolutionEntries.size(); i++) {
                final byte[] resolutionEntriesBytes = this.resolutionEntries.get(i).serialize();
                dataOutputStream.write(resolutionEntriesBytes, 0, resolutionEntriesBytes.length);
            }
        });
    }
}
