/*
 * Copyright (c) 2019. NUM Technology Ltd
 */

package uk.num.numlib.internal.module;

import uk.num.numlib.exc.NumBadURLException;
import uk.num.numlib.exc.NumInvalidParameterException;
import uk.num.numlib.internal.util.SimpleCache;

/**
 * A factory for ModuleDNSQuery objects.
 *
 * @author tonywalmsley
 */
public class ModuleFactory {
    /**
     * A cache for module/domain name combinations.
     */
    private static final SimpleCache<String, ModuleDNSQueries> moduleMap = new SimpleCache<>();

    /**
     * Default constructor
     */
    private ModuleFactory() {

    }

    /**
     * Create and initialise a ModuleDNSQueries object or use a cached object.
     *
     * @param moduleName the module name string, e.g. "1"
     * @param domainName the domain name, URL, or email address to be queried for a NUM record.
     * @return a ModuleDNSQueries object
     * @throws NumInvalidParameterException on error
     * @throws NumBadURLException on error
     */
    public static ModuleDNSQueries getInstance(final String moduleName, final String domainName) throws
                                                                                                 NumInvalidParameterException,
                                                                                                 NumBadURLException {
        if (moduleName == null || moduleName.trim().isEmpty()) {
            throw new NumInvalidParameterException("Invalid moduleName supplied: null or empty.");
        }
        if (domainName == null || domainName.trim().isEmpty()) {
            throw new NumInvalidParameterException("Invalid domainName supplied: null or empty.");
        }

        ModuleDNSQueries result;

        final String key = moduleName + "_" + domainName;
        // Critical section - we're reading then updating moduleMap, which is a potential race condition
        synchronized (moduleMap) {
            result = moduleMap.get(key);
            if (result == null) {
                result = new ModuleDNSQueries(moduleName, domainName);

                // Initialisation as a separate step since its an 'expensive' operation. Allows us to create lots of
                // Modules if necessary but then only initialise the ones we use.
                result.initialise();

                // Do this last in case there's an exception so we don't store an invalid ModuleDNSQueries object
                moduleMap.put(key, result);
            }
        }
        return result;
    }
}
