/*
 * Copyright (c) Terl Tech Ltd • 01/04/2021, 12:31 • goterl.com
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v2.0. If a copy of the MPL was not distributed with this
 * file, you can obtain one at http://mozilla.org/MPL/2.0/.
 */

package com.codevasp.lazysodium.interfaces;


import com.codevasp.lazysodium.exceptions.SodiumException;
import com.codevasp.lazysodium.utils.BaseChecker;
import com.codevasp.lazysodium.utils.Key;
import com.sun.jna.Structure;

import java.util.Arrays;
import java.util.List;

public interface Auth {

    enum Type {
        SHA256,
        SHA512,
        SHA512256
    }    int HMACSHA512256_BYTES = 32,
            HMACSHA512256_KEYBYTES = 32,
            HMACSHA256_BYTES = 32,
            HMACSHA256_KEYBYTES = 32,
            HMACSHA512_BYTES = 64,
            HMACSHA512_KEYBYTES = 32,

    BYTES = HMACSHA512256_BYTES,
            KEYBYTES = HMACSHA512256_KEYBYTES;


    interface Native {

        /**
         * Generate an authentication key.
         *
         * @param k Auth key of size {@link #KEYBYTES}.
         */
        void cryptoAuthKeygen(byte[] k);

        /**
         * Computes a tag for the message in, whose length is inLen bytes, and the key k.
         *
         * @param tag   Tag of size {@link #BYTES}.
         * @param in    A message.
         * @param inLen Message size.
         * @param key   The key as generated by {@link #cryptoAuthKeygen(byte[])}.
         * @return True if successful.
         */
        boolean cryptoAuth(byte[] tag, byte[] in, long inLen, byte[] key);

        /**
         * Verifies that the tag stored at h is a
         * valid tag for the message in whose length
         * is inLen bytes, and the key k.
         *
         * @param tag   The tag.
         * @param in    The message.
         * @param inLen Message bytes.
         * @param key   The key as generated by {@link #cryptoAuthKeygen(byte[])}.
         * @return True if successful verification.
         */
        boolean cryptoAuthVerify(byte[] tag, byte[] in, long inLen, byte[] key);


        void cryptoAuthHMACSha256Keygen(byte[] key);

        boolean cryptoAuthHMACSha256(
                byte[] out,
                byte[] in,
                long inLen,
                byte[] k
        );

        boolean cryptoAuthHMACSha256Verify(
                byte[] h,
                byte[] in,
                long inLen,
                byte[] k
        );

        boolean cryptoAuthHMACSha256Init(
                Auth.StateHMAC256 state,
                byte[] key,
                int keyLen
        );

        boolean cryptoAuthHMACSha256Update(
                Auth.StateHMAC256 state,
                byte[] in,
                long inLen
        );

        boolean cryptoAuthHMACSha256Final(
                Auth.StateHMAC256 state,
                byte[] out
        );


        void cryptoAuthHMACSha512Keygen(byte[] key);

        boolean cryptoAuthHMACSha512(
                byte[] out,
                byte[] in,
                long inLen,
                byte[] k
        );

        boolean cryptoAuthHMACSha512Verify(
                byte[] h,
                byte[] in,
                long inLen,
                byte[] k
        );

        boolean cryptoAuthHMACSha512Init(
                Auth.StateHMAC512 state,
                byte[] key,
                int keyLen
        );

        boolean cryptoAuthHMACSha512Update(
                Auth.StateHMAC512 state,
                byte[] in,
                long inLen
        );

        boolean cryptoAuthHMACSha512Final(
                Auth.StateHMAC512 state,
                byte[] out
        );


        void cryptoAuthHMACSha512256Keygen(byte[] key);

        boolean cryptoAuthHMACSha512256(
                byte[] out,
                byte[] in,
                long inLen,
                byte[] k
        );

        boolean cryptoAuthHMACSha512256Verify(
                byte[] h,
                byte[] in,
                long inLen,
                byte[] k
        );

        boolean cryptoAuthHMACSha512256Init(
                Auth.StateHMAC512256 state,
                byte[] key,
                int keyLen
        );

        boolean cryptoAuthHMACSha512256Update(
                Auth.StateHMAC512256 state,
                byte[] in,
                long inLen
        );

        boolean cryptoAuthHMACSha512256Final(
                Auth.StateHMAC512256 state,
                byte[] out
        );

    }

    interface Lazy {

        /**
         * Generate an authentication key.
         *
         * @return An authentication key.
         */
        Key cryptoAuthKeygen();

        /**
         * Computes a tag for the message in.
         *
         * @param message A message.
         * @param key     The key as generated by {@link #cryptoAuthKeygen()}.
         * @return True if successful.
         */
        String cryptoAuth(String message, Key key) throws SodiumException;


        /**
         * Verifies that the tag
         * valid tag for the message.
         *
         * @param tag The tag.
         * @param key The key as generated by {@link #cryptoAuthKeygen()}.
         * @return True if successful verification.
         */
        boolean cryptoAuthVerify(String tag, String message, Key key) throws SodiumException;


        Key cryptoAuthHMACShaKeygen(Type type);

        String cryptoAuthHMACSha(Type type, String in, Key key);

        boolean cryptoAuthHMACShaVerify(
                Type type,
                String authenticator,
                String message,
                Key key
        );

        boolean cryptoAuthHMACShaInit(
                Auth.StateHMAC256 state,
                Key key
        );

        boolean cryptoAuthHMACShaUpdate(
                Auth.StateHMAC256 state,
                String in
        );

        String cryptoAuthHMACShaFinal(
                Auth.StateHMAC256 state
        ) throws SodiumException;

        boolean cryptoAuthHMACShaInit(
                Auth.StateHMAC512 state,
                Key key
        );

        boolean cryptoAuthHMACShaUpdate(
                Auth.StateHMAC512 state,
                String in
        );

        String cryptoAuthHMACShaFinal(
                Auth.StateHMAC512 state
        ) throws SodiumException;


        boolean cryptoAuthHMACShaInit(
                Auth.StateHMAC512256 state,
                Key key
        );

        boolean cryptoAuthHMACShaUpdate(
                Auth.StateHMAC512256 state,
                String in
        );

        String cryptoAuthHMACShaFinal(
                Auth.StateHMAC512256 state
        ) throws SodiumException;


    }

    class Checker extends BaseChecker {

    }

    class StateHMAC256 extends Structure {
        public Hash.State256 ictx;
        public Hash.State256 octx;

        @Override
        protected List<String> getFieldOrder() {
            return Arrays.asList("ictx", "octx");
        }
    }

    class StateHMAC512 extends Structure {
        public Hash.State512 ictx;
        public Hash.State512 octx;


        @Override
        protected List<String> getFieldOrder() {
            return Arrays.asList("ictx", "octx");
        }
    }

    class StateHMAC512256 extends Structure {
        public Hash.State512 ictx;
        public Hash.State512 octx;

        @Override
        protected List<String> getFieldOrder() {
            return Arrays.asList("ictx", "octx");
        }
    }




}
