/*
 * Copyright (C) 2016 Codota
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.codota.service.client.requests;

import com.codota.service.client.CodotaHttpException;
import com.codota.service.client.CodotaResponse;
import com.codota.service.client.requests.base.Request;
import com.codota.service.connector.ConnectorSettings;
import com.codota.service.connector.ServiceConnector;
import com.codota.service.model.Bookmark;
import com.codota.service.model.Location;
import com.codota.service.utils.LogUtils;
import com.google.gson.JsonElement;
import com.google.gson.JsonParser;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;

/**
 * Created by yahave on 3/3/16.
 * (C) Codota
 */
public class DecryptBatchRequest extends Request {
    final Collection<Bookmark> bookmarks;

    public DecryptBatchRequest(ServiceConnector connector, String route, String token, Collection<Bookmark> bookmarks) {
        super(connector, route, token);
        this.bookmarks = bookmarks;
    }

    /**
     * misleading API - bookmarks are updated in-place, careful!
     */
    @Nullable
    public Collection<Bookmark> run() throws CodotaHttpException {
        Map<String, String[][]> payload = new HashMap<String, String[][]>();

        Map<String[], Bookmark> srcLineTriplets = new HashMap<String[], Bookmark>();
        Map<String[], Bookmark> locationTriplets = new HashMap<String[], Bookmark>();
        List<String[]> allTriplets = new ArrayList<String[]>();

        // 1. create triplets
        for (Bookmark b : bookmarks) {
            ConnectorSettings.LOG.debug("processing bookmark " + b._id);
            // we skip ones where the source line is null, which means that they will remain encrypted
            if (b.isEncrypted() && b.sourceLine != null) {
                ConnectorSettings.LOG.debug("collecting non-null sourceLine " + b._id);
                String[] srcLineTriplet = new String[]{b.encryption_key, b.encryption_iv, b.sourceLine};
                srcLineTriplets.put(srcLineTriplet, b);
                allTriplets.add(srcLineTriplet);
                if (b.location instanceof String) {
                    ConnectorSettings.LOG.debug("collecting bookmark location " + b._id);
                    String location = (String) b.location;
                    String[] locationTriplet = new String[]{b.encryption_key, b.encryption_iv, location};
                    locationTriplets.put(locationTriplet, b);
                    allTriplets.add(locationTriplet);
                }
            } else  {
                // nothing to decrypt
                ConnectorSettings.LOG.debug("nothing to decrypt " + b._id);
                b.setDecrypted(true);
            }
        }

        if (allTriplets.isEmpty()) {
            return bookmarks;
        }

        String[][] reqData = allTriplets.toArray(new String[allTriplets.size()][3]);

        payload.put("reqData", reqData);

        String g = gson.toJson(payload);

        CodotaResponse response = connector.postJson(getRoute(), g, getToken());

        if (response == null) {
            // even on failure, just return the original ones
            ConnectorSettings.LOG.debug("DecryptBatchRequest got null response, returning original bookmarks");
            return bookmarks;
        }

        String[] strings = handleDecryptBatchResponse(response);

        // inplace update for now, will create new bookmark eventually
        for (int i = 0; i < allTriplets.size(); i++) {
            String[] triplet = reqData[i];
            String decryptedData = strings[i];
            Bookmark bm = srcLineTriplets.get(triplet);
            if (bm != null) {
                ConnectorSettings.LOG.debug("DecryptBatchRequest: decrypted info " + bm._id);
                bm.setSourceLine(decryptedData);
                bm.setDecrypted(true);
            } else {
                bm = locationTriplets.get(triplet);
                if (bm != null) {
                    ConnectorSettings.LOG.debug("DecryptBatchRequest: decrypted location " + bm._id);
                    JsonParser parser = new JsonParser();
                    JsonElement root = parser.parse(decryptedData);
                    Location loc = gson.fromJson(root, Location.class);
                    bm.setLocation(loc);
                } else {
                    ConnectorSettings.LOG.debug("DecryptBatchRequest: something bad happened with " + Arrays.toString(triplet));
                }
            }
        }
        return bookmarks;
    }

    private String[] handleDecryptBatchResponse(@NotNull CodotaResponse response) throws CodotaHttpException {
        String json;
        if (response.isOK()) {
            json = response.content;
        } else {
            ConnectorSettings.LOG.debug("handleDecryptBatchResponse response code " + response.status);
            ConnectorSettings.LOG.warn("status: " + response.status);
            ConnectorSettings.LOG.warn("content: " + LogUtils.shorten(response.content));
            throw new CodotaHttpException(response.status, response.content);
        }
        JsonParser parser = new JsonParser();
        JsonElement root = parser.parse(json);

        return gson.fromJson(root, String[].class);
    }

}
