/*
 * Copyright (C) 2016 Codota
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.codota.service.connector;

import com.codota.service.client.CodotaResponse;
import com.codota.service.client.requests.base.GetRequest;
import com.codota.service.client.requests.base.PutRequest;
import com.codota.service.client.requests.base.Request;
import org.apache.http.HttpRequest;
import org.apache.http.HttpResponse;
import org.apache.http.NameValuePair;
import org.apache.http.client.HttpClient;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.HttpEntityEnclosingRequestBase;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.message.BasicNameValuePair;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.io.IOException;
import java.net.HttpURLConnection;
import java.net.URI;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

@SuppressWarnings("UnusedDeclaration")
public class ApacheServiceConnector extends ServiceConnector {

    private static final int TIMEOUT_SECONDS = 30;

    private static final RequestConfig config = RequestConfig.custom()
            .setSocketTimeout(TIMEOUT_SECONDS * 1000)
            .setConnectTimeout(TIMEOUT_SECONDS * 1000)
            .build();

    // the single httpclient of the entire system
    private static final HttpClient client = HttpClientBuilder.create().setDefaultRequestConfig(config).build();

    protected static ApacheServiceConnector instance;

    public ApacheServiceConnector(String base) {
        super(base);
    }

    public static ServiceConnector instance() {
        if (instance == null) {
            instance = new ApacheServiceConnector(ConnectorSettings.getDefaultBase());
        }
        return instance;
    }


    @Nullable
    public CodotaResponse getByURI(String uri, @Nullable String token) {
        tracker.init(this.getClass());
        HttpGet request = new HttpGet(uri);
        addFrameworkHeaders(request, token);

        tracker.connected();

        HttpResponse response;
        try {
            response = client.execute(request);
        } catch (IOException e) {
            ConnectorSettings.LOG.warn("getByURI failed " + e);
            return null;
        }

        tracker.gotResponse();

        String content;
        try {
            content = consumeResponse(response);
        } catch (IOException e) {
            e.printStackTrace();
            return null;
        }

        CodotaResponse result;
        if (response.getStatusLine().getStatusCode() == HttpURLConnection.HTTP_OK) {
            // set the actual result
            result = new CodotaResponse(content, HttpURLConnection.HTTP_OK);
        } else if (response.getStatusLine().getStatusCode() == HttpURLConnection.HTTP_UNAUTHORIZED) {
            result = new CodotaResponse(null,
                    HttpURLConnection.HTTP_UNAUTHORIZED);
        } else {
            result = new CodotaResponse(null, response.getStatusLine()
                    .getStatusCode());
        }
        tracker.finished();
        return result;
    }

    @Nullable
    @Override
    public CodotaResponse postJson(String route, @NotNull String jsonString,
                                   @Nullable String token) {
        HttpPost post = new HttpPost(route);
        post.addHeader("content-type", "application/json");
        CodotaResponse result = null;

        try {
            post.setEntity(new StringEntity(jsonString));
            addFrameworkHeaders(post, token);
            HttpResponse response = client.execute(post);
            String content = consumeResponse(response);

            result = new CodotaResponse(content, response.getStatusLine()
                    .getStatusCode());
        } catch (IOException e) {
            ConnectorSettings.LOG.warn("postJson failed " + e);
            e.printStackTrace();
        }
        return result;
    }

    @Override
    public CodotaResponse put(PutRequest request) {
        final HttpPut httpPut = new HttpPut();
        if (request.getBodyJsonString() != null) {
            httpPut.addHeader("content-type", "application/json");
            httpPut.setEntity(new StringEntity(request.getBodyJsonString(), UTF8));
        }
        return httpEntityEnclosingRequestBase(httpPut, request);
    }

    @Override
    public CodotaResponse post(Request request) {
        return httpEntityEnclosingRequestBase(new HttpPost(), request);
    }

    private CodotaResponse httpEntityEnclosingRequestBase(HttpEntityEnclosingRequestBase httpRequest, Request request) {
        CodotaResponse result = null;
        try {
            httpRequest.setURI(URI.create(getURI(request.getRoute(), "", request.getQueryParameters())));
            if (httpRequest.getEntity() == null) {
                addEncodedBodyAttributes(httpRequest, request.getBodyAttributes());
            }
            addFrameworkHeaders(httpRequest, request.getToken());
            addRequestHeaders(httpRequest, request);
            HttpResponse response = client.execute(httpRequest);
            String content = consumeResponse(response);
            result = new CodotaResponse(content, response.getStatusLine()
                    .getStatusCode());
        } catch (IOException e) {
            ConnectorSettings.LOG.warn(httpRequest.getMethod() + " failed " + e);
            e.printStackTrace();
        }
        return result;
    }

    @Override
    public CodotaResponse get(GetRequest request) {
        try {
            return getByURI(getURI(request.getRoute(), request.getQuery(), request.getQueryParameters()), request.getToken());
        } catch (IOException e) {
            e.printStackTrace();
            ConnectorSettings.LOG.warn("get failed " + e);
            return null;
        }
    }


    private void addFrameworkHeaders(HttpRequest request, String token) {
        request.addHeader("User-Agent", ConnectorSettings.USER_AGENT);
        if (token != null) {
            request.addHeader(AUTH_ATTR, "bearer " + token);
        }
        request.addHeader(ORIGIN_ATTR, ORIGIN);
        if (ConnectorSettings.getClientVersion() != null) {
            request.addHeader(CLIENT_ATTR, ConnectorSettings.getClientVersion());
        }
    }

    private void addRequestHeaders(HttpRequest httpRequest, Request request) {
        for (Map.Entry<String, String> headerEntry : request.getHeaderAttributes().entrySet()) {
            httpRequest.addHeader(headerEntry.getKey(), headerEntry.getValue());
        }
    }

    public void addEncodedBodyAttributes(HttpEntityEnclosingRequestBase request, Map<String, String> attributes) {
        List<NameValuePair> nameValuePairs = new ArrayList<NameValuePair>(1);
        for (String key : attributes.keySet()) {
            String value = attributes.get(key);
            BasicNameValuePair bvp = new BasicNameValuePair(key, value);
            nameValuePairs.add(bvp);
        }
        request.setEntity(new UrlEncodedFormEntity(nameValuePairs, UTF8));
    }

    @Nullable
    @Override
    public CodotaResponse post(String route, @NotNull Map<String, String> attributes,
                               @Nullable String token) {
        HttpPost post = new HttpPost(route);
        CodotaResponse result = null;
        try {
            addEncodedBodyAttributes(post, attributes);
            addFrameworkHeaders(post, token);
            HttpResponse response = client.execute(post);
            String content = consumeResponse(response);
            result = new CodotaResponse(content, response.getStatusLine()
                    .getStatusCode());
        } catch (IOException e) {
            ConnectorSettings.LOG.warn("post failed " + e);
            e.printStackTrace();
        }
        return result;
    }

    @SuppressWarnings("EmptyMethod")
    public void close() {
        // client.close();
    }

}
