/*
 * Copyright (C) 2016 Codota
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.codota.service.connector;

import com.codota.service.client.CodotaResponse;
import com.codota.service.client.EmptyTimeTracker;
import com.codota.service.client.ITimeTracker;
import com.codota.service.client.requests.base.GetRequest;
import com.codota.service.client.requests.base.PutRequest;
import com.codota.service.client.requests.base.Request;
import org.apache.http.HttpResponse;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.io.*;
import java.net.URLEncoder;
import java.nio.charset.Charset;
import java.util.Map;

@SuppressWarnings("UnusedDeclaration")
public abstract class ServiceConnector {

    protected static final String ENCODING = "UTF-8";
    protected static final String AUTH_ATTR = "authorization";
    protected static final String ORIGIN_ATTR = "origin";
    protected static final String ORIGIN = "https://intellij.codota.com";
    protected static final String CLIENT_ATTR = "apiversion";
    protected static final Charset UTF8 = Charset.forName("UTF-8");
    protected final ITimeTracker tracker = EmptyTimeTracker.instance();

    /**
     * A ServiceConnector has a single base URL string associated with it.
     * This string is set on construction, and cannot be modified (final).
     */
    private final String base;


    public ServiceConnector(String base) {
        this.base = base;
    }


    /**
     * note that it does not add "?" for the first parameter, that is the responsibility of the caller
     *
     * @throws UnsupportedEncodingException
     */
    @NotNull
    public static String encodeAttributes(@NotNull Map<String, String> attributes) throws UnsupportedEncodingException {
        StringBuilder result = new StringBuilder();
        boolean first = true;

        for (Map.Entry<String, String> pair : attributes.entrySet()) {
            if (first)
                first = false;
            else
                result.append("&");

            result.append(URLEncoder.encode(pair.getKey(), ENCODING));
            result.append("=");
            result.append(URLEncoder.encode(pair.getValue(), ENCODING));
        }
        return result.toString();
    }

    /**
     * @TODO: merge with other encode implementation, this is silly
     */
    @NotNull
    public static String encodeFullAttributes(@NotNull Map<String, String> attributes) throws UnsupportedEncodingException {
        StringBuilder result = new StringBuilder();
        boolean first = true;

        for (Map.Entry<String, String> pair : attributes.entrySet()) {
            if (first) {
                first = false;
                result.append("?");
            } else {
                result.append("&");
            }
            result.append(URLEncoder.encode(pair.getKey(), ENCODING));
            result.append("=");
            result.append(URLEncoder.encode(pair.getValue(), ENCODING));
        }
        return result.toString();
    }


    @NotNull
    public String getURI(String route, @Nullable String query, @Nullable Map<String, String> attributes) throws IOException {
        String encodedQuery = "";
        if (query != null) {
            encodedQuery = URLEncoder.encode(query, ENCODING);
        }
        String baseString =  route + encodedQuery;

        String encodedAttributes = "";
        if (attributes != null) {
            try {
                encodedAttributes = encodeFullAttributes(attributes);
            } catch (UnsupportedEncodingException e) {
                e.printStackTrace();
            }
        }
        return baseString + encodedAttributes;

    }

    @NotNull
    protected String consumeResponse(@NotNull HttpResponse response)
            throws IllegalStateException, IOException {
        StringBuilder sb = new StringBuilder();
        BufferedReader br = new BufferedReader(new InputStreamReader(
                new BufferedInputStream(response.getEntity().getContent()),
                "utf-8"));
        String line;
        while ((line = br.readLine()) != null) {
            sb.append(line).append("\n");
        }
        br.close();
        return sb.toString();
    }

    @SuppressWarnings("EmptyMethod")
    public void clear() {
    }


    @Nullable
    abstract public CodotaResponse post(String route,
                                        Map<String, String> attributes, String token);

    @Nullable
    abstract public CodotaResponse postJson(String route,
                                            String jsonString, String token);

    abstract public CodotaResponse post(Request request);
    abstract public CodotaResponse get(GetRequest request);
    abstract public CodotaResponse put(PutRequest request);

    public String getBase() {
        return base;
    }
}