/*
 * Copyright (C) 2016 Codota
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.codota.service.client;

import com.codota.service.client.requests.GetRequestEmptyResponse;
import com.codota.service.connector.ConnectorSettings;
import com.codota.service.connector.ServiceConnector;
import com.codota.service.model.UserInfo;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonElement;
import com.google.gson.JsonParser;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.net.HttpURLConnection;

@SuppressWarnings("UnusedDeclaration")
public class AuthClient {

    protected static final Gson gson = new GsonBuilder().create();

    private static final String TOKEN_TEST = "/api/auth/user/tokentest?access_token=";
    private static final String USER_INFO = "/api/auth/user/tokentest";

    @Nullable
    protected String token;
    protected final ServiceConnector connector;

    public AuthClient(ServiceConnector connector) {
        this.connector = connector;
    }


    public void clearToken() {
        this.token = null;
        // clear the cache if it exists, clean previous requests with token now invalidated
        connector.clear();
    }

    public void setToken(@NotNull String token) {
        this.token = token;
        // clear the cache if it exists, clean previous requests without valid token
        connector.clear();
    }

    public boolean isTokenValid(String query) {
        try {
            CodotaResponse response = connector.get(new GetRequestEmptyResponse(connector,connector.getBase() + TOKEN_TEST,null,query));
            return (response != null && response.status == HttpURLConnection.HTTP_OK);
        } catch (Exception e) {
            // swallow anything, we cannot afford to leak user info here
        }
        return false;
    }

    @NotNull
    public UserInfo userInfo() {
        CodotaResponse response = connector.get(new GetRequestEmptyResponse(connector, connector.getBase() + USER_INFO, token));
        if (response != null && response.status == HttpURLConnection.HTTP_OK) {
            return parseUserInfoJson(response.content);
        } else {
            return UserInfo.UNKNOWN;
        }
    }

    public UserInfo parseUserInfoJson(@NotNull String json) {
        UserInfo userInfo = UserInfo.UNKNOWN;
        try {
            JsonParser parser = new JsonParser();
            JsonElement root = parser.parse(json);
            userInfo = gson.fromJson(root, UserInfo.class);
        } catch (IllegalStateException e) {
            System.err.println("json" + json);
            e.printStackTrace();
        } catch (com.google.gson.JsonSyntaxException e) {
            System.err.println("json" + json);
            e.printStackTrace();
        }
        return userInfo;
    }

}
