/*
 * Copyright (C) 2016 Codota
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.codota.service.client;

import com.codota.service.client.requests.*;
import com.codota.service.connector.ServiceConnector;
import com.codota.service.model.*;
import org.jetbrains.annotations.Nullable;
import org.jetbrains.annotations.NotNull;


import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.*;

@SuppressWarnings("UnusedDeclaration")
public class SearchClient extends AuthClient {

    private static final String XREF_METHODS_ROUTE = "/api/completions/";


    @Nullable
    private static SearchClient instance;
    private String defaultCodePack;

    public static void clearClient() {
        instance = null;
    }

    public SearchClient(ServiceConnector connector) {
        super(connector);
    }

    @NotNull
    public static SearchClient client(ServiceConnector connector) {
        if (instance == null) {
            instance = new SearchClient(connector);
        }
        return instance;
    }

    public void setDefaultCodePack(String codePack) {
        this.defaultCodePack = codePack;
    }

    public SearchResults search(String query) throws
            CodotaHttpException, CodotaConnectionException {
        return new SearchRequest(connector, token, query).run();
    }

    public SearchResults searchByTask(@SuppressWarnings("SameParameterValue") String query) throws
            CodotaHttpException, CodotaConnectionException {
        return new SearchTaskRequest(connector, token, query).run();

    }

    public List<TypeaheadResult> typeahead(@SuppressWarnings("SameParameterValue") String prefix) throws
            CodotaHttpException, CodotaConnectionException {
        return new TypeaheadRequest(connector, token, prefix).run();
    }


    public XRefTypeaheadResult xreftypeahead(String codepack, @SuppressWarnings("SameParameterValue") String prefix) throws
            CodotaHttpException, CodotaConnectionException {
        return new XRefTypeaheadRequest(connector, prefix, makeBasicPropertyMap(codepack), token).run();
    }


    public XRefTypeaheadResult xreftypeahead(@SuppressWarnings("SameParameterValue") String prefix) throws
            CodotaHttpException, CodotaConnectionException {
        return xreftypeahead(null,prefix);
    }

    public List<XRefMethodItem> xrefMethodsPerClass(String codepack, String classKey) throws
            CodotaHttpException, CodotaConnectionException {

        Map<String,String> props = makeBasicPropertyMap(codepack);
        props.put("derived", "false");

        try {
            // this ugly thing is here because XRefMethodsRequest neesd the route to call super
            // TODO: refactor this case and make the route construction internal to XRefMethodsRequest
            String route = "/api/completions/"+ URLEncoder.encode(codepack, "UTF-8") + "/methods/" + URLEncoder.encode(classKey, "UTF-8");
            return new XRefMethodsRequest(connector, route, props, token).run();
        } catch (UnsupportedEncodingException e) {
            e.printStackTrace();
        }
        return null;
    }

    public CrossRefResults searchCrossRef(Map<String, String> props) throws
            CodotaHttpException, CodotaConnectionException {
        return new SearchCrossRefRequest(connector, props, token).run();
    }


    public List<TextualMatch> textSearch(String term, boolean all) throws
            CodotaHttpException, CodotaConnectionException {
        return new TextSearchRequest(connector, term, defaultCodePack, all, token).run();
    }

    public DependencyInfo getDependencies(String filepath, String artifactName) throws
            CodotaHttpException, CodotaConnectionException {
        return new GetDependenciesRequest(connector, filepath, artifactName, defaultCodePack, token).run();
    }

    public Map<String, DependencyInfo> getArtifactDependencies(String artifactName) throws
            CodotaHttpException, CodotaConnectionException {
        return new GetArtifactDependenciesRequest(connector, artifactName, defaultCodePack, token).run();
    }

    public Artifact readArtifact(String artifactName) throws
            CodotaHttpException, CodotaConnectionException {
        return new GetArtifactRequest(connector, artifactName, defaultCodePack, token).run();
    }

    public List<String> allFilesForArtifact(String artifactName) throws
            CodotaHttpException, CodotaConnectionException {
        return new ListFilesRequest(connector, artifactName, defaultCodePack, token).run();
    }

    public List<String> allClassesForArtifact(String artifactName) throws
            CodotaHttpException, CodotaConnectionException {
        return new ListClassesRequest(connector, artifactName, defaultCodePack, token).run();
    }


    public SearchStats searchStats(Set<String> queries) throws CodotaHttpException, CodotaConnectionException {
        return new SearchStatsRequest(connector, token, queries).run();
    }


    /**
     * misleading API - bookmarks are updated in-place, careful!
     */
    @Nullable
    public Collection<Bookmark> decryptBatch(String decrypt_url, Collection<Bookmark> bookmarks) throws CodotaHttpException {
        return new DecryptBatchRequest(connector, decrypt_url, token, bookmarks).run();
    }

    @Nullable
    public String decrypt(String decrypt_url, Bookmark bookmark) throws CodotaHttpException {
        return new DecryptRequest(connector, decrypt_url, token, bookmark).run();
    }

    public List<String> getManualDependencies(String codePack, String artifactName, String filePath)
            throws CodotaHttpException, CodotaConnectionException {
        final GetManualDependenciesRequest request = new GetManualDependenciesRequest(connector,
                artifactName, codePack, filePath, token);
        final List<String> response = request.run();
        return response;
    }

    /**
     * Attaches dependencies to a source file.
     * If the source file already has attached dependencies, they shall be replaced with the newly given ones.
     * Returns true iff the operation succeeded.
     *
     * @param codePack The code pack to which the artifcat belongs.
     * @param artifactName The name of the artifact to which the source file belongs.
     * @param filePath The path of the file to which the dependencies shall be attached.
     * @param dependenciesFullyQualifiedNames The fully qualified names of the dependencies to attach.
     *                                        use '/' instead of '.' between package names.
     * @return
     * @throws CodotaHttpException
     * @throws CodotaConnectionException
     */
    public void putManualDependencies(String codePack, String artifactName, String filePath,
                                         Set<String> dependenciesFullyQualifiedNames)
            throws CodotaHttpException, CodotaConnectionException {
        final PutManualDependenciesRequest request = new PutManualDependenciesRequest(connector,
                token,codePack, artifactName, filePath, dependenciesFullyQualifiedNames);
        final String response = request.run();
        if (!"OK".equals(response.trim())) {
            throw new RuntimeException("Error putting manual dependencies: " + response);
        }
    }

    /**
     * create a property map containing the given code pack, or the default one if no codepack is provided
     */
    @NotNull
    private Map<String,String> makeBasicPropertyMap(@Nullable String codepack) {
        Map<String,String> props = new HashMap<String, String>();
        if (codepack == null) {
            // if given codepack was null, we take the default one instead
            codepack = defaultCodePack;
        }
        // sadly, even after trying the default one, the codepack may still be null
        // so we check before putting it into the property map
        if (codepack != null) {
            props.put("codePack", codepack);
        }
        return props;
    }


}
