/*
 * Decompiled with CFR 0.152.
 */
package com.github.containersolutions.operator.processing;

import com.github.containersolutions.operator.processing.CustomResourceEvent;
import com.github.containersolutions.operator.processing.EventConsumer;
import com.github.containersolutions.operator.processing.EventDispatcher;
import com.github.containersolutions.operator.processing.EventStore;
import com.github.containersolutions.operator.processing.retry.Retry;
import io.fabric8.kubernetes.client.CustomResource;
import io.fabric8.kubernetes.client.KubernetesClientException;
import io.fabric8.kubernetes.client.Watcher;
import java.util.Optional;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReentrantLock;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class EventScheduler
implements Watcher<CustomResource> {
    private static final Logger log = LoggerFactory.getLogger(EventScheduler.class);
    private final EventDispatcher eventDispatcher;
    private final ScheduledThreadPoolExecutor executor;
    private final EventStore eventStore = new EventStore();
    private final Retry retry;
    private final boolean generationAware;
    private ReentrantLock lock = new ReentrantLock();

    public EventScheduler(EventDispatcher eventDispatcher, Retry retry, boolean generationAware) {
        this.eventDispatcher = eventDispatcher;
        this.retry = retry;
        this.generationAware = generationAware;
        this.executor = new ScheduledThreadPoolExecutor(1);
        this.executor.setRemoveOnCancelPolicy(true);
    }

    public void eventReceived(Watcher.Action action, CustomResource resource) {
        log.debug("Event received for action: {}, {}: {}", new Object[]{action.toString().toLowerCase(), resource.getClass().getSimpleName(), resource.getMetadata().getName()});
        CustomResourceEvent event = new CustomResourceEvent(action, resource, this.retry);
        this.scheduleEventFromApi(event);
    }

    void scheduleEventFromApi(CustomResourceEvent event) {
        try {
            this.lock.lock();
            log.debug("Scheduling event from Api: {}", (Object)event);
            if (event.getResource().getMetadata().getDeletionTimestamp() != null && event.getAction() == Watcher.Action.DELETED) {
                log.debug("Skipping delete event since deletion timestamp is present on resource, so finalizer was in place.");
                return;
            }
            if (this.eventStore.containsNotScheduledEvent(event.resourceUid())) {
                log.debug("Replacing not scheduled event with actual event. New event: {}", (Object)event);
                this.eventStore.addOrReplaceEventAsNotScheduledAndUpdateLastGeneration(event);
                return;
            }
            if (this.generationAware && !this.eventStore.hasLargerGenerationThanLastStored(event)) {
                log.debug("Skipping event, has not larger generation than last stored, actual generation: {}, last stored: {} ", (Object)event.getResource().getMetadata().getGeneration(), (Object)this.eventStore.getLastStoredGeneration(event));
                return;
            }
            if (this.eventStore.containsEventUnderProcessing(event.resourceUid())) {
                log.debug("Scheduling event for later processing since there is an event under processing for same kind. New event: {}", (Object)event);
                this.eventStore.addOrReplaceEventAsNotScheduledAndUpdateLastGeneration(event);
                return;
            }
            this.scheduleEventForExecution(event);
            log.trace("Scheduling event from API finished: {}", (Object)event);
        }
        finally {
            this.lock.unlock();
        }
    }

    private void scheduleEventForExecution(CustomResourceEvent event) {
        try {
            this.lock.lock();
            log.trace("Current queue size {}", (Object)this.executor.getQueue().size());
            log.debug("Scheduling event for execution: {}", (Object)event);
            Optional<Long> nextBackOff = event.nextBackOff();
            if (!nextBackOff.isPresent()) {
                log.warn("Event max retry limit reached. Will be discarded. {}", (Object)event);
                return;
            }
            this.eventStore.addEventUnderProcessingAndUpdateLastGeneration(event);
            this.executor.schedule(new EventConsumer(event, this.eventDispatcher, this), (long)nextBackOff.get(), TimeUnit.MILLISECONDS);
            log.trace("Scheduled task for event: {}", (Object)event);
        }
        finally {
            this.lock.unlock();
        }
    }

    void eventProcessingFinishedSuccessfully(CustomResourceEvent event) {
        try {
            this.lock.lock();
            this.eventStore.removeEventUnderProcessing(event.resourceUid());
            if (this.eventStore.containsNotScheduledEvent(event.resourceUid())) {
                log.debug("Scheduling recent event for processing: {}", (Object)event);
                this.scheduleNotYetScheduledEventForExecution(event.resourceUid());
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    void eventProcessingFailed(CustomResourceEvent event) {
        try {
            this.lock.lock();
            this.eventStore.removeEventUnderProcessing(event.resourceUid());
            if (this.eventStore.containsNotScheduledEvent(event.resourceUid())) {
                log.debug("Event processing failed. Scheduling the most recent event. Failed event: {}", (Object)event);
                this.scheduleNotYetScheduledEventForExecution(event.resourceUid());
            } else {
                log.debug("Event processing failed. Attempting to re-schedule the event: {}", (Object)event);
                this.scheduleEventForExecution(event);
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    private void scheduleNotYetScheduledEventForExecution(String uuid) {
        CustomResourceEvent notScheduledEvent = this.eventStore.removeEventNotScheduled(uuid);
        this.scheduleEventForExecution(notScheduledEvent);
    }

    public void onClose(KubernetesClientException e) {
        log.error("Error: ", (Throwable)e);
        System.exit(1);
    }
}

