/*
 * Decompiled with CFR 0.152.
 */
package com.github.hypfvieh.db;

import com.github.hypfvieh.util.StringUtil;
import com.github.hypfvieh.util.SystemUtil;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.TreeSet;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class HyperSqlUtil {
    private static final List<String> HYPERSQL_EXTENSIONS = Arrays.asList(".script", ".properties", ".data", ".log", ".tmp", ".lck", ".lobs");
    private final Logger logger = LoggerFactory.getLogger(this.getClass());
    private final Set<String> copiedHsqlBasenames = new TreeSet<String>();
    private final String dbBaseName;
    private final File targetDir;
    private final boolean targetDirCreated;
    private final boolean deleteOnExit;

    public HyperSqlUtil(String dbBaseName) {
        this(dbBaseName, HyperSqlUtil.createTempDir(), true);
    }

    public HyperSqlUtil(String dbBaseName, File targetDir, boolean deleteOnExit) {
        this.dbBaseName = Objects.requireNonNull(dbBaseName, "Database base name required").replace("\\", "/");
        this.deleteOnExit = deleteOnExit;
        this.targetDir = Objects.requireNonNull(targetDir, "Target directory required");
        this.cleanupTargetDir();
        this.targetDirCreated = this.createTargetDir(targetDir);
        this.logger.debug("{} created with target dir '{}'", (Object)this.getClass().getSimpleName(), (Object)this.targetDir);
    }

    private boolean createTargetDir(File targetDir) {
        if (!targetDir.exists()) {
            if (!targetDir.mkdirs()) {
                throw new RuntimeException("Unable to create target directory " + String.valueOf(targetDir));
            }
            return true;
        }
        if (!(targetDir.canRead() && targetDir.canWrite() || targetDir.setReadable(true) && targetDir.setWritable(true))) {
            throw new RuntimeException("Unable to set read/write permissions on target directory " + String.valueOf(targetDir));
        }
        return false;
    }

    public String getDbPath() {
        return String.valueOf(this.targetDir) + "/" + this.extractFileName(this.dbBaseName);
    }

    private String extractFileName(String dbBaseName) {
        return StringUtil.substringAfterLast(dbBaseName, "/");
    }

    HyperSqlUtil copyHsqlDbFiles(String searchPath, String dbBaseName, boolean overwrite, String ... dbExtensions) {
        this.logger.debug("Copying hsql database files with base name '{}'", (Object)dbBaseName);
        File destFile = null;
        for (String ext : dbExtensions) {
            String srcFileName = dbBaseName + ext;
            String pkgSrcFileName = (String)(searchPath == null ? "" : searchPath + "/") + srcFileName;
            try (InputStream is = HyperSqlUtil.class.getClassLoader().getResourceAsStream(pkgSrcFileName);){
                if (is == null) {
                    if (".script".equalsIgnoreCase(ext)) {
                        throw new RuntimeException("Required database file '" + pkgSrcFileName + "' does not exist");
                    }
                    this.logger.warn("Skipping non-existing file '{}'", (Object)pkgSrcFileName);
                    continue;
                }
                this.copiedHsqlBasenames.add(dbBaseName);
                Objects.requireNonNull(is, "Could not get input " + srcFileName);
                destFile = new File(this.targetDir, this.extractFileName(srcFileName));
                if (this.deleteOnExit) {
                    destFile.deleteOnExit();
                }
                if (destFile.exists() && !overwrite) {
                    this.logger.info("Not copying file '{}' to existing file '{}'", (Object)srcFileName, (Object)destFile);
                    continue;
                }
                this.logger.debug("Copying file '{}' to '{}'", (Object)srcFileName, (Object)destFile);
                Files.copy(is, destFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
            }
            catch (IOException ex) {
                throw new UncheckedIOException("Failed to copy " + srcFileName + " to " + String.valueOf(destFile), ex);
            }
        }
        return this;
    }

    public HyperSqlUtil copyHsqlDbFiles(String searchPath, boolean overwrite) {
        return this.copyHsqlDbFiles(searchPath, this.dbBaseName, overwrite, ".script", ".properties", ".data", ".lobs");
    }

    public HyperSqlUtil copyHsqlDbFiles() {
        return this.copyHsqlDbFiles(null, true);
    }

    public void cleanupTargetDir() {
        this.logger.debug("Cleaning up target dir '{}'", (Object)this.targetDir);
        if (!this.targetDir.exists()) {
            this.logger.debug("Cannot delete non-existing target directory '{}'", (Object)this.targetDir);
            return;
        }
        for (String baseTarget : this.copiedHsqlBasenames) {
            for (String ext : HYPERSQL_EXTENSIONS) {
                File file = new File(this.targetDir, baseTarget + ext);
                if (!file.exists()) continue;
                if (file.isDirectory()) {
                    this.logger.debug("Deleting dir '{}'", (Object)file);
                    SystemUtil.deleteRecursivelyQuiet(file.getAbsolutePath());
                    continue;
                }
                this.logger.debug("Deleting file '{}'", (Object)file);
                boolean deleted = file.delete();
                if (deleted) continue;
                this.logger.warn("Failed to delete file '{}'", (Object)file);
            }
        }
        if (this.targetDirCreated) {
            this.logger.debug("Deleting target directory '{}'", (Object)this.targetDir);
            SystemUtil.deleteRecursivelyQuiet(this.targetDir.getAbsolutePath());
        }
    }

    public File getTargetDir() {
        return this.targetDir;
    }

    public String toString() {
        return this.getClass().getSimpleName() + "[baseName=" + this.dbBaseName + ", targetDir=" + String.valueOf(this.targetDir) + "]";
    }

    private static File createTempDir() {
        String subDirName = StackWalker.getInstance(StackWalker.Option.RETAIN_CLASS_REFERENCE).walk(e -> e.filter(f -> !f.getClassName().equals(HyperSqlUtil.class.getName())).findFirst()).map(f -> StringUtil.substringAfterLast(f.getClassName(), ".")).orElse("NOT_FOUND");
        File tmpDir = new File(System.getProperty("java.io.tmpdir"), subDirName);
        tmpDir.deleteOnExit();
        return tmpDir;
    }
}

