package internal.chart;

import ec.util.chart.ColorScheme;
import java.lang.Iterable;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.ServiceLoader;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;

/**
 * Custom service loader for {@link ec.util.chart.ColorScheme}.
 * <br>This class is thread-safe.
 * <p>Properties:
 * <ul>
 * <li>Quantifier: MULTIPLE</li>
 * <li>Fallback: null</li>
 * <li>Preprocessing: null</li>
 * <li>Mutability: CONCURRENT</li>
 * <li>Singleton: true</li>
 * <li>Name: internal.chart.ColorSchemeLoader</li>
 * <li>Backend: null</li>
 * <li>Cleaner: null</li>
 * <li>Batch: false</li>
 * <li>Batch name: null</li>
 * </ul>
 */
public final class ColorSchemeLoader {
  private static final Iterable<ColorScheme> SOURCE = ServiceLoader.load(ColorScheme.class);

  private static final AtomicReference<List<ColorScheme>> RESOURCE = new AtomicReference<>(doLoad());

  private static final Consumer<Iterable> CLEANER = loader -> ((ServiceLoader)loader).reload();

  private ColorSchemeLoader() {
  }

  private static List<ColorScheme> doLoad() {
    return StreamSupport.stream(SOURCE.spliterator(), false)
        .collect(Collectors.collectingAndThen(Collectors.toList(), Collections::unmodifiableList));
  }

  /**
   * Gets a list of {@link ec.util.chart.ColorScheme} instances.
   * <br>This method is thread-safe.
   * @return the current non-null value
   */
  public static List<ColorScheme> get() {
    return RESOURCE.get();
  }

  /**
   * Sets a list of {@link ec.util.chart.ColorScheme} instances.
   * <br>This method is thread-safe.
   * @param newValue new non-null value
   */
  public static void set(List<ColorScheme> newValue) {
    RESOURCE.set(Objects.requireNonNull(newValue));
  }

  /**
   * Reloads the content by clearing the cache and fetching available providers.
   * <br>This method is thread-safe.
   */
  public static void reload() {
    synchronized(SOURCE) {
      CLEANER.accept(SOURCE);
      set(doLoad());
    }
  }

  /**
   * Resets the content without clearing the cache.
   * <br>This method is thread-safe.
   */
  public static void reset() {
    synchronized(SOURCE) {
      set(doLoad());
    }
  }
}
