package internal.chart.swing;

import ec.util.chart.swing.JFreeChartWriter;
import java.lang.Iterable;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.ServiceLoader;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;

/**
 * Custom service loader for {@link ec.util.chart.swing.JFreeChartWriter}.
 * <br>This class is thread-safe.
 * <p>Properties:
 * <ul>
 * <li>Quantifier: MULTIPLE</li>
 * <li>Fallback: null</li>
 * <li>Preprocessing: null</li>
 * <li>Mutability: CONCURRENT</li>
 * <li>Singleton: true</li>
 * <li>Name: internal.chart.swing.JFreeChartWriterLoader</li>
 * <li>Backend: null</li>
 * <li>Cleaner: null</li>
 * <li>Batch: false</li>
 * <li>Batch name: null</li>
 * </ul>
 */
public final class JFreeChartWriterLoader {
  private static final Iterable<JFreeChartWriter> SOURCE = ServiceLoader.load(JFreeChartWriter.class);

  private static final AtomicReference<List<JFreeChartWriter>> RESOURCE = new AtomicReference<>(doLoad());

  private static final Consumer<Iterable> CLEANER = loader -> ((ServiceLoader)loader).reload();

  private JFreeChartWriterLoader() {
  }

  private static List<JFreeChartWriter> doLoad() {
    return StreamSupport.stream(SOURCE.spliterator(), false)
        .collect(Collectors.collectingAndThen(Collectors.toList(), Collections::unmodifiableList));
  }

  /**
   * Gets a list of {@link ec.util.chart.swing.JFreeChartWriter} instances.
   * <br>This method is thread-safe.
   * @return the current non-null value
   */
  public static List<JFreeChartWriter> get() {
    return RESOURCE.get();
  }

  /**
   * Sets a list of {@link ec.util.chart.swing.JFreeChartWriter} instances.
   * <br>This method is thread-safe.
   * @param newValue new non-null value
   */
  public static void set(List<JFreeChartWriter> newValue) {
    RESOURCE.set(Objects.requireNonNull(newValue));
  }

  /**
   * Reloads the content by clearing the cache and fetching available providers.
   * <br>This method is thread-safe.
   */
  public static void reload() {
    synchronized(SOURCE) {
      CLEANER.accept(SOURCE);
      set(doLoad());
    }
  }

  /**
   * Resets the content without clearing the cache.
   * <br>This method is thread-safe.
   */
  public static void reset() {
    synchronized(SOURCE) {
      set(doLoad());
    }
  }
}
