package com.github.xuejike.query.dataverse;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.convert.Convert;
import cn.hutool.core.util.StrUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.github.xuejike.query.core.base.BaseDao;
import com.github.xuejike.query.core.criteria.IJPage;
import com.github.xuejike.query.core.enums.OrderType;
import com.github.xuejike.query.core.exception.LambdaQueryException;
import com.github.xuejike.query.core.po.FieldInfo;
import com.github.xuejike.query.core.po.QueryInfo;
import com.github.xuejike.query.dataverse.annotation.DataverseField;
import com.github.xuejike.query.dataverse.annotation.FormattedValue;
import com.github.xuejike.query.dataverse.client.DataverseClient;
import com.github.xuejike.query.dataverse.client.ODataQuery;
import com.github.xuejike.query.dataverse.client.ODataResponse;
import lombok.extern.slf4j.Slf4j;

import java.io.Serializable;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * DAO implementation for Microsoft Dataverse.
 * Extends BaseDao to provide Lambda-Query functionality for Dataverse entities.
 *
 * @param <T> the entity type
 * @author xuejike
 */
@Slf4j
public class DataverseDao<T> extends BaseDao<T, ODataQuery> {
    
    private final DataverseClient client;
    private final String entityName;
    private final String primaryKey;
    
    /**
     * Create a new DataverseDao.
     *
     * @param client the Dataverse client
     * @param entityCls the entity class
     * @param entityName the Dataverse entity logical name
     * @param primaryKey the primary key field name
     */
    public DataverseDao(DataverseClient client, Class<T> entityCls, 
                        String entityName, String primaryKey) {
        super(entityCls);
        this.client = client;
        this.entityName = entityName;
        this.primaryKey = primaryKey;
    }
    
    @Override
    public ODataQuery buildQuery() {
        ODataQuery query = new ODataQuery(entityName);
        
        // 构建过滤条件
        if (baseWhereQuery != null) {
            QueryInfo queryInfo = baseWhereQuery.buildQueryInfo();
            if (queryInfo != null) {
                // 传递entityCls用于字段名解析
                DataverseQueryBuilder.buildFilter(query, queryInfo, entityCls);
            }
            
            // 构建字段选择
            // 如果未显式调用select()，则自动提取实体类的所有字段（排除ignored字段）
            DataverseQueryBuilder.buildSelect(query, baseWhereQuery.getSelectList(), entityCls);
            
            // 构建排序
            if (CollUtil.isNotEmpty(baseWhereQuery.getOrderMap())) {
                // 传递entityCls用于字段名解析
                DataverseQueryBuilder.buildOrderBy(query, baseWhereQuery.getOrderMap(), entityCls);
            }
        }
        
        // 处理baseConditionsVo中的条件
        if (baseConditionsVo != null) {
            if (baseConditionsVo.getWhere() != null) {
                // 传递entityCls用于字段名解析
                DataverseQueryBuilder.buildFilter(query, baseConditionsVo.getWhere(), entityCls);
            }
            
            if (CollUtil.isNotEmpty(baseConditionsVo.getSelectList())) {
                DataverseQueryBuilder.buildSelect(query, baseConditionsVo.getSelectList(), entityCls);
            }
            
            if (CollUtil.isNotEmpty(baseConditionsVo.getOrderMap())) {
                // 传递entityCls用于字段名解析
                DataverseQueryBuilder.buildOrderBy(query, baseConditionsVo.getOrderMap(), entityCls);
            }
        }
        
        // 自动检测并添加查找字段的$expand
        buildExpandForLookupFields(query);
        
        return query;
    }
    
    /**
     * 自动为查找字段构建$expand子句
     * 扫描实体类中标记为lookup的字段，并添加到expand中
     *
     * @param query OData查询对象
     */
    private void buildExpandForLookupFields(ODataQuery query) {
        List<String> expandFields = new ArrayList<>();
        
        Field[] fields = entityCls.getDeclaredFields();
        for (Field field : fields) {
            DataverseField annotation = field.getAnnotation(DataverseField.class);
            if (annotation != null && annotation.lookup()) {
                // 获取查找字段的导航属性名
                // 通常是去掉_value后缀的字段名
                String fieldName = annotation.value();
                String navigationProperty = extractNavigationProperty(fieldName);
                if (StrUtil.isNotBlank(navigationProperty)) {
                    expandFields.add(navigationProperty);
                }
            }
        }
        
        if (!expandFields.isEmpty()) {
            String expand = String.join(",", expandFields);
            query.setExpand(expand);
            log.debug("Auto-detected lookup fields for expand: {}", expand);
        }
    }
    
    /**
     * 从查找字段名提取导航属性名
     * 例如: _primarycontactid_value -> primarycontactid
     *
     * @param lookupFieldName 查找字段名
     * @return 导航属性名
     */
    private String extractNavigationProperty(String lookupFieldName) {
        if (StrUtil.isBlank(lookupFieldName)) {
            return null;
        }
        
        // Dataverse查找字段通常格式为: _fieldname_value
        if (lookupFieldName.startsWith("_") && lookupFieldName.endsWith("_value")) {
            return lookupFieldName.substring(1, lookupFieldName.length() - 6);
        }
        
        return lookupFieldName;
    }
    
    @Override
    public List<T> list() {
        ODataQuery query = buildQuery();
        List<T> results = new ArrayList<>();
        String nextLink = null;
        
        // 自动跟随分页链接，检索所有页面
        do {
            ODataResponse response = client.execute(query, nextLink);
            
            // 处理null响应
            if (response == null) {
                log.warn("Received null response from Dataverse client");
                break;
            }
            
            List<T> pageResults = parseResponse(response);
            results.addAll(pageResults);
            nextLink = response.getNextLink();
            
            log.debug("Retrieved {} records, nextLink={}", pageResults.size(), 
                    nextLink != null ? "present" : "null");
        } while (nextLink != null);
        
        log.debug("Total records retrieved: {}", results.size());
        return results;
    }
    
    @Override
    public T getFirst() {
        ODataQuery query = buildQuery();
        query.setTop(1);
        
        ODataResponse response = client.execute(query, null);
        List<T> results = parseResponse(response);
        
        if (results.isEmpty()) {
            return null;
        }
        
        return results.get(0);
    }
    
    @Override
    public Long count() {
        ODataQuery query = buildQuery();
        query.setCount(true);
        query.setTop(0); // 只获取计数，不获取数据
        
        ODataResponse response = client.execute(query, null);
        
        if (response.getCount() != null) {
            return response.getCount();
        }
        
        // 如果响应中没有count，返回value数组的大小
        // 这种情况不应该发生，但作为后备方案
        log.warn("Count not returned in response, falling back to value array size");
        return response.getValue() != null ? (long) response.getValue().size() : 0L;
    }
    
    @Override
    public IJPage<T> page(IJPage<T> page) {
        // Dataverse不支持$skip参数（错误代码0x80060888）
        // 只能查询第一页（pageNo=0）
        if (page.getPageNo() > 0) {
            throw new UnsupportedOperationException(
                "Dataverse不支持$skip参数，无法跳转到指定页码。" +
                "请使用list()方法获取所有数据（会自动跟随@odata.nextLink分页），" +
                "或仅查询第一页（pageNo=0）。" +
                "当前请求: pageNo=" + page.getPageNo());
        }
        
        ODataQuery query = buildQuery();
        
        // 只设置$top参数，不设置$skip
        query.setTop(page.getPageSize());
        
        // 如果需要总计数，添加$count=true
        if (page.isHaveTotal()) {
            query.setCount(true);
        }
        
        ODataResponse response = client.execute(query, null);
        
        // 设置总计数
        if (page.isHaveTotal() && response.getCount() != null) {
            page.setTotal(response.getCount());
        }
        
        // 解析并设置数据
        List<T> results = parseResponse(response);
        page.setData(results);
        
        log.debug("Page query: pageNo={}, pageSize={}, total={}, resultSize={}", 
                page.getPageNo(), page.getPageSize(), page.getTotal(), results.size());
        
        return page;
    }
    
    @Override
    public T findById(Serializable id) {
        if (id == null) {
            throw new IllegalArgumentException("ID cannot be null");
        }
        
        // 构建实体路径: entityName(id)
        String entityPath = String.format("%s(%s)", entityName, formatIdForUrl(id));
        
        ODataResponse response = client.get(entityPath);
        List<T> results = parseResponse(response);
        
        if (results.isEmpty()) {
            return null;
        }
        
        return results.get(0);
    }
    
    /**
     * 格式化ID用于URL
     * GUID不需要引号，其他类型需要引号
     *
     * @param id ID值
     * @return 格式化后的ID字符串
     */
    private String formatIdForUrl(Serializable id) {
        String idStr = id.toString();
        
        // 检查是否为GUID格式 (xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx)
        if (idStr.matches("[0-9a-fA-F]{8}-[0-9a-fA-F]{4}-[0-9a-fA-F]{4}-[0-9a-fA-F]{4}-[0-9a-fA-F]{12}")) {
            return idStr;
        }
        
        // 其他类型用引号包裹
        return "'" + idStr + "'";
    }
    
    /**
     * 解析OData响应并映射到实体对象
     * 处理格式化值和字段映射
     *
     * @param response OData响应
     * @return 实体对象列表
     */
    private List<T> parseResponse(ODataResponse response) {
        if (response == null || response.getValue() == null) {
            return new ArrayList<>();
        }
        
        JSONArray valueArray = response.getValue();
        List<T> results = new ArrayList<>(valueArray.size());
        
        for (int i = 0; i < valueArray.size(); i++) {
            JSONObject jsonObject = valueArray.getJSONObject(i);
            T entity = mapJsonToEntity(jsonObject);
            results.add(entity);
        }
        
        return results;
    }
    
    /**
     * 将JSON对象映射到实体对象
     * 处理@DataverseField、@FormattedValue注解和查找字段
     *
     * @param jsonObject JSON对象
     * @return 实体对象
     */
    private T mapJsonToEntity(JSONObject jsonObject) {
        try {
            T entity = entityCls.newInstance();
            Field[] fields = entityCls.getDeclaredFields();
            
            for (Field field : fields) {
                field.setAccessible(true);
                
                // 检查是否有@FormattedValue注解
                FormattedValue formattedValueAnnotation = field.getAnnotation(FormattedValue.class);
                if (formattedValueAnnotation != null) {
                    // 这是一个格式化值字段
                    String sourceField = formattedValueAnnotation.sourceField();
                    if (StrUtil.isBlank(sourceField)) {
                        sourceField = field.getName();
                    }
                    
                    // 查找格式化值
                    String formattedKey = sourceField + "@OData.Community.Display.V1.FormattedValue";
                    if (jsonObject.containsKey(formattedKey)) {
                        Object formattedValue = jsonObject.get(formattedKey);
                        setFieldValue(field, entity, formattedValue);
                    }
                    continue;
                }
                
                // 获取Dataverse字段名
                String dataverseFieldName = getDataverseFieldName(field);
                
                // 检查是否为查找字段
                DataverseField dataverseFieldAnnotation = field.getAnnotation(DataverseField.class);
                boolean isLookupField = dataverseFieldAnnotation != null && dataverseFieldAnnotation.lookup();
                
                // 从JSON中获取值
                if (jsonObject.containsKey(dataverseFieldName)) {
                    Object value = jsonObject.get(dataverseFieldName);
                    
                    // 处理null查找字段
                    if (value == null && isLookupField) {
                        setFieldValue(field, entity, null);
                        continue;
                    }
                    
                    // 对于查找字段，如果值是对象（通过$expand获取），提取ID
                    if (isLookupField && value instanceof JSONObject) {
                        JSONObject lookupObject = (JSONObject) value;
                        // 查找字段通常有一个主键字段，尝试常见的ID字段名
                        Object lookupId = extractLookupId(lookupObject);
                        setFieldValue(field, entity, lookupId);
                    } else {
                        setFieldValue(field, entity, value);
                    }
                }
            }
            
            return entity;
            
        } catch (Exception e) {
            log.error("Failed to map JSON to entity: {}", e.getMessage(), e);
            throw new LambdaQueryException("Failed to map JSON to entity", e);
        }
    }
    
    /**
     * 从查找对象中提取ID值
     * 尝试常见的ID字段名模式
     *
     * @param lookupObject 查找对象
     * @return ID值
     */
    private Object extractLookupId(JSONObject lookupObject) {
        // 尝试常见的ID字段名
        String[] idFieldNames = {
            "id", 
            lookupObject.keySet().stream()
                .filter(key -> key.endsWith("id"))
                .findFirst()
                .orElse(null)
        };
        
        for (String idFieldName : idFieldNames) {
            if (idFieldName != null && lookupObject.containsKey(idFieldName)) {
                return lookupObject.get(idFieldName);
            }
        }
        
        // 如果找不到ID字段，返回整个对象的字符串表示
        log.warn("Could not find ID field in lookup object, available keys: {}", 
                lookupObject.keySet());
        return lookupObject.toJSONString();
    }
    
    /**
     * 获取字段的Dataverse字段名
     * 如果有@DataverseField注解，使用注解值；否则使用Java字段名
     *
     * @param field Java字段
     * @return Dataverse字段名
     */
    private String getDataverseFieldName(Field field) {
        DataverseField annotation = field.getAnnotation(DataverseField.class);
        if (annotation != null && StrUtil.isNotBlank(annotation.value())) {
            return annotation.value();
        }
        return field.getName();
    }
    
    /**
     * 设置字段值，处理类型转换
     *
     * @param field 字段
     * @param entity 实体对象
     * @param value 值
     */
    private void setFieldValue(Field field, T entity, Object value) {
        try {
            if (value == null) {
                field.set(entity, null);
                return;
            }
            
            Class<?> fieldType = field.getType();
            
            // 处理类型转换
            if (fieldType == String.class) {
                field.set(entity, value.toString());
            } else if (fieldType == Integer.class || fieldType == int.class) {
                field.set(entity, ((Number) value).intValue());
            } else if (fieldType == Long.class || fieldType == long.class) {
                field.set(entity, ((Number) value).longValue());
            } else if (fieldType == Double.class || fieldType == double.class) {
                field.set(entity, ((Number) value).doubleValue());
            } else if (fieldType == Boolean.class || fieldType == boolean.class) {
                field.set(entity, Boolean.parseBoolean(value.toString()));
            } else {
                // 对于其他类型，尝试直接设置或使用JSON转换
                field.set(entity, Convert.convert(fieldType,value,value));
            }
            
        } catch (Exception e) {
            log.warn("Failed to set field {}: {}", field.getName(), e.getMessage());
        }
    }
    
    /**
     * Get the entity name.
     *
     * @return the entity logical name
     */
    public String getEntityName() {
        return entityName;
    }
    
    /**
     * Get the primary key field name.
     *
     * @return the primary key field name
     */
    public String getPrimaryKey() {
        return primaryKey;
    }
}
