package com.github.xuejike.query.dataverse;

import cn.hutool.core.util.StrUtil;
import com.github.xuejike.query.core.base.BaseDao;
import com.github.xuejike.query.core.config.DaoFactory;
import com.github.xuejike.query.core.exception.LambdaQueryException;
import com.github.xuejike.query.dataverse.annotation.DataverseDaoSelect;
import com.github.xuejike.query.dataverse.client.DataverseClient;
import lombok.extern.slf4j.Slf4j;

import java.util.HashMap;
import java.util.Map;

/**
 * Factory for creating DataverseDao instances.
 * Extends the DaoFactory abstract class to integrate with Lambda-Query framework.
 * Supports multiple Dataverse environments with different clients.
 *
 * @author xuejike
 */
@Slf4j
public class DataverseDaoFactory extends DaoFactory {
    
    /**
     * Map of environment name to DataverseClient
     */
    private final Map<String, DataverseClient> clients;
    
    /**
     * Default environment name
     */
    private static final String DEFAULT_ENVIRONMENT = "default";
    
    /**
     * Create a new DataverseDaoFactory with a default client.
     *
     * @param defaultClient the default Dataverse client
     */
    public DataverseDaoFactory(DataverseClient defaultClient) {
        super(DataverseDao.class);
        this.clients = new HashMap<>();
        this.clients.put(DEFAULT_ENVIRONMENT, defaultClient);
        log.info("DataverseDaoFactory initialized with default client");
    }
    
    /**
     * Create a new DataverseDaoFactory with multiple environment clients.
     *
     * @param clients map of environment name to DataverseClient
     */
    public DataverseDaoFactory(Map<String, DataverseClient> clients) {
        super(DataverseDao.class);
        if (clients == null || clients.isEmpty()) {
            throw new IllegalArgumentException("Clients map cannot be null or empty");
        }
        this.clients = new HashMap<>(clients);
        log.info("DataverseDaoFactory initialized with {} environments", clients.size());
    }
    
    @Override
    public <T> BaseDao<T, ?> createDao(Class<T> entityCls) {
        // 验证实体类有@DataverseDaoSelect注解
        DataverseDaoSelect annotation = entityCls.getAnnotation(DataverseDaoSelect.class);
        if (annotation == null) {
            throw new LambdaQueryException(
                    "Class " + entityCls.getName() + " is not annotated with @DataverseDaoSelect");
        }
        
        // 提取配置
        String entityName = extractEntityName(entityCls);
        String primaryKey = extractPrimaryKey(entityCls);
        String environment = extractEnvironment(entityCls);
        
        // 根据环境选择对应的客户端
        DataverseClient client = getClientForEnvironment(environment);
        
        log.debug("Creating DataverseDao for entity={}, primaryKey={}, environment={}", 
                entityName, primaryKey, environment);
        
        // 创建并返回DataverseDao实例
        return new DataverseDao<>(client, entityCls, entityName, primaryKey);
    }
    
    /**
     * Get the DataverseClient for the specified environment.
     *
     * @param environment the environment name
     * @return the DataverseClient
     * @throws LambdaQueryException if the environment is not configured
     */
    protected DataverseClient getClientForEnvironment(String environment) {
        DataverseClient client = clients.get(environment);
        
        if (client == null) {
            throw new LambdaQueryException(
                    "No DataverseClient configured for environment: " + environment + 
                    ". Available environments: " + clients.keySet());
        }
        
        return client;
    }
    
    /**
     * Add a client for a specific environment.
     *
     * @param environment the environment name
     * @param client the DataverseClient
     */
    public void addClient(String environment, DataverseClient client) {
        if (StrUtil.isBlank(environment)) {
            throw new IllegalArgumentException("Environment name cannot be blank");
        }
        if (client == null) {
            throw new IllegalArgumentException("Client cannot be null");
        }
        
        clients.put(environment, client);
        log.info("Added DataverseClient for environment: {}", environment);
    }
    
    /**
     * Extract entity name from annotation.
     *
     * @param entityCls the entity class
     * @return the entity name
     */
    protected String extractEntityName(Class<?> entityCls) {
        DataverseDaoSelect annotation = entityCls.getAnnotation(DataverseDaoSelect.class);
        if (annotation == null) {
            throw new LambdaQueryException(
                    "Class " + entityCls.getName() + " is not annotated with @DataverseDaoSelect");
        }
        
        String entityName = annotation.entityName();
        if (StrUtil.isBlank(entityName)) {
            throw new LambdaQueryException(
                    "entityName in @DataverseDaoSelect cannot be blank for class " + entityCls.getName());
        }
        
        return entityName;
    }
    
    /**
     * Extract primary key from annotation.
     *
     * @param entityCls the entity class
     * @return the primary key field name
     */
    protected String extractPrimaryKey(Class<?> entityCls) {
        DataverseDaoSelect annotation = entityCls.getAnnotation(DataverseDaoSelect.class);
        if (annotation == null) {
            throw new LambdaQueryException(
                    "Class " + entityCls.getName() + " is not annotated with @DataverseDaoSelect");
        }
        
        String primaryKey = annotation.primaryKey();
        if (StrUtil.isBlank(primaryKey)) {
            // 默认: entityName + "id"
            primaryKey = annotation.entityName() + "id";
        }
        
        return primaryKey;
    }
    
    /**
     * Extract environment name from annotation.
     *
     * @param entityCls the entity class
     * @return the environment name
     */
    protected String extractEnvironment(Class<?> entityCls) {
        DataverseDaoSelect annotation = entityCls.getAnnotation(DataverseDaoSelect.class);
        if (annotation == null) {
            throw new LambdaQueryException(
                    "Class " + entityCls.getName() + " is not annotated with @DataverseDaoSelect");
        }
        
        String environment = annotation.environment();
        if (StrUtil.isBlank(environment)) {
            environment = DEFAULT_ENVIRONMENT;
        }
        
        return environment;
    }
    
    /**
     * Get all configured environment names.
     *
     * @return set of environment names
     */
    public java.util.Set<String> getEnvironments() {
        return clients.keySet();
    }
}
