package com.github.xuejike.query.dataverse;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.StrUtil;
import com.github.xuejike.query.core.enums.OrderType;
import com.github.xuejike.query.core.enums.WhereOperation;
import com.github.xuejike.query.core.po.BetweenObj;
import com.github.xuejike.query.core.po.FieldInfo;
import com.github.xuejike.query.core.po.QueryInfo;
import com.github.xuejike.query.core.po.QueryItem;
import com.github.xuejike.query.dataverse.annotation.DataverseField;
import com.github.xuejike.query.dataverse.client.ODataQuery;
import lombok.extern.slf4j.Slf4j;

import java.lang.reflect.Field;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.stream.Collectors;

/**
 * Builder for converting Lambda-Query QueryInfo to OData v4.0 queries.
 * Handles filter, select, orderby, and other OData query components.
 * 
 * <p><b>重要设计原则：字段名解析一致性</b></p>
 * <p>
 * buildFilter()、buildSelect()和buildOrderBy()方法都使用同一个resolveDataverseFieldName()方法
 * 来解析字段名，确保$filter、$select和$orderby子句中的字段名完全一致。
 * 这避免了因字段名不匹配导致的OData查询错误。
 * </p>
 * 
 * <p>字段名解析规则：</p>
 * <ul>
 *   <li>如果字段有@DataverseField注解，使用注解指定的Dataverse字段名</li>
 *   <li>如果字段没有注解，使用Java字段名</li>
 *   <li>不进行任何命名转换（如驼峰转下划线）</li>
 *   <li>所有OData子句使用相同的字段名</li>
 * </ul>
 *
 * @author xuejike
 */
@Slf4j
public class DataverseQueryBuilder {
    
    private static final SimpleDateFormat ISO_DATE_FORMAT = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss'Z'");
    
    static {
        ISO_DATE_FORMAT.setTimeZone(TimeZone.getTimeZone("UTC"));
    }
    
    /**
     * Build the $filter clause from QueryInfo.
     * 
     * <p>使用resolveDataverseFieldName()解析字段名，与buildSelect()和buildOrderBy()保持一致。</p>
     *
     * @param query the OData query to populate
     * @param queryInfo the Lambda-Query query info
     * @param entityClass the entity class (用于字段名解析)
     */
    public static void buildFilter(ODataQuery query, QueryInfo queryInfo, Class<?> entityClass) {
        if (queryInfo == null) {
            return;
        }
        
        StringBuilder filter = new StringBuilder();
        buildFilterRecursive(filter, queryInfo, entityClass);
        
        if (filter.length() > 0) {
            query.setFilter(filter.toString());
        }
    }
    
    /**
     * Build the $filter clause from QueryInfo (overload for backward compatibility).
     * 
     * @param query the OData query to populate
     * @param queryInfo the Lambda-Query query info
     * @deprecated Use {@link #buildFilter(ODataQuery, QueryInfo, Class)} instead
     */
    @Deprecated
    public static void buildFilter(ODataQuery query, QueryInfo queryInfo) {
        buildFilter(query, queryInfo, null);
    }
    
    /**
     * Recursively build filter expressions handling nested queries and OR logic.
     * 
     * @param sb the StringBuilder to append to
     * @param queryInfo the query info
     * @param entityClass the entity class (用于字段名解析)
     */
    private static void buildFilterRecursive(StringBuilder sb, QueryInfo queryInfo, Class<?> entityClass) {
        List<String> conditions = new ArrayList<>();
        
        // Process AND conditions
        if (CollUtil.isNotEmpty(queryInfo.getAnd())) {
            for (QueryItem item : queryInfo.getAnd()) {
                String condition = buildCondition(item, entityClass);
                if (StrUtil.isNotBlank(condition)) {
                    conditions.add(condition);
                }
            }
        }
        
        // Process OR conditions
        if (CollUtil.isNotEmpty(queryInfo.getOr())) {
            List<String> orConditions = new ArrayList<>();
            for (QueryInfo orQuery : queryInfo.getOr()) {
                StringBuilder orBuilder = new StringBuilder();
                buildFilterRecursive(orBuilder, orQuery, entityClass);
                if (orBuilder.length() > 0) {
                    orConditions.add("(" + orBuilder.toString() + ")");
                }
            }
            
            if (!orConditions.isEmpty()) {
                conditions.add("(" + String.join(" or ", orConditions) + ")");
            }
        }
        
        // Join all conditions with AND
        if (!conditions.isEmpty()) {
            sb.append(String.join(" and ", conditions));
        }
    }
    
    /**
     * Build a single condition from a QueryItem.
     * 
     * @param item the query item
     * @param entityClass the entity class (用于字段名解析)
     * @return the condition string
     */
    private static String buildCondition(QueryItem item, Class<?> entityClass) {
        if (item == null || item.getField() == null || item.getVal() == null) {
            return "";
        }
        
        String fieldName = resolveDataverseFieldName(item.getField(), entityClass);
        Map<WhereOperation, Object> valMap = item.getVal();
        
        List<String> conditions = new ArrayList<>();
        
        for (Map.Entry<WhereOperation, Object> entry : valMap.entrySet()) {
            WhereOperation operation = entry.getKey();
            Object value = entry.getValue();
            
            String condition = buildOperationCondition(fieldName, operation, value);
            if (StrUtil.isNotBlank(condition)) {
                conditions.add(condition);
            }
        }
        
        return String.join(" and ", conditions);
    }
    
    /**
     * Build condition for a specific operation.
     */
    private static String buildOperationCondition(String fieldName, WhereOperation operation, Object value) {
        switch (operation) {
            case eq:
                return fieldName + " eq " + formatValue(value);
            case ne:
                return fieldName + " ne " + formatValue(value);
            case gt:
                return fieldName + " gt " + formatValue(value);
            case gte:
                return fieldName + " ge " + formatValue(value);
            case lt:
                return fieldName + " lt " + formatValue(value);
            case lte:
                return fieldName + " le " + formatValue(value);
            case in:
                return buildInCondition(fieldName, value);
            case notIn:
                return "not (" + buildInCondition(fieldName, value) + ")";
            case isNull:
                return fieldName + " eq null";
            case notNull:
                return fieldName + " ne null";
            case between:
                return buildBetweenCondition(fieldName, value);
            default:
                log.warn("Unsupported operation: {}", operation);
                return "";
        }
    }
    
    /**
     * Build IN condition (field eq value1 or field eq value2 or ...).
     */
    private static String buildInCondition(String fieldName, Object value) {
        Collection<?> values;
        
        if (value instanceof Collection) {
            values = (Collection<?>) value;
        } else if (value.getClass().isArray()) {
            values = Arrays.asList((Object[]) value);
        } else {
            values = Collections.singletonList(value);
        }
        
        if (values.isEmpty()) {
            return "1 eq 0"; // Always false condition
        }
        
        List<String> conditions = values.stream()
            .map(v -> fieldName + " eq " + formatValue(v))
            .collect(Collectors.toList());
        
        return "(" + String.join(" or ", conditions) + ")";
    }
    
    /**
     * Build BETWEEN condition (field ge value1 and field le value2).
     */
    private static String buildBetweenCondition(String fieldName, Object value) {
        if (value instanceof BetweenObj) {
            BetweenObj between = (BetweenObj) value;
            return "(" + fieldName + " ge " + formatValue(between.getFirst()) + 
                   " and " + fieldName + " le " + formatValue(between.getSecond()) + ")";
        } else if (value instanceof Object[] && ((Object[]) value).length == 2) {
            Object[] range = (Object[]) value;
            return "(" + fieldName + " ge " + formatValue(range[0]) + 
                   " and " + fieldName + " le " + formatValue(range[1]) + ")";
        }
        
        log.warn("Invalid BETWEEN value: {}", value);
        return "";
    }
    
    /**
     * Format a value for OData query.
     * Handles strings, numbers, dates, booleans, and GUIDs.
     */
    private static String formatValue(Object value) {
        if (value == null) {
            return "null";
        }
        
        if (value instanceof String) {
            // Escape single quotes and wrap in quotes
            String str = value.toString().replace("'", "''");
            return "'" + str + "'";
        }
        
        if (value instanceof Date) {
            // Format as ISO 8601 date
            return ISO_DATE_FORMAT.format((Date) value);
        }
        
        if (value instanceof Boolean) {
            return value.toString().toLowerCase();
        }
        
        if (value instanceof Number) {
            return value.toString();
        }
        
        // For GUIDs and other types, convert to string
        return value.toString();
    }
    
    /**
     * Build the $select clause.
     * 如果selectList为空，则根据实体类定义自动生成所有字段（排除ignored字段）
     * 
     * <p>使用resolveDataverseFieldName()解析字段名，与buildFilter()和buildOrderBy()保持一致。</p>
     *
     * @param query the OData query to populate
     * @param selectList the list of fields to select (如果为空，则自动提取)
     * @param entityClass the entity class (用于自动提取字段和字段名解析)
     */
    public static void buildSelect(ODataQuery query, List<FieldInfo> selectList, Class<?> entityClass) {
        List<FieldInfo> fieldsToSelect = selectList;
        
        // 如果未指定select字段，则使用实体类定义的所有字段（排除ignored字段）
        if (CollUtil.isEmpty(selectList) && entityClass != null) {
            fieldsToSelect = extractAllFieldsFromEntity(entityClass);
        }
        
        if (CollUtil.isEmpty(fieldsToSelect)) {
            return;
        }
        
        String select = fieldsToSelect.stream()
            .map(fieldInfo -> resolveDataverseFieldName(fieldInfo, entityClass))
            .filter(StrUtil::isNotBlank)
            .collect(Collectors.joining(","));
        
        if (StrUtil.isNotBlank(select)) {
            query.setSelect(select);
        }
    }
    
    /**
     * Build the $select clause (overload for backward compatibility).
     *
     * @param query the OData query to populate
     * @param selectList the list of fields to select
     */
    public static void buildSelect(ODataQuery query, List<FieldInfo> selectList) {
        buildSelect(query, selectList, null);
    }
    
    /**
     * 从实体类中提取所有字段，用于自动生成$select子句
     * 
     * <p><b>核心原则：FieldInfo.field永远存储Java字段名，不存储Dataverse字段名</b></p>
     * 
     * 提取规则：
     * 1. 排除静态字段（static修饰的字段）
     * 2. 排除@RefValue注解的字段（这些字段通过LoadRef填充，不应在主查询中包含）
     * 3. 排除@DataverseField注解中ignored=true的字段（除非显式调用select()方法）
     * 4. 包含所有其他字段
     * 
     * 字段名存储：
     * - FieldInfo.field存储Java字段名（不是Dataverse字段名）
     * - 后续通过resolveDataverseFieldName()方法将Java字段名转换为Dataverse字段名
     * - 这确保了字段名解析的一致性
     * 
     * 使用场景：
     * 当开发者未显式调用select()方法时，系统会自动调用此方法生成$select子句，
     * 以确保查询返回实体所需的所有字段（除了ignored字段）
     * 
     * @param entityClass 实体类
     * @return 字段信息列表（field属性存储Java字段名）
     */
    public static List<FieldInfo> extractAllFieldsFromEntity(Class<?> entityClass) {
        if (entityClass == null) {
            return new ArrayList<>();
        }
        
        List<FieldInfo> fields = new ArrayList<>();
        Field[] declaredFields = entityClass.getDeclaredFields();
        
        for (Field field : declaredFields) {
            // 跳过静态字段
            if (java.lang.reflect.Modifier.isStatic(field.getModifiers())) {
                log.debug("Skipping static field: {}", field.getName());
                continue;
            }
            
            // 跳过@RefValue注解的字段（这些是LoadRef填充的）
            if (field.isAnnotationPresent(com.github.xuejike.query.core.annotation.RefValue.class)) {
                log.debug("Skipping @RefValue field: {}", field.getName());
                continue;
            }
            
            // 检查@DataverseField注解的ignored属性
            DataverseField annotation = field.getAnnotation(DataverseField.class);
            if (annotation != null && annotation.ignored()) {
                log.debug("Skipping ignored field: {}", field.getName());
                continue;
            }
            
            // 创建FieldInfo并存储Java字段名（不是Dataverse字段名）
            // 后续通过resolveDataverseFieldName()转换为Dataverse字段名
            FieldInfo fieldInfo = new FieldInfo();
            fieldInfo.setField(field.getName()); // 存储Java字段名
            
            fields.add(fieldInfo);
            log.debug("Added field to extract: {} (Java field name)", field.getName());
        }
        
        return fields;
    }
    
    /**
     * Build the $orderby clause.
     * 
     * <p>使用resolveDataverseFieldName()解析字段名，与buildFilter()和buildSelect()保持一致。</p>
     *
     * @param query the OData query to populate
     * @param orderMap the order map (FieldInfo to OrderType)
     * @param entityClass the entity class (用于字段名解析)
     */
    public static void buildOrderBy(ODataQuery query, Map<FieldInfo, OrderType> orderMap, Class<?> entityClass) {
        if (CollUtil.isEmpty(orderMap)) {
            return;
        }
        
        String orderBy = orderMap.entrySet().stream()
            .map(entry -> {
                String fieldName = resolveDataverseFieldName(entry.getKey(), entityClass);
                String direction = entry.getValue() == OrderType.desc ? " desc" : " asc";
                return fieldName + direction;
            })
            .filter(StrUtil::isNotBlank)
            .collect(Collectors.joining(","));
        
        if (StrUtil.isNotBlank(orderBy)) {
            query.setOrderBy(orderBy);
        }
    }
    
    /**
     * Build the $orderby clause (overload for backward compatibility).
     *
     * @param query the OData query to populate
     * @param orderMap the order map (FieldInfo to OrderType)
     * @deprecated Use {@link #buildOrderBy(ODataQuery, Map, Class)} instead
     */
    @Deprecated
    public static void buildOrderBy(ODataQuery query, Map<FieldInfo, OrderType> orderMap) {
        buildOrderBy(query, orderMap, null);
    }
    
    /**
     * 解析Dataverse字段名称
     * 
     * <p><b>核心原则：FieldInfo.field永远存储Java字段名，需要通过反射和@DataverseField注解转换为Dataverse字段名</b></p>
     * 
     * <p><b>重要：此方法被buildFilter()、buildSelect()和buildOrderBy()共同使用，
     * 确保$filter、$select、$orderby子句中的字段名解析规则完全一致。
     * 这保证了查询条件、字段选择和排序使用相同的字段名称，避免OData查询错误。</b></p>
     * 
     * <p>字段名映射规则：</p>
     * <ol>
     *   <li>FieldInfo.field存储的是Java字段名（不是Dataverse字段名）</li>
     *   <li>使用entityClass通过反射获取Field对象</li>
     *   <li>从Field对象读取@DataverseField注解</li>
     *   <li>如果注解存在且value不为空，使用注解值作为Dataverse字段名</li>
     *   <li>如果注解不存在或value为空，使用Java字段名作为Dataverse字段名</li>
     *   <li>不进行任何命名转换（如驼峰转下划线、大小写转换等）</li>
     *   <li>特殊字符（下划线、数字等）保持原样</li>
     *   <li>大小写完全保持原样</li>
     * </ol>
     * 
     * <p>示例：</p>
     * <ul>
     *   <li>Java字段名 "firstName" (无注解) - OData字段名 "firstName"</li>
     *   <li>Java字段名 "ownerId" + @DataverseField("_ownerid_value") - OData字段名 "_ownerid_value"</li>
     *   <li>Java字段名 "customField" + @DataverseField("cr_customname") - OData字段名 "cr_customname"</li>
     *   <li>Java字段名 "myField" + @DataverseField("MyField") - OData字段名 "MyField"</li>
     * </ul>
     *
     * @param fieldInfo 字段信息（field属性存储Java字段名）
     * @param entityClass 实体类（用于反射获取Field对象和@DataverseField注解）
     * @return Dataverse字段名称，如果fieldInfo为null或字段名为空则返回空字符串
     */
    public static String resolveDataverseFieldName(FieldInfo fieldInfo, Class<?> entityClass) {
        if (fieldInfo == null || StrUtil.isBlank(fieldInfo.getField())) {
            return "";
        }
        
        String javaFieldName = fieldInfo.getField();
        
        // 如果提供了entityClass，通过反射获取@DataverseField注解
        if (entityClass != null) {
            try {
                Field field = entityClass.getDeclaredField(javaFieldName);
                DataverseField annotation = field.getAnnotation(DataverseField.class);
                if (annotation != null && StrUtil.isNotBlank(annotation.value())) {
                    // 使用注解指定的Dataverse字段名
                    log.debug("Resolved field name: {} -> {} (from @DataverseField)", 
                            javaFieldName, annotation.value());
                    return annotation.value();
                }
            } catch (NoSuchFieldException e) {
                // 字段不存在，使用Java字段名
                log.debug("Field {} not found in class {}, using Java field name", 
                        javaFieldName, entityClass.getName());
            } catch (Exception e) {
                log.debug("Error resolving field annotation for {}: {}", 
                        javaFieldName, e.getMessage());
            }
        }
        
        // 对于嵌套字段，构建字段路径
        if (CollUtil.isNotEmpty(fieldInfo.getSubList())) {
            StringBuilder fieldPath = new StringBuilder(javaFieldName);
            for (FieldInfo subField : fieldInfo.getSubList()) {
                fieldPath.append("/").append(resolveDataverseFieldName(subField, entityClass));
            }
            return fieldPath.toString();
        }
        
        // 没有注解或entityClass为null时，直接返回Java字段名
        log.debug("Resolved field name: {} -> {} (no annotation or entityClass)", 
                javaFieldName, javaFieldName);
        return javaFieldName;
    }
    
    /**
     * 解析Dataverse字段名称（向后兼容的重载方法）
     * 
     * @param fieldInfo 字段信息
     * @return Dataverse字段名称
     * @deprecated Use {@link #resolveDataverseFieldName(FieldInfo, Class)} instead
     */
    @Deprecated
    public static String resolveDataverseFieldName(FieldInfo fieldInfo) {
        return resolveDataverseFieldName(fieldInfo, null);
    }
}
