package com.github.xuejike.query.dataverse.annotation;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

/**
 * 用于将Java字段映射到Dataverse表字段的注解
 * 允许在Java实体和Dataverse表之间进行自定义字段名映射
 * 
 * 字段名映射规则：
 * - 注解值会被直接使用，不做任何转换（不会进行驼峰转下划线、大小写转换等）
 * - 如果不使用此注解，将直接使用Java字段名，同样不做任何转换
 * - 特殊字符（下划线、数字等）会保持原样
 * - 大小写完全保持原样
 * 
 * 示例：
 * <pre>
 * &#64;DataverseField("cr_customname")  // 使用Dataverse字段名 "cr_customname"
 * private String customName;
 * 
 * &#64;DataverseField("_ownerid_value")  // 查找字段，保持下划线
 * private String ownerId;
 * 
 * private String firstName;  // 没有注解，使用Java字段名 "firstName"（不转换为first_name）
 * </pre>
 *
 * @author xuejike
 */
@Target(ElementType.FIELD)
@Retention(RetentionPolicy.RUNTIME)
public @interface DataverseField {
    
    /**
     * Dataverse表中的字段名称
     * 如果不指定，将使用Java字段名
     * 
     * 注意：此值会被直接使用，不会进行任何命名转换
     *
     * @return Dataverse字段名称
     */
    String value() default "";
    
    /**
     * Whether this field is the primary key.
     *
     * @return true if this is the primary key field
     */
    boolean primaryKey() default false;
    
    /**
     * Whether this field is a lookup field (reference to another entity).
     *
     * @return true if this is a lookup field
     */
    boolean lookup() default false;
    
    /**
     * 是否在自动生成$select子句时忽略此字段
     * 
     * 当为true时，该字段不会自动添加到$select中，除非显式调用select()方法选择它。
     * 这对于以下场景很有用：
     * - 大型文本字段（如备注、描述）：避免不必要的数据传输
     * - 计算字段：仅在需要时查询
     * - 仅通过LoadRef填充的字段：避免重复查询
     * - 性能优化：减少查询返回的数据量
     * 
     * 注意：
     * - 当显式调用select()方法选择该字段时，ignored属性会被忽略，字段仍会被包含在查询中
     * - 即使字段被标记为ignored，如果响应中包含该字段，映射时仍会正常填充
     * 
     * 示例：
     * <pre>
     * &#64;DataverseField(value = "description", ignored = true)
     * private String description;  // 默认不查询，需要时显式select
     * 
     * &#64;DataverseField(value = "_ownerid_value")
     * &#64;RefValue(...)
     * private Owner owner;  // 通过LoadRef填充，不需要在主查询中包含
     * </pre>
     *
     * @return true表示在自动生成$select时忽略此字段，false表示包含（默认）
     */
    boolean ignored() default false;
}
